# -*- coding: utf-8 -*-

import logging

from lxml import etree

from common.models.geo import Settlement, Region
from travel.rasp.admin.lib.maps.client import GeocoderClient


log = logging.getLogger(__name__)

GEOCODER_GOOD_KINDS = ('locality', 'province', 'railway', 'street')


class GeocoderException(Exception):
    pass


def safe_get_geocode_coords(settlement):
    try:
        return get_geocode_coords(settlement)
    except Exception as ex:
        log.error("Cannot call geocoder for %s: %s", unicode(settlement), ex)
        return None, None


def get_geocode_coords(settlement):
    title = settlement.L_title()

    # Если в запрос добавлять регион, то для многих российских городов
    # геокодер ничего не находит
    if settlement.country:
        title = u"%s, %s" % (title, settlement.country.L_title())

    client = GeocoderClient()
    objs, proto_response = client.search(title, max_results=5, kinds=['locality'])

    for obj in objs:
        if obj['coords']['lat'] and obj['coords']['lon']:
            return obj['coords']['lat'], obj['coords']['lon']


def safe_get_key_text(obj, key):
    try:
        return obj.find(key).text
    except:
        return None


def parse_geocoder_answer(xml, query_text):
    tree = etree.fromstring(xml)

    geoobjects = tree.xpath('//featureMember/GeoObject')

    def relevant_estimate(geoobject):
        if safe_get_key_text(geoobject, 'name') == query_text:
            return 0
        else:
            return 1

    geoobjects.sort(key=relevant_estimate)

    try:
        lon, lat = geoobjects[0].find('Point/pos').text.split()
    except (AttributeError, IndexError):
        lon, lat = None, None

    return lon, lat


class RaspGeocodedObject(object):
    def __init__(self, geo_object, latitude, longitude, maps_geoobject):
        self.geo_object = geo_object
        self.latitude = latitude
        self.longitude = longitude
        self.maps_name = maps_geoobject.get('name')
        self.maps_kind = maps_geoobject.get('kind')


class RaspGeocodeResult(object):
    def __init__(self, rasp_geoocode_objects, geocoder_response):
        self.rasp_geoocode_objects = rasp_geoocode_objects
        self.geocoder_response = geocoder_response


def geocode_to_rasp_geo_objects(query):
    """
    Используем вот этот вот геокодер
    https://a.yandex-team.ru/arc/trunk/arcadia/extsearch/geo
    """
    client = GeocoderClient()
    objs, proto_response = client.search(query, max_results=5, kinds=GEOCODER_GOOD_KINDS)

    rasp_geoobjects = []
    for maps_geoobject in objs:
        rasp_geoobject = get_rasp_geoobject(maps_geoobject)
        if rasp_geoobject:
            rasp_geoobjects.append(rasp_geoobject)

    return RaspGeocodeResult(rasp_geoobjects, GeocoderClient.get_dict_response(proto_response))


def get_rasp_geoobject(obj):
    geo_id = obj.get('geo_id')
    if not geo_id:
        return

    if not(obj['coords']['lat'] and obj['coords']['lon']):
        return

    lon, lat = obj['coords']['lon'], obj['coords']['lat']

    geo_object = None
    try:
        geo_object = Settlement.objects.get(_geo_id=geo_id)
    except Settlement.DoesNotExist:
        pass
    try:
        geo_object = Region.objects.get(_geo_id=geo_id)
    except Region.DoesNotExist:
        pass

    if geo_object:
        return RaspGeocodedObject(geo_object, lat, lon, obj)
