import os
from importlib import import_module

import django
from django.conf import settings
from django.core.urlresolvers import get_callable
from django.http import HttpResponse
from django.utils import translation
from jinja2 import BaseLoader, PackageLoader, Environment, ChoiceLoader, FileSystemLoader, TemplateNotFound
from library.python import resource
from travel.library.python.resource import RES_FS_PREFIX

if django.VERSION[:2] >= (1, 9):
    from django.template.exceptions import TemplateDoesNotExist
    from django.template.library import InvalidTemplateLibrary
else:
    from django.template.base import InvalidTemplateLibrary, TemplateDoesNotExist


APP_PREFIX = 'travel.rasp.admin.'
_env = None


class Library(object):
    def __init__(self):
        self.globals = []
        self.filters = []
        self.tests = []

    def global_(self, func):
        self.globals.append(func)
        return func

    callable = global_

    def filter(self, func):
        self.filters.append(func)
        return func

    def test(self, func):
        return self.tests.append(func)


class ResourceLoader(BaseLoader):

    """ Allow render jinja templates from Resources """

    def __init__(self, prefixes):
        self.prefixes = prefixes

    def get_source(self, environment, template):
        for prefix in self.prefixes:
            path = os.path.join(prefix, template)
            source = resource.find(path)
            if source:
                return source.decode('utf-8'), path, lambda: True
        source = resource.find(template)
        if source:
            return source.decode('utf-8'), template, lambda: True
        raise TemplateNotFound('Template {0!r}, paths: {1!r}'.format(template, self.prefixes))


def get_env():
    global _env

    if _env:
        return _env

    # Setup template loaders

    loader_array = []
    for pth in getattr(settings, 'JINJA_TEMPLATE_DIRS', ()):
        loader_array.append(FileSystemLoader(pth))

    resource_prefixes = []
    for app in settings.INSTALLED_APPS:
        loader_array.append(PackageLoader(app))
        if app.startswith(APP_PREFIX):
            resource_prefixes.append(RES_FS_PREFIX + app[len(APP_PREFIX):].replace('.', '/') + '/templates/')
    loader_array.append(ResourceLoader(resource_prefixes))

    # Setup environment
    global_exts = getattr(settings, 'JINJA_EXTS', ())

    __env = Environment(extensions=global_exts, loader=ChoiceLoader(loader_array))

    if 'jinja2.ext.i18n' in global_exts:
        __env.install_gettext_translations(translation)

    # Add user Globals, Filters, Tests
    global_imports = getattr(settings, 'JINJA_GLOBALS', ())
    for imp in global_imports:
        method = get_callable(imp)
        method_name = getattr(method, 'jinja_name', None)
        if method_name is not None:
            __env.globals[method_name] = method
        else:
            __env.globals[method.__name__] = method

    global_filters = getattr(settings, 'JINJA_FILTERS', ())
    for imp in global_filters:
        method = get_callable(imp)
        method_name = getattr(method, 'jinja_name', None)
        if method_name is not None:
            __env.filters[method_name] = method
        else:
            __env.filters[method.__name__] = method

    global_tests = getattr(settings, 'JINJA_TESTS', ())
    for imp in global_tests:
        method = get_callable(imp)
        method_name = getattr(method, 'jinja_name', None)
        if method_name is not None:
            __env.tests[method_name] = method
        else:
            __env.tests[method.__name__] = method

    global_libraries = getattr(settings, 'JINJA_LIBRARIES', ())
    for module_name in global_libraries:
        try:
            mod = import_module(module_name)
        except ImportError as e:
            raise InvalidTemplateLibrary("Could not load template library from %s, %s" % (module_name, e))

        try:
            lib = mod.jinja

            for type_ in ['globals', 'filters', 'tests']:
                section = getattr(__env, type_)

                for method in getattr(lib, type_):
                    method_name = getattr(method, 'jinja_name', None)

                    if method_name is not None:
                        section[method_name] = method
                    else:
                        section[method.__name__] = method

        except AttributeError:
            raise InvalidTemplateLibrary("Template library %s does not have a variable named 'jinja'" % module_name)

    _env = __env

    return __env


_context_processors = []


def get_context_processors():
    if _context_processors:
        return _context_processors

    for imp in getattr(settings, 'JINJA_CONTEXT_PROCESSORS', ()):
        method = get_callable(imp)
        _context_processors.append(method)

    return _context_processors


def select_template(templates, globals=None):
    """Try to load one of the given templates."""
    for template in templates:
        try:
            return get_env().get_template(template, globals=globals)
        except TemplateNotFound:
            continue
    raise TemplateDoesNotExist(', '.join(templates))


def make_template(template):
    if not hasattr(template, 'render'):
        template = get_env().get_template(template)

    return template


def make_context(template_context, request):
    template_context = dict(template_context or {})

    if request is not None:
        context = {'request': request}

        for processor in get_context_processors():
            context.update(processor(request))

        context.update(template_context)

        return context

    return template_context


def make_module(template, context=None, request=None):
    template = make_template(template)
    context = make_context(context)

    return template.make_module(context)


def render_to_string(template, context=None, request=None):
    template = make_template(template)
    context = make_context(context, request)

    rendered = template.render(**context)
    return rendered


def render_to_response(template, context=None, request=None, mimetype=getattr(settings, 'DEFAULT_CONTENT_TYPE')):
    rendered = render_to_string(template, context, request)
    return HttpResponse(rendered, content_type=mimetype)
