# -*- coding: utf-8 -*-

import json
import logging
import inspect
import os
import subprocess
import sys
import traceback
import uuid
from copy import copy
from io import StringIO
from contextlib import contextmanager

from django.conf import settings
from django.core import mail
from django.utils import translation
from django.utils.encoding import force_bytes, force_text
from django.utils.translation import ugettext
from django.views.debug import get_exception_reporter_filter
from ylog.context import log_context as ylog_context  # noqa

from travel.rasp.library.python.common23.date import environment
from common.utils.fileutils import get_my_caller_file
from travel.rasp.library.python.common23.logging import (
    LogError, ContextRaspFormatter, get_file_logger_name, capture_stdstreams_to_file, _default_tail_name, _DictFormatter
)
from travel.rasp.library.python.common23.logging import patchedHandleError, create_current_file_run_log, create_run_log  # noqa

STDOUT_DEFALUT_FD = 1
STDERR_DEFALUT_FD = 2


EXPANDED_LOG_FORMAT = u"%(filename)s %(levelname)s %(asctime)s: %(message)s Line: %(lineno)d"
DT_FORMAT = '%Y-%m-%d %H:%M:%S'


_created_logs = {}


# Логи с контекстом, подключены всегда по умолчанию
class ContextRaspLogger(logging.Logger):
    """
    Специальный логгер, который позволяет дописывать в начале лога одну и туже строку.
    Строка вставляется, если используется ContextRaspFormatter
    """
    def __init__(self, *args, **kwargs):
        super(ContextRaspLogger, self).__init__(*args, **kwargs)

        import __main__ as main_module
        self.main_file_path = None

        file_path = getattr(main_module, '__file__', None)
        if not file_path:
            return

        file_path = os.path.abspath(file_path)
        if file_path.endswith('.pyc'):
            file_path = file_path[:-1]

        admin_project_path = os.path.abspath(os.path.join(__file__, '../../'))

        if file_path.startswith(admin_project_path):
            file_path = os.path.relpath(file_path, admin_project_path)

        self.main_file_path = file_path

    __context = []
    context_message = None

    def set_context(self, context=None):
        if context:
            self.__context.append(context)
        elif self.__context:
            self.__context.pop()

        self.context_message = u": ".join(self.__context)

    def add_context(self, context):
        if context:
            self.__context.append(context)
        else:
            raise LogError(u"Контекст не должен быть пустым")

        self.context_message = u": ".join(self.__context)

    def remove_context(self, context=None):
        if context:
            try:
                del self.__context[self.__context.index(context)]
            except IndexError:
                raise LogError(u"Контекст '%s' не найден в списке" % context)
        else:
            self.__context.pop()

        self.context_message = u": ".join(self.__context)

    def handle(self, record):
        if self.context_message:
            record.context = self.context_message

        record.main_file_path = self.main_file_path

        return super(ContextRaspLogger, self).handle(record)

logging.setLoggerClass(ContextRaspLogger)


@contextmanager
def log_context(log, context):
    log = _get_log_from_string_or_log_instance(log)

    log.add_context(context)

    try:
        yield
    finally:
        log.remove_context()


def get_current_file_run_log_path(script_path='special.script_runs.', tail_name=_default_tail_name, log_name=None):
    if not log_name:
        stack = inspect.stack()
        caller_file = get_my_caller_file(stack)
        log_name = script_path + get_file_logger_name(caller_file)

    path = os.path.join(settings.LOG_PATH, *log_name.split('.'))
    path = os.path.join(path, tail_name())

    if not os.path.exists(os.path.dirname(path)):
        os.makedirs(os.path.dirname(path))

    return path


def get_current_file_run_log_handler(script_path='special.script_runs.', fmt=None, tail_name=_default_tail_name, capture_stdstreams=False):
    fmt = fmt or settings.LOG_FORMAT

    stack = inspect.stack()
    caller_file = get_my_caller_file(stack)
    log_name = script_path + get_file_logger_name(caller_file)
    path = get_current_file_run_log_path(script_path, tail_name, log_name)

    handler = logging.FileHandler(path)
    handler.setFormatter(ContextRaspFormatter(fmt))

    if capture_stdstreams:
        capture_stdstreams_to_file(handler.stream)

    return handler


@contextmanager
def copy_std_streams_to_file(file_name):
    tee = subprocess.Popen(['tee', file_name], stdin=subprocess.PIPE)
    sys.stdout.flush()
    sys.stderr.flush()
    os.dup2(tee.stdin.fileno(), STDOUT_DEFALUT_FD)
    os.dup2(tee.stdin.fileno(), STDERR_DEFALUT_FD)

    yield

    tee.terminate()


# Позволяет генерировать лог на основе места вызова функции
def get_current_file_logger_name():
    stack = inspect.stack()
    caller_file = get_my_caller_file(stack)
    return get_file_logger_name(caller_file)


def get_current_file_logger():
    stack = inspect.stack()
    caller_file = get_my_caller_file(stack)
    return logging.getLogger(get_file_logger_name(caller_file))


@contextmanager
def stdstream_file_capturing(file):
    fd = file.fileno()
    stdout_fd_backup = os.dup(STDOUT_DEFALUT_FD)
    stderr_fd_backup = os.dup(STDERR_DEFALUT_FD)

    stdout_is_redirected = sys.stdout.fileno() != STDOUT_DEFALUT_FD
    if stdout_is_redirected:
        special_stdout_file_no = sys.stdout.fileno()
        special_stdout_fd_backup = os.dup(special_stdout_file_no)

    stderr_is_redirected = sys.stderr.fileno() != STDERR_DEFALUT_FD
    if stderr_is_redirected:
        special_stderr_file_no = sys.stderr.fileno()
        special_stderr_fd_backup = os.dup(special_stderr_file_no)

    sys.stdout.flush()
    sys.stderr.flush()

    os.close(STDOUT_DEFALUT_FD)
    os.dup2(fd, STDOUT_DEFALUT_FD)

    os.close(STDERR_DEFALUT_FD)
    os.dup2(fd, STDERR_DEFALUT_FD)

    if stdout_is_redirected:
        os.close(special_stdout_file_no)
        os.dup2(fd, special_stdout_file_no)

    if stderr_is_redirected:
        os.close(special_stderr_file_no)
        os.dup2(fd, special_stderr_file_no)

    yield

    sys.stdout.flush()
    sys.stderr.flush()

    os.close(STDOUT_DEFALUT_FD)
    os.dup2(stdout_fd_backup, STDOUT_DEFALUT_FD)
    os.close(stdout_fd_backup)

    os.close(STDERR_DEFALUT_FD)
    os.dup2(stderr_fd_backup, STDERR_DEFALUT_FD)
    os.close(stderr_fd_backup)

    if stdout_is_redirected:
        os.close(special_stdout_file_no)
        os.dup2(special_stdout_fd_backup, special_stdout_file_no)
        os.close(special_stdout_fd_backup)

    if stderr_is_redirected:
        os.close(special_stderr_file_no)
        os.dup2(special_stderr_fd_backup, special_stderr_file_no)
        os.close(special_stderr_fd_backup)


capture_stdstreams = capture_stdstreams_to_file


# Функции добалений handler'ов,

def add_stdout_handler(log, format=None):
    from django.conf import settings
    format = format or settings.LOG_FORMAT
    """Добавляем handler для stdout, если еще не добавлен"""
    # Проверяем есть ли у нас handler который пишет в stdout
    add_stream_handler(log, sys.stdout, format)


def print_log_to_stdout(log=''):
    add_stdout_handler(log)


def add_stream_handler(log, stream, format=None, filters=None):
    """Добавляем handler для stream, если еще не добавлен"""

    from django.conf import settings
    format = format or settings.LOG_FORMAT

    log = _get_log_from_string_or_log_instance(log)
    filters = filters or []

    for handler in log.handlers:
        if isinstance(handler, logging.StreamHandler) and handler.stream == stream:
            return

    stream_handler = logging.StreamHandler(stream)
    stream_handler.setFormatter(ContextRaspFormatter(format))

    for log_filter in filters:
        stream_handler.addFilter(log_filter)

    log.addHandler(stream_handler)

    return stream_handler


def remove_stream_handler(log, stream):
    log = _get_log_from_string_or_log_instance(log)

    for handler in log.handlers[:]:
        if isinstance(handler, logging.StreamHandler) and handler.stream == stream:
            log.removeHandler(handler)


def add_file_handler(log, log_path, format=None, filters=None):
    """Добавляем file-handler для log_path, если еще не добавлен"""
    from django.conf import settings
    format = format or settings.LOG_FORMAT

    filters = filters or []

    for handler in log.handlers:
        if isinstance(handler, logging.FileHandler) and handler.baseFilename == log_path:
            return

    try:
        handler = logging.FileHandler(log_path, 'a', encoding='utf-8')
        handler.setFormatter(ContextRaspFormatter(format))

        for log_filter in filters:
            handler.addFilter(log_filter)

        log.addHandler(handler)
    except IOError:
        add_stdout_handler(log, format)


def get_rasp_log_path(log_name):
    name = log_name.strip(u'.')
    filename = name.split('.')[-1]
    dirs = name.split('.')[:-1]

    logpath = settings.LOG_PATH
    logpath = os.path.join(logpath, *dirs)
    if not os.path.exists(logpath):
        os.makedirs(logpath)

    log_filename = os.path.join(logpath, filename + '.log')
    log_path = os.path.abspath(log_filename)

    return log_path


class CollectorHandler(logging.StreamHandler):
    def __init__(self):
        stream = StringIO()
        super(CollectorHandler, self).__init__(stream)

    def format(self, record):
        msg = super(CollectorHandler, self).format(record)
        return msg.decode('utf-8') if isinstance(msg, bytes) else msg

    def get_collected(self, clean=True):
        data = self.stream.getvalue()

        if clean:
            self.stream = StringIO()

        return data


def get_collector(log_or_logname, format=None, log_level=logging.INFO, lang='ru'):
    from django.conf import settings
    format = format or settings.LOG_COLLECTOR_FORMAT

    log = _get_log_from_string_or_log_instance(log_or_logname)

    if lang == 'ru':
        handler = CollectorHandler()

    elif lang == 'en':
        handler = EnCollectorHandler()

    else:
        raise NotImplementedError('Unknown language for log collector')

    handler.setLevel(log_level)
    handler.setFormatter(ContextRaspFormatter(format))
    log.addHandler(handler)

    return handler


def remove_collector(log_or_logname, collector):
    log = _get_log_from_string_or_log_instance(log_or_logname)
    log.removeHandler(collector)


class LangCollectors(object):
    def __init__(self, log_or_logname, format=None, log_level=logging.INFO, languages=None):
        from django.conf import settings

        self.log_or_logname = log_or_logname
        self.format = format or settings.LOG_FORMAT
        self.log_level = log_level
        self.languages = languages or ('ru',)

        self.handlers = dict()

    def __enter__(self):
        for lang in self.languages:
            self.handlers[lang] = get_collector(self.log_or_logname, self.format,
                                                self.log_level, lang)
        return self

    def __exit__(self, exception_type, exception_value, traceback):
        for lang in self.languages:
            remove_collector(self.log_or_logname, self.handlers[lang])

    def get_collected(self, lang, clean=True):
        return self.handlers[lang].get_collected(clean)


# Для записи логов на разных языках

class EnMixin(object):
    def format(self, record):
        with translation.override('en'):
            copy_of_record = copy(record)

            if isinstance(copy_of_record.msg, basestring):
                copy_of_record.msg = ugettext(copy_of_record.msg)

            return super(EnMixin, self).format(copy_of_record)


class EnFileHandler(EnMixin, logging.FileHandler):
    pass


class EnCollectorHandler(EnMixin, CollectorHandler):
    pass


@contextmanager
def get_collector_context(log_or_logname='', format=None, level=None):
    log_collector = get_collector(log_or_logname, format=format)

    level = level if level is not None else logging.INFO
    log_collector.setLevel(level or logging.INFO)
    try:
        yield log_collector

    finally:
        remove_collector(log_or_logname, log_collector)


# Копипаст handler'а рассылающего письма

class AdminEmailHandler(logging.Handler):
    """An exception log handler that emails log entries to site admins.

    If the request is passed as the first argument to the log record,
    request data will be provided in the
    """

    def emit(self, record):
        if '\n' in record.msg:
            message_subject, message_data = record.msg.split('\n', 1)

            if message_subject.endswith(':'):
                message_subject = message_subject[:-1]
        else:
            message_subject, message_data = record.msg, None

        try:
            request = record.request
            subject = '%s: %s' % (
                record.levelname,
                message_subject
            )
            filter = get_exception_reporter_filter(request)
            request_repr = filter.get_request_repr(request)
        except:
            subject = '%s: %s' % (
                record.levelname,
                message_subject
            )

            request = None
            request_repr = "Request repr() unavailable"

        if record.exc_info:
            stack_trace = '\n'.join(traceback.format_exception(*record.exc_info))
        else:
            stack_trace = 'No stack trace available'

        message = "%s\n\n%s" % (stack_trace, request_repr)

        boilerplate = []

        if request:
            try:
                boilerplate.append(request.build_absolute_uri())
            except:
                boilerplate.append("request.build_absolute_uri() failed")

        if record.exc_info:
            type_, value, _ = record.exc_info

            if value:
                try:
                    boilerplate.append("%s: %s" % (type_.__name__, value))
                except:
                    boilerplate.append("exception formatting failed")

        boilerplate.append(settings.PKG_VERSION)

        message = "%s\n\n%s" % ("\n".join(boilerplate), message)

        if message_data:
            message = "%s\n\n%s" % (message_data, message)

        mail.send_mail(subject, message, settings.SERVER_EMAIL, settings.ERRORS_RECIPIENTS, fail_silently=True)


# Вспомогательные функции
def _get_log_from_string_or_log_instance(log_or_logname):
    if isinstance(log_or_logname, basestring) or log_or_logname is None:
        return logging.getLogger(log_or_logname)
    else:
        return log_or_logname


class JsonFormatterContextKeys(_DictFormatter):
    DEFAULT_CONTEXT_KEYS_LOG = {
        'start_script_name', 'start_script_dt', 'start_caller_function', 'start_script_id', 'parent_pid'
    }

    def __init__(self, context_field='context', *args, **kwargs):
        self.context_field = context_field
        super(JsonFormatterContextKeys, self).__init__(datefmt=DT_FORMAT, *args, **kwargs)

    def format(self, record):
        record_dict = super(JsonFormatterContextKeys, self).log_record_to_dict(record)

        record_dict['pid'] = force_text(record_dict.pop('process'))
        if self.context_field in record_dict and isinstance(record_dict[self.context_field], dict):
            context = record_dict.pop(self.context_field)
            record_dict['context'] = {k: force_text(v) for k, v in context.items()}

            for key in self.DEFAULT_CONTEXT_KEYS_LOG:
                if key in record_dict['context']:
                    record_dict[key] = record_dict['context'][key]

        if 'exception' in record_dict:
            for k, v in record_dict['exception'].items():
                if isinstance(v, str):
                    record_dict['exception'][k] = v.decode('utf8')

        try:
            return force_bytes(json.dumps(record_dict, ensure_ascii=False))
        except TypeError:
            return force_bytes(json.dumps(
                {'created': record_dict['created'], 'message': record_dict['message']}, ensure_ascii=False))


def get_script_log_context(script_name=None):
    stack = inspect.stack()
    script_name = script_name or get_my_caller_file(stack)

    return {
        'start_script_name': script_name,
        'start_script_dt': environment.now().strftime(DT_FORMAT),
        'start_caller_function': inspect.stack()[1][3],
        'start_script_id': str(uuid.uuid4()),
        'parent_pid': os.getppid()
    }
