# -*- coding: utf-8 -*-
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from functools import partial

from django.conf import settings
from django.core.mail.message import EmailMultiAlternatives


log = logging.getLogger(__name__)


def mail_message(subject, message, recipients, fail_silently=False, connection=None, html_message=None):
    """Sends a message to the admins, as defined by the ADMINS setting."""
    if not recipients:
        return

    recipients_addresses = [a[1] for a in recipients]
    log.info('Sending email to: {}'.format(', '.join(recipients_addresses)))

    mail = EmailMultiAlternatives(
        u'{}{}'.format(settings.EMAIL_SUBJECT_PREFIX, subject),
        message,
        settings.SERVER_EMAIL,
        recipients_addresses,
        connection=connection
    )
    if html_message:
        mail.attach_alternative(html_message, 'text/html')
    mail.send(fail_silently=fail_silently)


mail_datachanges = partial(mail_message, recipients=settings.DATACHANGE_RCPNTS)
mail_tablo_error = partial(mail_message, recipients=settings.TABLO_ERROR_RCPNTS)
mail_process = partial(mail_message, recipients=settings.PROCESS_RCPNTS)
mail_train_import = partial(mail_message, recipients=settings.TRAIN_IMPORT_RCPNTS)
mail_free_stations = partial(mail_message, recipients=settings.RASP_FREE_STATIONS_RCPNTS)


def mail_with_attaches(subject, message, recipients, fail_silently=False, connection=None,
                       html_message=None, files=None):
    """Sends a message to the recipients, with attached files
    files = ((path, mimetype), ...)
    """
    if not recipients:
        return
    mail = EmailMultiAlternatives(
        u'{}{}'.format(settings.EMAIL_SUBJECT_PREFIX, subject),
        message,
        settings.SERVER_EMAIL,
        [a[1] for a in recipients],
        connection=connection
    )

    if html_message:
        mail.attach_alternative(html_message, 'text/html')

    if files:
        for path, mimetype in files:
            mail.attach_file(path, mimetype)

    mail.send(fail_silently=fail_silently)


mail_manager_with_attaches = partial(mail_with_attaches, recipients=settings.MANAGERS)
mail_process_with_attaches = partial(mail_with_attaches, recipients=settings.PROCESS_RCPNTS)
