# -*- coding: utf-8 -*-
import logging
from datetime import timedelta, datetime, time

import MySQLdb
from django.conf import settings

from common.db.switcher import get_connection_by_role


log = logging.getLogger(__name__)


def get_flag(name, role=None):
    conn = get_connection_by_role(role)
    db_name = conn.conn_params['db']
    cursor = conn.cursor()

    try:
        cursor.execute("SELECT state FROM `%s`.flags WHERE name = %%s" % db_name,
                       (name,))
    except MySQLdb.ProgrammingError:
        # Таблицы ещё нет в базе, скопируем глобальную
        cursor.execute("CREATE TABLE `%s`.flags LIKE %s.flags" % (
            db_name,
            settings.MAINTENANCE_DB
        ))

        cursor.execute("INSERT INTO `%s`.flags SELECT * FROM %s.flags" % (
            db_name,
            settings.MAINTENANCE_DB
        ))

        cursor.execute("SELECT state FROM `%s`.flags WHERE name = %s",
                       (db_name, name,))

    row = cursor.fetchone()
    if row:
        return row[0]
    else:
        raise KeyError("No flag %s found in database %s" % (name, db_name))


def get_flag_note(name, role=None):
    conn = get_connection_by_role(role)
    db_name = conn.conn_params['db']
    cursor = conn.cursor()

    cursor.execute("SELECT note FROM `%s`.flags WHERE name = %%s" % db_name, (name,))

    row = cursor.fetchone()

    if row:
        return row[0]
    else:
        raise KeyError("No flag %s found in database %s" % (name, db_name))


def set_flag(name, state, role=None, note=None):
    # Проверим, есть ли такой флаг
    get_flag(name, role)

    conn = get_connection_by_role(role)
    db_name = conn.conn_params['db']
    cursor = conn.cursor()
    cursor.execute("UPDATE `%s`.flags SET state = %%s, note = %%s WHERE name = %%s" % db_name, (state, note, name))
    cursor.execute("COMMIT")


def set_or_create_flag(name, state, description=u'', role=None):
    # Проверим, есть-ли такой флаг
    try:
        set_flag(name, state, role)
    except KeyError:
        conn = get_connection_by_role(role)
        db_name = conn.conn_params['db']
        cursor = conn.cursor()
        cursor.execute("INSERT INTO `%s`.flags (`name`, `state`, `description`) VALUES (%%s, %%s, %%s)" %
                       db_name, (name, state, description))
        cursor.execute("COMMIT")


class Flags(object):
    def set(self, name, state, note):
        set_flag(name, state, note=note)

    def __getitem__(self, name):
        return get_flag(name)

    def __setitem__(self, name, state):
        set_flag(name, state)

flags = Flags()


class MaintenanceState(object):
    warn_delta_seconds = 60 * 60 * 1

    def __init__(self, name, flag_value, verbose_name, block_admins=None,
                 planned_launch_time=None, days=None, warn_text=None, need_to_block_imports=False):
        self.name = name
        self.flag_value = flag_value
        self.verbose_name = verbose_name
        self.block_admins = block_admins or []
        self.planned_launch_time = planned_launch_time
        self.days = days or []
        self.warn_text = warn_text or u"Скоро возможно '{}' осталось:".format(
            verbose_name.replace(u'Идет', u'').strip().upper()
        )
        self.need_to_block_imports = need_to_block_imports

    def get_remaining_seconds(self, now):
        if self.planned_launch_time is not None:
            launch_dt = datetime.combine(now.date(), self.planned_launch_time)
            if launch_dt < now:
                launch_dt += timedelta(1)

            if self.days and launch_dt.isoweekday() not in self.days:
                return

            return (launch_dt - now).total_seconds()

    def need_to_warn(self, now):
        remaining_seconds = self.get_remaining_seconds(now)

        if remaining_seconds and remaining_seconds < self.warn_delta_seconds:
            return True

    def as_dict(self):
        copy_attrs = ['name', 'verbose_name', 'warn_text', 'block_admins', 'flag_value',
                      'need_to_block_imports']

        state_dict = {}
        for attr in copy_attrs:
            state_dict[attr] = getattr(self, attr)

        return state_dict


class JOBS(object):
    NO_JOB = MaintenanceState('NO_JOB', 0, u'Запуск скриптов разрешен')
    BIG_IMPORT = MaintenanceState('BIG_IMPORT', 1, u"Идет большой импорт", days=[7],
                                  need_to_block_imports=True, planned_launch_time=time(10, 0))
    SWITCH_BASES = MaintenanceState('SWITCH_BASES', 2, u"Идет переключение баз",
                                    block_admins=['blue', 'red', 'lime'], planned_launch_time=time(1, 0),
                                    need_to_block_imports=True)
    DAILY_UPDATE = MaintenanceState('DAILY_UPDATE', 3, u"Идет ежедневное обновление ТИС и СПК")
    UPDATE = MaintenanceState('UPDATE', 4, u"Идет обновление базы данных")
    AF_REIMPORT = MaintenanceState('AF_REIMPORT', 5, u"Идет переимпорт расписаний от А.Ф.")
    PREPARE_ALL = MaintenanceState('PREPARE_ALL', 6, u"Идет пересчет плоских таблиц",
                                   block_admins=['red'])
    AF_UPDATE_IMPORT = MaintenanceState('AF_UPDATE_IMPORT', 7, u'Идет импорт обновлений от А.Ф.')
    AF_ADDIMPORT = MaintenanceState('AF_ADDIMPORT', 8, u"Идет доимпорт расписаний от А.Ф.")
    RE_IMPORT = MaintenanceState('RE_IMPORT', 9, u"Идет переимпорт расписаний")
    BUSSTATION_IMPORT = MaintenanceState('BUSSTATION_IMPORT', 10, u"Идет импорт автобусных станций")
    PRECALC_TARIFFS = MaintenanceState('PRECALC_TARIFFS', 11, u"Идет пересчет тарифов")
    SUBWAY_IMPORT = MaintenanceState('SUBWAY_IMPORT', 12, u'Идет импорт метро')
    CACHE_EXPORT = MaintenanceState('CACHE_EXPORT', 13, u'Идет экспорт дальних рейсов')
    TWO_STAGE_IMPORT = MaintenanceState('TWO_STAGE_IMPORT', 14, u'Идет импорт автобусов в две стадии')
    TRIANGLE_IMPORT = MaintenanceState('TRIANGLE_IMPORT', 15, u'Идет импорт треугольный импорт')
    RED_META_IMPORT = MaintenanceState('RED_META_IMPORT', 16, u'Идет импорт метарейсов красного геоадмина')
    MIGRATION = MaintenanceState('MIGRATION', 17, u'Идет миграция',
                                 block_admins=['blue', 'red', 'lime'])
    TIS_UPDATE = MaintenanceState('TIS_UPDATE', 18, u'Обновляем ТИС')
    MAKE_DUMP = MaintenanceState('MAKE_DUMP', 19, u'Делается дамп базы')
    DEPLOY = MaintenanceState('DEPLOY', 20, u'Выкладка пакета')
    LIME_PATH_UPDATE = MaintenanceState('LIME_PATH_UPDATE', 21, u'Пересчет Лимонных данных')
    AF_TRAIN_TURNOVER = MaintenanceState('AF_TRAIN_TURNOVER', 22, u"Идет загрузка оборотов")
    EMERGENCY = MaintenanceState('EMERGENCY', 100, u'Аварийные работы')
    UNKNOWN_STATE = MaintenanceState('UNKNOWN_STATE', None, u'Неизвестное состояние')

    @classmethod
    def get(cls, flag_value):
        for state in cls.__dict__.values():
            if not isinstance(state, MaintenanceState):
                continue

            if state.flag_value == flag_value:
                return state

        return cls.UNKNOWN_STATE

    @classmethod
    def get_text(cls, flag_value):
        return cls.get(flag_value).verbose_name

    @classmethod
    def get_current_state(cls):
        if flags['maintenance']:
            return cls.get(flags['maintenance'])
        else:
            return None

    @classmethod
    def get_by_name(cls, flag_name):
        for state in cls.__dict__.values():
            if not isinstance(state, MaintenanceState):
                continue

            if state.name == flag_name:
                return state

        return None

    @classmethod
    def iter_states(cls):
        for state in cls.__dict__.values():
            if not isinstance(state, MaintenanceState):
                continue

            yield state

    @classmethod
    def get_next_warn(cls, now):
        remaining_seconds = None
        warn_state = None
        for state in cls.iter_states():
            if state.need_to_warn(now):
                if remaining_seconds is None or \
                        remaining_seconds > state.get_remaining_seconds(now):

                    warn_state = state
                    remaining_seconds = state.get_remaining_seconds(now)

        return warn_state


job = JOBS


def get_current_job():
    if flags['maintenance']:
        state = job.get(flags['maintenance'])
        return state.flag_value, state.verbose_name
    else:
        return None, None


def get_text(job_number):
    return job.get_text(job_number)
