# coding: utf-8

from __future__ import absolute_import, unicode_literals

import logging
import os
import subprocess
import sys
import time
from collections import OrderedDict
from copy import deepcopy
from importlib import import_module
from datetime import datetime

import six
from django.conf import settings
from django.db import connections
from ylog.context import log_context

from travel.rasp.admin.admin.maintenance_settings.models import Script, UserPermission
from common.data_api.file_wrapper.config import get_wrapper_creator
from common.db.switcher import switcher
from common.models.transport import TransportType
from common.settings import WorkInstance
from travel.rasp.admin.lib.exceptions import SimpleUnicodeException
from travel.rasp.admin.lib.logs import add_stream_handler, remove_stream_handler, stdstream_file_capturing, get_script_log_context
from travel.rasp.admin.lib.maintenance.deploy import finalize_deploy
from travel.rasp.admin.lib.maintenance.flags import job, flags
from travel.rasp.admin.lib.osutils import setproctitle
from travel.rasp.admin.lib.processes import child_finalizer
from travel.rasp.admin.scripts.utils.file_wrapper.registry import FileType
from travel.rasp.admin.scripts.utils.file_wrapper.uploaders import ObjWrapperSyncer
from travel.rasp.admin.scripts.utils.file_wrapper.mds_utils import get_admin_log_key


log = logging.getLogger(__name__)


IMPORT_OPTIONS = OrderedDict([
    ('mta', {
        'supplier_code': 'mta',
        'module': 'travel.rasp.admin.scripts.schedule.bus.import_mta',
        'description': 'МТА из залитого файла.',
        'download': False
    })
])

REIMPORT_OPTIONS = [
    (import_option_key, IMPORT_OPTIONS[import_option_key]['description'])
    for import_option_key in IMPORT_OPTIONS
]

WORK_INSTANCE_PATH = settings.INSTANCE_PATHS[WorkInstance.code]
WORK_EXPORT_PATH = getattr(settings, 'WORK_EXPORT_PATH', '{}/www/db/scripts/export/'.format(WORK_INSTANCE_PATH))

SCRIPTS = {
    'switch': {
        'entry_point': 'travel.rasp.admin.scripts.cron_switch_bases',
        'args': ('-v',),
    },
    'switch_continue': {
        'entry_point': 'travel.rasp.admin.scripts.cron_switch_bases',
        'args': ('-v', '--continue'),
    },
    'switch_no_prepare_all': {
        'entry_point': 'travel.rasp.admin.scripts.cron_switch_bases',
        'args': ('-v', '--no-prepare-all'),
    },
    'update': {
        'entry_point': 'travel.rasp.admin.app:manage',
        'args': ('update', '--verbosity=1'),
    },
    'import_all': {
        'entry_point': 'travel.rasp.admin.scripts.cron_schedule',
        'args': ('-v',),
    },
    'import_all_continue': {
        'entry_point': 'travel.rasp.admin.scripts.cron_schedule',
        'args': ('-v', '--continue'),
    },
    'prepare_all': {
        'entry_point': 'travel.rasp.admin.scripts.prepare_all',
        'args': ('--standalone', '-v'),
    },
    'prepare_all_continue': {
        'entry_point': 'travel.rasp.admin.scripts.prepare_all',
        'args': ('--standalone', '-v', '--continue'),
    },
    'prepare_all_short': {
        'entry_point': 'travel.rasp.admin.scripts.prepare_all',
        'args': ('--standalone', '--short', '-v'),
        'users': ['af1461', 'mityan', 'strannitsa77rus'],
    },
    'prepare_all_partial': {
        'entry_point': 'travel.rasp.admin.scripts.prepare_all',
        'args': ('--standalone', '--partial', '-v'),
    },
    'prepare_all_partial_short': {
        'entry_point': 'travel.rasp.admin.scripts.prepare_all',
        'args': ('--standalone', '--partial', '--short', '-v'),
        'users': ['af1461', 'mityan', 'strannitsa77rus'],
    },
    'prepare_all_partial_short_skip_min_prices': {
        'entry_point': 'travel.rasp.admin.scripts.prepare_all',
        'args': ('--standalone', '--partial', '--short', '--skip-min-prices', '-v'),
        'users': ['af1461', 'mityan', 'strannitsa77rus'],
    },
    'addimport_af': {
        'entry_point': 'travel.rasp.admin.scripts.schedule.re_import_af_schedule',
        'args': ('python', 'scripts/schedule/re_import_af_schedule.py', '-va'),
        'users': ['af1461'],
        'select': lambda: [(t.code, t.L_title()) for t in TransportType.objects.all()],
    },
    'reimport_af': {
        'entry_point': 'travel.rasp.admin.scripts.schedule.re_import_af_schedule',
        'args': ('-v',),
        'users': ['af1461'],
        'select': lambda: [(t.code, t.L_title()) for t in TransportType.objects.all()],
    },
    're_import': {
        'entry_point': 'travel.rasp.admin.scripts.schedule.re_import',
        'args': ('-v',),
        'select': REIMPORT_OPTIONS,
        'users': ['strannitsa77rus'],
    },
    're_import_with_lock': {
        'entry_point': 'travel.rasp.admin.scripts.schedule.re_import',
        'lock_tablo': True,
        'set_flag': job.RE_IMPORT.flag_value,
        'args': ('-v', '--ignore-flag'),
        'select': REIMPORT_OPTIONS,
        'users': ['strannitsa77rus'],
    },
    'precalc_tariffs': {
        'entry_point': 'travel.rasp.admin.scripts.precalc_tariffs',
        'args': ('-v',),
    },
    'invalidate_cache': {
        'entry_point': 'travel.rasp.admin.scripts.invalidate_cache',
        'args': ('-v',),
    },
    'export_main_threads': {
        'entry_point': 'travel.rasp.admin.scripts.export.export_main_threads',
        'args': ('-v',),
    },
    'update_tis': {
        'callable': 'travel.rasp.admin.scripts.schedule.tis_train.import_tis.update_tis',
        'set_flag': job.TIS_UPDATE.flag_value,
    },
    'check_suburban_mask_intersections': {
        'entry_point': 'travel.rasp.admin.scripts.schedule.check_suburban_mask_intersections',
        'args': ('-v',),
    },
    'make_dump': {
        'entry_point': 'travel.rasp.admin.scripts.make_fresh_dump',
        'args': ('-v', '-p', 'manual', WORK_EXPORT_PATH),
    },
    'load_fresh_dump_to_tmpdb': {
        'entry_point': 'travel.rasp.admin.scripts.load_latest_db',
        'args': ('-v', 'load-to-tmpdb', '-t'),
        'select': (
            ('any', 'Самый последний, не важно какая база'),
            ('work_db', 'work_db'),
            ('service_db', 'service_db'),
        ),
    },
    'update_from_tmpdb': {
        'entry_point': 'travel.rasp.admin.scripts.load_latest_db',
        'args': ('-v', 'update-from-tmpdb'),
    },
    'check_route_changes': {
        'entry_point': 'travel.rasp.admin.scripts.check_route_changes',
        'args': ('-v',),
    },
    'finalize_deploy': {
        'callable': finalize_deploy,
    },
    'load_train_turnovers': {
        'entry_point': 'travel.rasp.admin.scripts.suburban_events.reimport_turnovers',
        'set_flag': job.AF_TRAIN_TURNOVER.flag_value,
    }
}

if getattr(settings, 'APPLIED_CONFIG', None) != 'production':
    SCRIPTS['download_production_static'] = {
        'entry_point': 'travel.rasp.admin.scripts.utils.download_production_static',
        'args': ('-v',),
    }


SCRIPT_DB_PARAMS = 'title', 'button_text', 'warning_text'


def get_script_params(scriptname):
    script = deepcopy(SCRIPTS[scriptname])

    try:
        script_from_db = Script.objects.filter(code=scriptname)[0]

        for param in SCRIPT_DB_PARAMS:
            script[param] = getattr(script_from_db, param)

    except KeyError:
        pass

    return script


def get_allowed_scripts():
    all_scripts = Script.objects.order_by('order')

    scripts = [s.code for s in all_scripts if getattr(s, 'on_%s' % settings.INSTANCE_ROLE.code)]

    return scripts


def has_script_permission(user, script_name):
    """
    Проверяем может ли пользователь на текущем инстансе запускать данный скрипт.
    @param user:
    @param script_name:
    """
    if script_name not in get_allowed_scripts():
        return False

    script = SCRIPTS.get(script_name)
    if script is None:
        return False

    if user.is_superuser:
        return True

    user_has_permission = UserPermission.objects.filter(
        user__username=user.username, script__code=script_name, can_run=True).exists()

    if user_has_permission:
        return True

    return False


def run_script(script_name, request, username='system', fobj_to_close=None):
    if fobj_to_close is None:
        fobj_to_close = []

    script = SCRIPTS[script_name]
    script['code'] = script_name

    return run_admin_task(script, request, username, fobj_to_close)


def run_admin_task(task, request, username, fobj_to_close=None):
    if fobj_to_close is None:
        fobj_to_close = []

    task_code = task['code']

    pid = os.fork()

    if pid:
        # Немного ждем, чтобы скрипт успел поставить флаг
        time.sleep(3)

        return 'Скрипт {} запущен'.format(task_code)

    else:
        # Меняем директорию в порожденном процессе,
        # на случай если будут запускаться скрипты проекта,
        # которые пишут в файл в текущей директории
        os.chdir(settings.PROJECT_PATH)

        for c in connections.all():
            c.close()

        for fobj in fobj_to_close:
            fobj.close()

        switcher.sync_db()
        setproctitle('rasp-{instance}-{db_name}: {process}'.format(
            instance=settings.INSTANCE_ROLE.code, db_name=switcher.get_db_alias(), process=task_code
        ))

        with child_finalizer():
            try:
                _run_admin_task_in_this_process(request, task, username)
            except Exception:
                log.exception(u'Ошибка запуска скрипта')


def _run_admin_task_in_this_process(request, task, username):
    task_code = task['code']
    args = task.get('args', tuple())

    timepoint = datetime.today().strftime("%Y-%m-%d_%H.%M.%S")
    dirpath = os.path.join(settings.LOG_PATH, 'special/admin_run')

    if not os.path.exists(dirpath):
        os.makedirs(dirpath)

    log_name = "%s_%s_%s.log" % (timepoint, settings.INSTANCE_ROLE.code, task_code)
    logpath = os.path.join(dirpath, log_name)
    file_wrapper = get_wrapper_creator(FileType.ADMIN_RUNS, key=get_admin_log_key(log_name)).get_file_wrapper(logpath)

    flag = task.get('set_flag')

    if 'select' in task:
        option = request.POST.get('select_' + task_code)
        args += (option,)

    with ObjWrapperSyncer(file_wrapper, interval=settings.MDS_LOG_UPLOAD_INTERVAL), \
            open(logpath, 'w') as f, \
            stdstream_file_capturing(f):
        add_stream_handler(log, f)

        if task.get('lock_tablo'):
            tablo_lock_file = '/tmp/rasp-cron_tablo_%s.lock' % settings.INSTANCE_ROLE.code

            args = ['flock', tablo_lock_file, '-c', ' '.join(args)]

        if flag:
            if flags['maintenance']:
                log.warning('!- Нелья запускать скрипт, идет работа с базой данных')
                return
            else:
                log.info('!- Ставим флаг')
                flags['maintenance'] = flag

        log.info(
            'Запускаем скрипт {}, пользователь {}, {} инстанс, база {} {}'.format(
                task_code, username, settings.INSTANCE_ROLE.title, settings.INSTANCE_ROLE.role, switcher.get_db_alias()
            )
        )

        if 'callable' in task:
            with log_context(**get_script_log_context(script_name=task['callable'])):
                try:
                    add_stream_handler('', f)

                    func = task['callable']

                    if isinstance(func, six.string_types):
                        module = import_module('.'.join(func.split('.')[:-1]))
                        func = getattr(module, func.split('.')[-1])

                    func(*args, **task.get('kwargs', {}))
                except SimpleUnicodeException as e:
                    remove_stream_handler('', f)
                    log.error('Ошибка запуска скрипта. ' + e.msg_template, *e.msg_args)
                except Exception:
                    remove_stream_handler('', f)
                    log.exception('Ошибка запуска скрипта')
                finally:
                    remove_stream_handler('', f)

        else:
            process_env = os.environ.copy()
            if 'entry_point' in task:  # admin script call in Arcadia
                process_env.update({
                    'Y_PYTHON_ENTRY_POINT': task['entry_point'],
                })
                p_args = [sys.executable]
                p_args.extend(args)
            else:  # another external calls
                p_args = args
            process = subprocess.Popen(p_args, env=process_env, cwd=settings.PROJECT_PATH,
                                       stdin=None, stdout=f, stderr=subprocess.STDOUT, close_fds=True)
            process.wait()

        if flag:
            log.info('!- Убираем флаг')
            flags['maintenance'] = 0

        log.info(u"Завершили работу")
