# coding: utf8
from __future__ import print_function, unicode_literals, absolute_import

import json
import requests
from django.conf import settings
from google.protobuf import json_format

from common.data_api.tvm.instance import tvm_factory
from common.settings.configuration import Configuration
from common.settings.utils import define_setting
import travel.rasp.admin.lib.maps.protobuf_models.yandex.maps.proto.search.kind_pb2 as kind_pb2
from travel.rasp.admin.lib.maps.protobuf_models.yandex.maps.proto.common2.response_pb2 import Response
from travel.rasp.admin.lib.maps.protobuf_models.yandex.maps.proto.search.geocoder_pb2 import GEO_OBJECT_METADATA
from travel.rasp.admin.lib.maps.protobuf_models.yandex.maps.proto.search.geocoder_internal_pb2 import TOPONYM_INFO


define_setting(
    'GEOCODER_URL',
    {Configuration.PRODUCTION: 'http://addrs.yandex.ru:17140/yandsearch'},
    default='http://addrs-testing.search.yandex.net/search/stable/yandsearch'
)

kind_to_str = {
    kind_pb2.PROVINCE: 'province',
    kind_pb2.LOCALITY: 'locality',
    kind_pb2.STREET: 'street',
    kind_pb2.RAILWAY_STATION: 'railway'
}


class GeocoderClient(object):
    """https://a.yandex-team.ru/arc/trunk/arcadia/extsearch/geo"""
    def __init__(self, origin='rasp', tvm_ticket=None, lang=None):
        if not tvm_ticket:
            tvm_ticket = tvm_factory.get_provider().get_ticket(settings.TVM_MAPS_API)

        self.origin = origin
        self.lang = lang
        self.tvm_ticket = tvm_ticket

    def _call(self, text, max_results):
        params = {
            'type': 'geo',
            'ms': 'pb',
            'origin': self.origin,
            'text': text,
            'results': max_results
        }
        if self.lang:
            params['lang'] = self.lang

        headers = {'X-Ya-Service-Ticket': self.tvm_ticket} if self.tvm_ticket else {}

        response = requests.get(
            settings.GEOCODER_URL,
            params=params,
            headers=headers,
            timeout=settings.GEOCODER_SERVICE_TIMEOUT
        )
        response.raise_for_status()

        return response.content

    def search(self, text, max_results=10, kinds=None):
        raw_response = self._call(text, max_results)

        proto_response = Response()
        proto_response.ParseFromString(raw_response)

        result_objs = []

        for geo_object in proto_response.reply.geo_object:
            for meta in geo_object.metadata:
                meta_data = meta.Extensions[GEO_OBJECT_METADATA]
                kind = kind_to_str.get(meta_data.address.component[-1].kind[0]) if meta_data.address.component else None
                if not kinds or (kind and kind in kinds):
                    toponym = meta_data.Extensions[TOPONYM_INFO]
                    if toponym:
                        result_objs.append({
                            'object': geo_object,
                            'geo_id': toponym.geoid,
                            'kind': kind,
                            'name': geo_object.name.decode('utf8'),
                            'coords': {
                                'lat': toponym.point.lat,
                                'lon': toponym.point.lon
                            }
                        })

        return result_objs, proto_response

    @staticmethod
    def get_dict_response(proto_response):
        return json.loads(json_format.MessageToJson(proto_response))
