# coding: utf-8

from __future__ import unicode_literals

from datetime import timedelta

from common.utils.date import RunMask, daterange


def mask_from_days_of_week(bounds, days, today=None):
    """
    Делаем маску из формата 1234567
    Воскресенье это 7

    :type bounds: MaskBounds
    :type days: six.text_type
    :type today: date | None
    :rtype: RunMask
    """

    def condition(day):
        return days and str(day.isoweekday()) in days

    return mask_from_day_condition(bounds, condition, today)


def odd_mask(bounds, today=None):
    """
    По нечетным дням
    :type bounds: MaskBounds
    :type today: date | None
    :rtype: RunMask
    """
    def condition(day):
        return day.day % 2 != 0

    return mask_from_day_condition(bounds, condition, today)


def even_mask(bounds, today=None):
    """
    По четным дням
    :type bounds: MaskBounds
    :type today: date | None
    :rtype: RunMask
    """

    def condition(day):
        return not day.day % 2

    return mask_from_day_condition(bounds, condition, today)


def empty_mask(today=None):
    """
    :type today: date | None
    :rtype: RunMask
    """
    return RunMask(today=today)


def daily_mask(bounds, today=None):
    """
    :type bounds: MaskBounds
    :type today: date | None
    :rtype: RunMask
    """
    return mask_from_day_condition(bounds, lambda x: True, today)


def through_the_day_mask(bounds, first_day, today=None):
    """
    Через день
    :type bounds: MaskBounds
    :type first_day: date
    :param first_day: - первый день хождения
    :type today: date | None
    :rtype: RunMask
    """
    return gap_days_mask(bounds, 1, first_day, today)


def gap_days_mask(bounds, days_in_gap, first_day, today=None):
    """
    Через days_in_gap дней, например, через день - days_in_gap == 1
    :type bounds: MaskBounds
    :type days_in_gap: int
    :type first_day: date
    :param first_day: - первый день хождения
    :type today: date | None
    :rtype: RunMask
    """
    mask = RunMask(today=today)

    for day in daterange(first_day, bounds.end_date + timedelta(1), timedelta(days_in_gap + 1)):
        if day >= bounds.start_date:
            mask[day] = True

    return mask


def mask_from_day_condition(bounds, condition, today=None):
    """
    :type bounds: MaskBounds
    :type condition: (datetime.date) -> bool
    :type today: date | None
    :rtype: RunMask
    """

    mask = RunMask(today=today)
    for day in bounds.iter_dates():
        if condition(day):
            mask[day] = True

    return mask


def one_day_mask(bounds, day, today=None):
    """
    :type bounds: MaskBounds
    :type day: date
    :type today: date | None
    :rtype: RunMask
    """
    mask = RunMask(today=today)

    if bounds.start_date <= day <= bounds.end_date:
        mask[day] = True

    return mask
