# -*- coding: utf-8 -*-

import logging
from urllib2 import URLError

from django.conf import settings
from django.contrib.auth.models import User, AnonymousUser
from blackbox import XmlBlackbox, BlackboxError, FIELD_LOGIN

from common.data_api.tvm.instance import tvm_factory

log = logging.getLogger(__name__)


FIELD_LANG = ('userinfo.lang.uid', 'lang')


blackbox = XmlBlackbox(
    url=settings.YANDEX_TEAM_BLACKBOX_URL,
    dbfields=[FIELD_LOGIN, FIELD_LANG],
)


class YandexUser(object):
    '''
    Модель яндексового пользователя.
    '''
    def __init__(self, uid, is_lite, fields, need_reset):
        self.uid = uid
        self.is_lite_uid = is_lite
        self.fields = fields
        self.need_reset = need_reset

    def is_authenticated(self):
        return not self.is_lite()

    def is_lite(self):
        return bool(self.is_lite_uid)

    def __getattr__(self, field):
        try:
            return self.fields[field]
        except KeyError:
            raise AttributeError(field)

    def get_django_user(self):
        try:
            return User.objects.get(username=self.login)
        except User.DoesNotExist:
            return AnonymousUser()

    def __str__(self):
        return self.login


class AnonymousYandexUser(object):
    '''
    Заглушка незалогиненного пользователя.
    '''
    login = '(Not logged in)'
    uid = None
    lang = None

    def is_authenticated(self):
        return False

    def is_lite(self):
        return False

    def __str__(self):
        return self.login

    def get_django_user(self):
        return AnonymousUser()


def add_yauser(request):
    """Возвращаемое значение: нужен-ли редирект на MDA"""

    request.yauser = AnonymousYandexUser()

    if settings.DISABLE_YAUTH:
        return False

    log.info('Try to add ya user')

    sessionid = request.COOKIES.get('Session_id')
    userip = request.META.get('HTTP_X_FORWARDED_FOR', None)
    if userip:
        userip = userip.split(',')[0]
    else:
        userip = request.META.get('REMOTE_ADDR', None)
    host = request.META.get('HTTP_HOST')

    log.info('session_id: %s', sessionid)

    if sessionid is None:
        return

    try:
        tvm_ticket = tvm_factory.get_provider().get_ticket(settings.TVM_BLACKBOX_YATEAM)

        sessionid_kwargs = {'headers': {'X-Ya-Service-Ticket': tvm_ticket}}

        session = blackbox.sessionid(sessionid, userip, host, **sessionid_kwargs)

        # Кука Session_id есть и имеет статус VALID, то пользователь авторизован
        if not session.valid:
            log.warning('Session is invalid')
            return

        if session.uid is None:
            log.warning('Session uid is None')
            return

        request.yauser = YandexUser(int(session.uid), False, session.fields, session.redirect)
    except (URLError, BlackboxError):  # если blackbox недоступен, то пользователь анонимен
        log.exception(u'Ошибка авторизации')

    except Exception:
        log.exception(u'Ошибка обращения в blackbox')
        raise


class MDAMiddleware(object):
    def process_request(self, request):
        add_yauser(request)

        if settings.DISABLE_YAUTH:
            return

        if request.method != 'GET':
            return

        if request.is_ajax():
            return

        if request.yauser.get_django_user().is_authenticated():
            return

        # FIXME: пока убираем редирект, т.к. для некоторых ручек не нужна обязательная авторизация
        # redirect_url = 'https://passport.yandex-team.ru/passport?{}'.format(urlencode(dict(
        #     mode='auth', retpath=request.build_absolute_uri()
        # )))
        #
        # return HttpResponseRedirect(redirect_url)
        #
        # @simon-ekb
        # Это устаревщий способ. https://doc.yandex-team.ru/Passport/passport-modes/reference/auth.xml
        # Нужно ходить на https://passport.yandex-team.ru/auth?
