# -*- coding: utf-8 -*-

import logging
import os
import Queue
import signal
import traceback

import multiprocessing

from django.db import connection


log = logging.getLogger('rasp.lib.parallelize')


class WorkerException(RuntimeError):
    pass


class Master(object):
    def __init__(self, func, iterable, n_workers=8):
        self.func = func
        self.workers = set()
        self.queue_length = 0
        self.iterator = iter(iterable)
        self.n_workers = n_workers
        self.queue = multiprocessing.Queue()

    def terminate_workers(self):
        for pid in self.workers:
            try:
                os.kill(pid, signal.SIGKILL)
                os.waitpid(pid)
            except:
                pass

    def spawn_worker_if_data_left(self):
        try:
            item = self.iterator.next()
        except StopIteration:
            return

        try:
            pid = os.fork()
        except:
            self.terminate_workers()
            raise

        if pid:
            self.workers.add(pid)

            self.queue_length += 1

            return

        connection.close()

        try:
            result = self.func(item)
        except Exception:
            message = traceback.format_exc()
            result = WorkerException(message)

        self.queue.put(result)

        self.queue.close()

        self.queue.join_thread()

        os._exit(0)

    def unwrap_exception(self, result):
        if isinstance(result, WorkerException):
            self.terminate_workers()

            raise result

        return result

    def handle_workers(self, wait=False):
        for pid in list(self.workers):
            try:
                pid, info = os.waitpid(pid, 0 if wait else os.WNOHANG)
            except OSError, e:
                if e.errno == 10:  # No child processes
                    self.workers.remove(pid)

                    return

                raise

            if (pid, info) == (0, 0):  # No workers to handle
                return

            status, killsig = divmod(info, 256)

            if killsig != 0:
                self.terminate_workers()

                raise RuntimeError("Child process %d was killed by signal %d" % (pid, killsig % 127))

            if status != 0:
                self.terminate_workers()

                raise RuntimeError("Child process exited with status %d" % status)

            self.workers.remove(pid)

    def run(self):
        # Запускаем n воркеров, каждому свой item
        for i in range(self.n_workers):
            self.spawn_worker_if_data_left()

        for result in self.get_results():
            yield result

            # Создаем взамен нового воркера
            self.spawn_worker_if_data_left()

    def get_results(self):
        were_no_workers = False

        while self.queue_length:
            try:
                result = self.queue.get(timeout=60)
            except Queue.Empty:
                self.handle_workers()

                if not self.workers:
                    if were_no_workers:
                        # На предущей итерации все воркеры уже завершились,
                        # а новых результатов за 60 секунд не появилось, значит
                        # что-то зависло.
                        raise RuntimeError("All workers dead, and no new results for 60 seconds")

                    were_no_workers = True
                else:
                    were_no_workers = False

                continue

            self.handle_workers()

            were_no_workers = False

            self.queue_length -= 1

            yield self.unwrap_exception(result)

        self.handle_workers(wait=True)


def fork_imap_unordered(func, iterable, n_workers=8):
    return Master(func, iterable, n_workers).run()
