# coding: utf-8

from __future__ import unicode_literals

from django.conf import settings
from django.core.exceptions import ImproperlyConfigured
from django.core.mail.backends.smtp import EmailBackend as SMTPEmailBackend
from django.core.mail.message import sanitize_address


class EmailBackend(SMTPEmailBackend):
    def send_messages(self, email_messages):
        """
        Sends one or more EmailMessage objects and returns the number of email
        messages sent.
        """
        if not email_messages:
            return
        with self._lock:
            new_conn_created = self.open()
            if not self.connection:
                # We failed silently on open().
                # Trying to send would be pointless.
                return
            num_sent = 0
            for message in email_messages:
                sent = self._redirect_send(message)
                if sent:
                    num_sent += 1
            if new_conn_created:
                self.close()
        return num_sent

    def _redirect_send(self, email_message):
        "Overrides default _send from smtp backend and redirect all mail to one recipient"
        if not email_message.recipients():
            return False

        if not hasattr(settings, 'REDIRECT_EMAILS'):
            raise ImproperlyConfigured('Add REDIRECT_EMAILS to settings')

        encoding = email_message.encoding or settings.DEFAULT_CHARSET
        from_email = sanitize_address(email_message.from_email, encoding)
        recipients = [sanitize_address(addr, encoding) for addr in settings.REDIRECT_EMAILS]

        emails_text_parts = []

        if email_message.to:
            emails_text_parts.append('to: {}'.format(', '.join(email_message.to)))
        if email_message.cc:
            emails_text_parts.append('cc: {}'.format(', '.join(email_message.cc)))
        if email_message.bcc:
            emails_text_parts.append('bcc: {}'.format(', '.join(email_message.bcc)))

        email_message.to = recipients
        email_message.cc = []
        email_message.bcc = []

        email_prefix = getattr(settings, 'REDIRECT_EMAILS_SUBJECT_PREFIX', 'REB')

        email_message.subject = '{}: {} {{{}}}'.format(email_prefix, email_message.subject,
                                                       ', '.join(emails_text_parts))

        try:
            self.connection.sendmail(from_email, recipients, email_message.message().as_bytes(linesep='\r\n'))
        except:
            if not self.fail_silently:
                raise
            return False
        return True

