# -*- coding: utf-8 -*-

import os
from zipfile import ZipFile, BadZipfile
from chardet.universaldetector import UniversalDetector
from collections import OrderedDict

from travel.rasp.admin.lib.exceptions import UnpackError
from travel.rasp.admin.lib.fileutils import remove_tmp_dir
from travel.rasp.admin.lib.tmpfiles import temp_manager


def unpack_zip_file(fileobj, temporary_directory=None, ignore_macos=True):
    if not temporary_directory:
        temporary_directory = temp_manager.get_tmp_dir('lib/unzip')

    remove_tmp_dir(temporary_directory)

    try:
        zip_file = ZipFile(fileobj)
    except (IOError, BadZipfile):
        raise UnpackError(u"Ошибка открытия zip файла")

    bad_file_name = zip_file.testzip()
    if zip_file.testzip() is not None:
        raise UnpackError(u"Ошибка чтения zip файла, плохой файл %s" % repr(bad_file_name))

    os.makedirs(temporary_directory)

    detector = UniversalDetector()

    not_unicode_file_count = 0
    for name in zip_file.namelist():
        if not isinstance(name, unicode):
            detector.feed(name)
            not_unicode_file_count += 1

    detector.close()

    if not_unicode_file_count and detector.result['confidence'] < 0.5:
        raise UnpackError(u"Непонятная кодировка имен файлов")

    name_encoding = detector.result['encoding']

    file_map = OrderedDict()
    for archive_file_path in zip_file.namelist():
        unicode_archive_file_path = (
            archive_file_path
            if isinstance(archive_file_path, unicode)
            else archive_file_path.decode(name_encoding)
        )
        out_archive_file_path = unicode_archive_file_path.encode('utf8')

        output_path = os.path.join(temporary_directory, out_archive_file_path)
        if ignore_macos and output_path.startswith('__MACOSX'):
            continue

        if output_path.endswith('/'):
            os.makedirs(output_path)
        else:
            if not os.path.exists(os.path.dirname(output_path)):
                os.makedirs(os.path.dirname(output_path))

            outfile = open(output_path, 'wb')
            outfile.write(zip_file.read(archive_file_path))
            outfile.close()

        file_map[unicode_archive_file_path] = output_path

    file_map_new = OrderedDict()
    for key in sorted(file_map.keys()):
        file_map_new[key] = file_map[key]

    return file_map_new
