# coding: utf-8

import json
import logging
import re
import string
import time as os_time
from StringIO import StringIO
from contextlib import closing
from datetime import datetime, timedelta
from functools import update_wrapper
from traceback import format_exc

import requests
from django import forms
from django.conf import settings
from django.conf.urls import url
from django.contrib import admin, messages
from django.contrib.admin import SimpleListFilter
from django.contrib.admin.options import csrf_protect_m
from django.contrib.admin.utils import unquote
from django.contrib.admin import widgets
from django.contrib.admin.widgets import AdminDateWidget, AdminFileWidget, ForeignKeyRawIdWidget
from django.contrib.contenttypes.models import ContentType
from django.core.exceptions import ObjectDoesNotExist
from django.core.mail import send_mail
from django.db import models, transaction
from django.db.models.fields.related import ManyToOneRel
from django.http import HttpResponseRedirect
from django.template.response import TemplateResponse
from django.utils.decorators import method_decorator
from django.utils.http import urlencode
from django.utils.safestring import mark_safe
from django.utils.translation import ugettext_lazy as _

from common.models.geo import Point, Settlement, Station
from common.models.partner import (
    BlablacarToken, DohopVendor, Partner, RegionalizePartnerQueryRule, PartnerLogoImageField,
    PartnerLogoSvgImageField, Svg2PngImageField, PartnerEmail
)
from common.models_utils.i18n import L_field
from common.utils.httpresponses import jsonp_response
from common.utils.safe_xml_parser import safe_parse_xml, safe_xml_fromstring
from common.xgettext.i18n import xngettext, gettext
from travel.rasp.admin.lib.admin_options import RaspExportModelAdmin
from order.models import CoachInfoBinding, CoachInfo, ServiceClass, CoachService, QueryBlackList

log = logging.getLogger(__name__)


jsonp_response_m = method_decorator(jsonp_response)
MAX_UPLOADED_PARTNER_LOGO_SIZE_IN_BYTES = 1 * 1024 * 1024

BILLING_API_TIMEOUT = 3


class CoachInfoBindingInline(admin.TabularInline):
    formfield_overrides = {
        models.CharField: {'widget': widgets.AdminTextInputWidget(attrs={'style': 'width: 11em;'})},
        models.TextField: {'widget': widgets.AdminTextareaWidget(attrs={'style': 'width: 11em; height: 1.5em;'})},
    }
    fields = ('klass', 'coach_subtype_code', 'road', 'service_class', 'international_service_class', 'train_number',
              'coach_number', 'priority', 'info', 'direction_confirmed', 'im_car_scheme_id')

    model = CoachInfoBinding


class CoachInfoAdmin(RaspExportModelAdmin):
    list_display = ('name',)
    inlines = [
        CoachInfoBindingInline,
    ]
    fieldsets = (
        ('Profile field', {
            'fields': ('name', 'two_storey', 'image', 'schema', 'seat_selection_algo')
        }),
        (_(u'Признаки мест'), {
            'fields': ('upper_places', 'side_places', 'near_toilet_places', 'group_places', 'details')
        }),
        (_(u'Новый svg'), {'fields': ('svg_schema',)}),
        (_(u'Правила тарификации'), {'fields': ('place_price_rules',)})
    )
    filter_horizontal = ('place_price_rules',)

admin.site.register(CoachInfo, CoachInfoAdmin)


class CoachInfoBindingAdmin(RaspExportModelAdmin):
    list_display = (
        'klass', 'coach_subtype_code', 'road', 'service_class', 'international_service_class',
        'train_number', 'coach_number', 'priority', 'info', 'direction_confirmed', 'im_car_scheme_id',
    )

admin.site.register(CoachInfoBinding, CoachInfoBindingAdmin)


class ServiceClassAdmin(RaspExportModelAdmin):
    filter_horizontal = ('services',)
    list_display = ('name', 'owner', 'code', 'is_brand', 'note')

admin.site.register(ServiceClass, ServiceClassAdmin)


class PartnerEmailAdminInline(admin.TabularInline):
    model = PartnerEmail


class PartnerAdminImageWidget(AdminFileWidget):
    def render(self, name, value, attrs=None, renderer=None):
        output = []

        if value and getattr(value, "url", None):
            output.append(
                u' <img src="%s?%s" alt="%s" />' % (value.url, os_time.time() * 1000, value)
            )

        output.append(super(AdminFileWidget, self).render(name, value, attrs=attrs, renderer=renderer))

        return mark_safe(u''.join(output))


class BillingOrderWidget(forms.TextInput):
    def render(self, name, value, *args, **kwargs):
        html = super(BillingOrderWidget, self).render(name, value, *args, **kwargs)
        if value:
            html += mark_safe(
                u'\n <a href="http://partner.rasp.yandex.ru/campaign.xml?id=%d">%s</a>' % (
                    value,
                    _(u'Перейти в партнерский интерфейс')
                )
            )

        return html


class PartnerAdminForm(forms.ModelForm):
    def clean_query_module_name(self):
        allowed_chars = string.ascii_lowercase + string.digits + '_'
        query_module_name = self.cleaned_data['query_module_name']

        if query_module_name and any([char not in allowed_chars for char in query_module_name]):
            raise forms.ValidationError(u'В имени модуля разрешены только: "%s"' % allowed_chars)

        return query_module_name

    class Meta:
        model = Partner
        widgets = {
            'billing_order_id': BillingOrderWidget
        }
        exclude = []


class DohopResidencesFilter(SimpleListFilter):
    title = _(u'Страны')
    parameter_name = 'dohop_residences__contains'

    def lookups(self, request, model_admin):
        return (
            ('RU', _(u'Россия')),
            ('UA', _(u'Украина')),
            ('TR', _(u'Турция')),
        )

    def queryset(self, request, queryset):
        if self.value():
            return queryset.filter(dohop_residences__contains=self.value())

        else:
            return queryset


class DohopVendorAdminForm(forms.ModelForm):
    class Meta:
        model = DohopVendor
        exclude = []

    def check_file_size(self, field_name):
        field_content = self.cleaned_data.get(field_name)

        if field_content and len(field_content) > MAX_UPLOADED_PARTNER_LOGO_SIZE_IN_BYTES:
            raise forms.ValidationError(
                _(u'Файл %s превышает допустимый лимит %.2f МБ')
                % (field_name, MAX_UPLOADED_PARTNER_LOGO_SIZE_IN_BYTES / 1024. / 1024.)
            )

    def check_svg(self, field_name):
        field_content = self.cleaned_data.get(field_name)
        try:
            if field_content:
                safe_parse_xml(field_content)

        except Exception:
            raise forms.ValidationError(_(
                u'%s: формат файла не поддерживается' % field_name
            ))

    def clean_logo_svg_ru(self):
        field_name = 'logo_svg_ru'
        self.check_file_size(field_name)
        self.check_svg(field_name)

        return self.cleaned_data.get(field_name)

    def clean_logo_svg_ua(self):
        field_name = 'logo_svg_ua'
        self.check_file_size(field_name)
        self.check_svg(field_name)

        return self.cleaned_data.get(field_name)

    def clean_logo_svg_tr(self):
        field_name = 'logo_svg_tr'
        self.check_file_size(field_name)
        self.check_svg(field_name)

        return self.cleaned_data.get(field_name)

    def clean_logo_svg_com(self):
        field_name = 'logo_svg_com'
        self.check_file_size(field_name)
        self.check_svg(field_name)

        return self.cleaned_data.get(field_name)


class DohopVendorAdmin(RaspExportModelAdmin):
    form = DohopVendorAdminForm

    fieldsets = (
        (_(u'Партнёр'), {'fields': [
            'title',
            'review_percent',
            'query_module_name',
            't_type',
            'dohop_cache_ttl',
            'is_aviacompany',
            'enabled',
        ]}),
        (_(u'Dohop'), {'fields': [
            'dohop_code',
            'dohop_id',
        ]}),
        (_(u'Переводы в русской национальной версии'),
         {'fields': L_field.admin_fields(Partner, ['national_ru_title'])}),
        (_(u'Переводы в украинской национальной версии'),
         {'fields': L_field.admin_fields(Partner, ['national_ua_title'])}),
        (_(u'Переводы в турецкой национальной версии'),
         {'fields': L_field.admin_fields(Partner, ['national_tr_title'])}),
        (_(u'Переводы в com национальной версии'), {'fields': L_field.admin_fields(Partner, ['national_com_title'])}),
        (_(u'Авиабилеты: опрос вендора'), {'fields': [
            ('enabled_in_ticket_ru', 'enabled_in_mobile_ticket_ru'),
            ('enabled_in_ticket_ua', 'enabled_in_mobile_ticket_ua'),
            ('enabled_in_ticket_tr', 'enabled_in_mobile_ticket_tr'),
            ('enabled_in_ticket_com', 'enabled_in_mobile_ticket_com'),
            'foreign_currency',
        ]}),
        (_(u'Расписания: опрос вендора'), {'fields': [
            ('enabled_in_rasp_ru', 'enabled_in_mobile_rasp_ru'),
            ('enabled_in_rasp_ua', 'enabled_in_mobile_rasp_ua'),
            ('enabled_in_rasp_tr', 'enabled_in_mobile_rasp_tr'),
        ]}),
        (_(u'Логотипы'), {'fields': [
            ('logo_svg_ru', 'logo_svg2png_ru'),
            ('logo_svg_ua', 'logo_svg2png_ua'),
            ('logo_svg_tr', 'logo_svg2png_tr'),
            ('logo_svg_com', 'logo_svg2png_com'),
        ]}),
        (_(u'Дополнительная информация'), {'fields': [
            'dohop_residences',
            'dohop_langs',
            'dohop_vendor_info'
        ]}),
        (_(u'Описания'), {'fields': [
            'description_ru_ru',
            'description_ua_uk',
            'description_ua_ru',
            'description_tr_tr',
            'description_tr_en',
        ], 'classes': ('collapse',)}),

    )

    list_filter = (
        DohopResidencesFilter,
        'foreign_currency',
        'enabled_in_ticket_ru',
        'enabled_in_mobile_ticket_ru',
        'enabled_in_ticket_ua',
        'enabled_in_mobile_ticket_ua',
        'enabled_in_ticket_tr',
        'enabled_in_mobile_ticket_tr',
        'enabled_in_ticket_com',
        'enabled_in_mobile_ticket_com',
        'enabled_in_rasp_ru',
        'enabled_in_mobile_rasp_ru',
        'enabled_in_rasp_ua',
        'enabled_in_mobile_rasp_ua',
        'enabled_in_rasp_tr',
        'enabled_in_mobile_rasp_tr',
    )

    ordering = ('dohop_code',)
    readonly_fields = ('dohop_residences', 'dohop_langs', 'dohop_vendor_info', 'dohop_code', 'dohop_id')

    list_display = ('title', 'dohop_code', 'dohop_id', 'enabled')


admin.site.register(DohopVendor, DohopVendorAdmin)


class PartnerAdmin(RaspExportModelAdmin):
    def regionalizepartnerqueryrule_url(self, obj):
        rule_count = RegionalizePartnerQueryRule.objects.filter(partner=obj).count()

        if rule_count:
            count_text = xngettext(
                rule_count,
                u'<n/> правило',
                u'<n/> правила',
                u'<n/> правил',
            )

            return '<a href="/admin/order/regionalizepartnerqueryrule/?partner__id__exact=%s">%s %s</a>' % (
                obj.id, count_text[0], count_text[1]
            )

        else:
            return u'-'

    regionalizepartnerqueryrule_url.allow_tags = True
    regionalizepartnerqueryrule_url.short_description = "Рег. правила"

    form = PartnerAdminForm

    fieldsets = (
        (_(u'Партнёр'), {'fields': [
            'title',
            'site_url',
            't_type',
            'code',
            'billing_order_id',
            'query_module_name',
            'review_percent',
            'is_aviacompany',
            'can_fetch_by_daemon',
            'disabled',
        ]}),
        (_(u'Переводы в русской национальной версии'),
         {'fields': L_field.admin_fields(Partner, ['national_ru_title'])}),
        (_(u'Переводы в украинской национальной версии'),
         {'fields': L_field.admin_fields(Partner, ['national_ua_title'])}),
        (_(u'Переводы в турецкой национальной версии'),
         {'fields': L_field.admin_fields(Partner, ['national_tr_title'])}),
        (_(u'Переводы в com версии'), {'fields': L_field.admin_fields(Partner, ['national_com_title'])}),
        (_(u'Авиабилеты: опрос партнёра'), {'fields': [
            ('enabled_in_ticket_ru', 'enabled_in_mobile_ticket_ru'),
            ('enabled_in_ticket_ua', 'enabled_in_mobile_ticket_ua'),
            ('enabled_in_ticket_tr', 'enabled_in_mobile_ticket_tr'),
            ('enabled_in_ticket_com', 'enabled_in_mobile_ticket_com'),
            'foreign_currency',
        ]}),
        (_(u'Расписания: опрос партнёра'), {'fields': [
            ('enabled_in_rasp_ru', 'enabled_in_mobile_rasp_ru'),
            ('enabled_in_rasp_ua', 'enabled_in_mobile_rasp_ua'),
            ('enabled_in_rasp_tr', 'enabled_in_mobile_rasp_tr'),
        ]}),
        (_(u'Информация о партнёре в разделе "Партнёры"'),
         {'fields': [
             ('logo_svg_ru', 'logo_svg2png_ru'),
             ('logo_svg_ua', 'logo_svg2png_ua'),
             ('logo_svg_tr', 'logo_svg2png_tr'),
             ('logo_svg_com', 'logo_svg2png_com'),
         ]}),
        (_(u'Логотипы'),
         {'fields': [
             'logo',
             'logo_ru',
             'logo_ua',
             'logo_tr',
         ]}),
        (_(u'Биллинг'), {'fields': [
            'click_price',
            'click_price_ru',
            'click_price_ua',
            'click_price_tr',
            'click_price_com',
            'current_balance',
            'notify_balance_threshold',
            'low_balance_notification_sent',
            'null_balance_notification_sent',
            'statistic_reminder_sent',
            'balance_updated_at',
            'billing_datasource_id_dev',
            'billing_datasource_id_testing',
            'billing_datasource_id_production',
        ]}),
    )
    inlines = [PartnerEmailAdminInline]
    list_filter = (
        'foreign_currency',
        'enabled_in_ticket_ru',
        'enabled_in_mobile_ticket_ru',
        'enabled_in_ticket_ua',
        'enabled_in_mobile_ticket_ua',
        'enabled_in_ticket_tr',
        'enabled_in_mobile_ticket_tr',
        'enabled_in_rasp_ru',
        'enabled_in_mobile_rasp_ru',
        'enabled_in_rasp_ua',
        'enabled_in_mobile_rasp_ua',
        'enabled_in_rasp_tr',
        'enabled_in_mobile_rasp_tr',
    )
    list_display = (
        'title', 'code', 'disabled', 'current_balance', 't_type', 'regionalizepartnerqueryrule_url'
    )
    readonly_fields = (
        'low_balance_notification_sent', 'null_balance_notification_sent', 'statistic_reminder_sent'
    )
    formfield_overrides = {
        PartnerLogoImageField: {'widget': PartnerAdminImageWidget},
        PartnerLogoSvgImageField: {'widget': PartnerAdminImageWidget},
        Svg2PngImageField: {'widget': PartnerAdminImageWidget},
    }

    def get_urls(self):
        def wrap(view):
            def wrapper(*args, **kwargs):
                return self.admin_site.admin_view(view)(*args, **kwargs)
            return update_wrapper(wrapper, view)

        urlpatterns = super(PartnerAdmin, self).get_urls()

        info = self.model._meta.app_label, self.model._meta.model_name

        urlpatterns = [
            url(r'^(.+)/track/$', wrap(self.track_view), name='%s_%s_track' % info),
            url(r'^(.+)/track/result/$', wrap(self.track_result_view), name='%s_%s_track_result' % info),
        ] + urlpatterns

        return urlpatterns

    @jsonp_response_m
    def track_result_view(self, request, key):
        """ Show import results """

        r = requests.get(
            settings.TICKETS_QUERY_URL + 'track/result/',
            params={'trid': request.GET.get('trid')},
            timeout=60,
            verify=False
        )

        if not r.ok:
            raise Exception('%r %r %r' % (r.status_code, r.url, r.content))

        return r.json()['data']

    def track_start(self, data, partner):
        query_params = dict([
            ('point_from', data['point_from'].point_key),
            ('point_to', data['point_to'].point_key),
            ('date_forward', data['when'].strftime('%Y-%m-%d')),
            ('date_backward', data['date_backward'] and
             data['date_backward'].strftime('%Y-%m-%d')),
            ('klass', data['klass']),
            ('passengers', {
                k: data[k]
                for k in ('adults', 'children', 'infants')
            }),
            ('national_version', data['national_version']),
        ])

        r = requests.post(
            settings.TICKETS_QUERY_URL + 'track/start/',
            headers={'Content-type': 'application/json'},
            data=json.dumps({
                'query_params': query_params,
                'p_code': partner.code,
            }),
            timeout=60,
            verify=False
        )

        if not r.ok:
            raise Exception('%r %r %r' % (r.status_code, r.url, r.content))

        return r.json()['data']['trid']

    @csrf_protect_m
    def track_view(self, request, partner_id):
        """ Query partner interface """

        model = self.model
        opts = model._meta
        app_label = opts.app_label

        partner = self.get_object(request, unquote(partner_id))

        context = {}

        if request.method == 'POST':
            form = TrackDaemonImportForm(request.POST)

            if form.is_valid():
                params = {
                    form[field_name].html_name:
                    field.prepare_value(form.cleaned_data[field_name])
                    for field_name, field in form.fields.items()
                }

                try:
                    params['trid'] = self.track_start(form.cleaned_data, partner)

                except Exception:
                    context.setdefault('errors', []).append('track_start error: %s' % format_exc())

                else:
                    return HttpResponseRedirect('?' + urlencode(
                        filter(lambda (k, _v): _v is not None, params.items())
                    ))

        else:
            form = TrackDaemonImportForm(initial=request.GET)
            trid = request.GET.get('trid')

            if trid:
                context.update({'trid': trid})

        context.update({
            'title': _(u'Проверить ответ партнера по направлению'),
            'form': form,
            'media': self.media + form.media,
            'object_id': partner_id,
            'original': partner,
            'is_popup': "_popup" in request.REQUEST,
            'app_label': app_label,
            'has_change_permission': self.has_change_permission(request, partner),
            'opts': opts,
            'content_type_id': ContentType.objects.get_for_model(self.model).id,
        })

        return TemplateResponse(
            request,
            "admin/%s/%s/track.html" % (app_label, opts.object_name.lower()),
            context,
            current_app=self.admin_site.name
        )

    def save_model(self, request, obj, form, change):
        origin = Partner.objects.get(id=obj.id) if change else None
        super(PartnerAdmin, self).save_model(request, obj, form, change)

        try:
            self._update_billing_datasource_id(request, obj, origin)
        except Exception as e:
            log.exception(u'Error update billing_datasource_id')
            self.message_user(request, 'Не удалось обновить datasource_id: %r' % e, messages.WARNING)

    def _update_billing_datasource_id(self, request, partner, origin):
        if not partner.billing_order_id:
            return

        if origin:
            if partner.billing_datasource_id != origin.billing_datasource_id:
                return

            if (
                partner.billing_order_id == origin.billing_order_id and
                partner.billing_datasource_id
            ):
                return

        env_type = settings.ENVIRONMENT

        prev_billing_datasource_id = partner.billing_datasource_id

        with closing(StringIO()) as buf:
            handler = logging.StreamHandler(buf)
            log.addHandler(handler)
            try:
                billing_datasource_id = _get_billing_datasource_id_by_order_id(
                    partner.billing_order_id, env_type, log
                )
            finally:
                log.removeHandler(handler)
                billing_query_log_msg = buf.getvalue()

        self.message_user(request, billing_query_log_msg)

        set_partner_billing_datasource_id(partner, env_type, billing_datasource_id)

        msg = 'Updated {} datasource_id: {} -> {} billing_order_id: {} by {}'.format(
            partner.code,
            prev_billing_datasource_id,
            partner.billing_datasource_id,
            partner.billing_order_id,
            request.user.username,
        )
        self.message_user(request, msg)
        log.info(msg)

        partner.save()

        send_partner_email(partner, message='%s\n\n%s' % (msg, billing_query_log_msg))


def send_partner_email(partner, message):
    if settings.ENVIRONMENT == 'production':
        avia_mangers_mails = ['avia@yandex-team.ru']
    else:
        avia_mangers_mails = ['avia-test@yandex-team.ru']

    send_mail(
        subject='Datasource id changed for %s' % partner.code,
        message=message,
        from_email=u"""Яндекс.Расписания <rasp-info@yandex-team.ru>""",
        recipient_list=avia_mangers_mails
    )


def partner_billing_datasource_id_field(env_type):
    assert env_type in ['dev', 'testing', 'production']
    return 'billing_datasource_id_%s' % env_type


def set_partner_billing_datasource_id(partner, env_type, value):
    fieldname = partner_billing_datasource_id_field(env_type)
    return setattr(partner, fieldname, value)


class BillingError(Exception):
    pass


def _get_billing_datasource_id_by_order_id(billing_order_id, env_type, logger):
    billing_url = settings.BILLING_API_URLS.get(env_type, settings.BILLING_API_URLS['testing'])
    r = requests.get(
        billing_url + '/getCampaign',
        params={
            'campaignId': billing_order_id,
            'serviceId': 114,
        },
        timeout=BILLING_API_TIMEOUT
    )
    logger.info('Environment: %s', env_type)
    logger.info('Query: %s', r.request.url)
    if r.ok:
        logger.info('Answer: %s', r.content)
    else:
        logger.info('Error: %s %s', r.status_code, r.content)
        r.raise_for_status()

    xml = r.content
    tree = safe_xml_fromstring(xml)
    errors = tree.xpath('./errors/*')
    if errors:
        logger.error('Billing error: %s', xml)
        raise BillingError('%s %s' % (r.request.url, xml))

    try:
        datasource_id_el = tree.xpath('./campaign-info/datasource-id')[0]
    except IndexError:
        raise BillingError('No datasource-id in response: %s %s' % (r.request.url, xml))

    try:
        datasource_id = int(datasource_id_el.text)
    except ValueError:
        raise BillingError('Wrong datasourcce-id value: %s %s' % (r.request.url, xml))

    logger.info('Successfully got datasource_id: %s', datasource_id)

    return datasource_id


class ForeignKeyRawModelIdWidget(ForeignKeyRawIdWidget):
    def __init__(self, model, lookup_params=None, *args, **kwargs):
        self.lookup_params = lookup_params

        super(ForeignKeyRawModelIdWidget, self).__init__(
            ManyToOneRel(None, model, 'id'), admin.site, *args, **kwargs
        )

    def url_parameters(self):
        params = super(ForeignKeyRawModelIdWidget, self).url_parameters()

        if self.lookup_params:
            params.update(self.lookup_params)

        return params


class ModelRawIdChoiceField(forms.ModelChoiceField):
    def __init__(self, model, *args, **kwargs):
        lookup_params = kwargs.pop('lookup_params', None)

        super(ModelRawIdChoiceField, self).__init__(
            model.objects,
            widget=ForeignKeyRawModelIdWidget(model, lookup_params),
            *args, **kwargs
        )


KLASS_CHOICES = [
    ('economy', 'Эконом'),
    ('business', 'Бизнес'),
]


NATIONAL_VERSION_CHOICES = [
    (v, v)
    for v in settings.DOMAIN_TO_NATIONAL_VERSION.values()
]


class TrackDaemonImportForm(forms.Form):
    from_code = forms.CharField(required=False)
    from_station = ModelRawIdChoiceField(Station, required=False, lookup_params={'t_type__code': 'plane'})
    from_settlement = ModelRawIdChoiceField(Settlement, required=False)

    to_code = forms.CharField(required=False)
    to_station = ModelRawIdChoiceField(
        Station, required=False, lookup_params={'t_type__code': 'plane'})
    to_settlement = ModelRawIdChoiceField(Settlement, required=False)

    when = forms.DateField(required=True, widget=AdminDateWidget)
    date_backward = forms.DateField(required=False, widget=AdminDateWidget)

    adults = forms.IntegerField(required=False, initial=1)
    children = forms.IntegerField(required=False, initial=0)
    infants = forms.IntegerField(required=False, initial=0)

    klass = forms.ChoiceField(
        required=False, choices=KLASS_CHOICES, initial=KLASS_CHOICES[0])

    national_version = forms.ChoiceField(
        required=False, choices=NATIONAL_VERSION_CHOICES,
        initial=NATIONAL_VERSION_CHOICES[0])

    def clean_from_code(self):
        self._point_from_by_code = _point_by_code(
            self.cleaned_data['from_code'])

        if self._point_from_by_code:
            return self.cleaned_data['from_code']

    def clean_to_code(self):
        self._point_to_by_code = _point_by_code(self.cleaned_data['to_code'])

        if self._point_to_by_code:
            return self.cleaned_data['to_code']

    def clean_adults(self):
        val = self.cleaned_data['adults']
        return val if val is not None else self.fields['adults'].initial

    def clean_children(self):
        val = self.cleaned_data['children']
        return val if val is not None else self.fields['children'].initial

    def clean_infants(self):
        val = self.cleaned_data['infants']
        return val if val is not None else self.fields['infants'].initial

    def clean(self):
        data = self.cleaned_data

        point_from = self._point_from_by_code

        if point_from:
            data['from_station'] = None
            data['from_settlement'] = None

        elif data.get('from_station'):
            data['from_settlement'] = None

        point_from = point_from or \
            data.get('from_station') or data.get('from_settlement')

        if not point_from:
            raise forms.ValidationError(_(u'Нужно указать пункт отправления'))

        data['point_from'] = point_from

        point_to = self._point_to_by_code

        if point_to:
            data['to_station'] = None
            data['to_settlement'] = None

        elif data.get('to_station'):
            data['to_settlement'] = None

        point_to = point_to or \
            data.get('to_station') or data.get('to_settlement')

        if not point_to:
            raise forms.ValidationError(_(u'Нужно указать пункт прибытия'))

        data['point_to'] = point_to

        if data['point_from'] == data['point_to']:
            raise forms.ValidationError(
                _(u'Пукнты отправления и прибытия должны различаться'))

        return self.cleaned_data


admin.site.register(Partner, PartnerAdmin)

admin.site.register(CoachService)


class CustomPartnersFilter(SimpleListFilter):
    title = _(u'Партнеры')
    parameter_name = 'partner__id__exact'

    def lookups(self, request, model_admin):
        partner_ids = RegionalizePartnerQueryRule.objects.values_list("partner", flat=True).distinct()
        partners = Partner.objects.in_bulk(partner_ids).items()

        return partners

    def queryset(self, request, queryset):
        if self.value():
            return queryset.filter(partner=self.value())

        else:
            return queryset


class RegionalizePartnerQueryForm(forms.ModelForm):
    def clean_iatas(self):
        iatas = filter(
            None,
            [iata.strip().upper() for iata in re.split('[\r\n;]+', self.cleaned_data['iatas'])]
        )

        unknown_iatas = set()
        settlements = set()

        if iatas:
            for iata in iatas:
                settlement = None

                try:
                    settlement = Settlement.objects.get(iata=iata)

                except Settlement.DoesNotExist:
                    pass

                if not settlement:
                    try:
                        station = Station.get_by_code('iata', iata)
                        settlement = station.settlement

                    except Station.DoesNotExist:
                        unknown_iatas.add(iata)
                        continue

                settlements.add(settlement)

            if unknown_iatas:
                raise forms.ValidationError(
                    gettext(u'Неизвестные коды ИАТА') + u': ' + u'; '.join(unknown_iatas)
                )

            else:
                self.multiple_settlements_to = settlements

        return ';'.join(iatas)

    iatas = forms.CharField(
        label=u'IATA\'s',
        widget=forms.Textarea(
            attrs={
                'cols': 120,
                'rows': 5
            }
        ),
        required=False,
        help_text=_(u'Если заполнен список IATA, то блок "To" полностью игнорируется. Пример: "SVX;SVO;DME".')
    )

    class Meta:
        model = RegionalizePartnerQueryRule
        exclude = []


class RegionalizePartnerQueryRuleAdmin(RaspExportModelAdmin):
    form = RegionalizePartnerQueryForm

    list_display = ('__unicode__', 'display_from', 'display_to',
                    'display_user_point', 'exclude')

    search_fields = ['settlement_from__title']

    raw_id_fields = ("settlement_from", "country_from", "pseudo_region_from",
                     "settlement_to", "country_to", "pseudo_region_to",
                     "user_settlement", "user_country", "user_pseudo_region",)

    fieldset_header = ('', {'fields': ("partner", "exclude")})
    fieldset_from = ('From', {'fields': ("settlement_from", "country_from", "pseudo_region_from")})
    fieldset_to = ('To', {'fields': ("settlement_to", "country_to", "pseudo_region_to")})
    fieldset_multiple_to = ('Multiple To', {'fields': ("iatas",)})
    fieldset_user = ('User', {'fields': ("user_settlement", "user_country", "user_pseudo_region")})

    list_filter = [CustomPartnersFilter]

    def get_fieldsets(self, request, obj=None):
        if obj:
            fieldsets = (
                self.fieldset_header,
                self.fieldset_from,
                self.fieldset_to,
                self.fieldset_user
            )
        else:
            fieldsets = (
                self.fieldset_header,
                self.fieldset_from,
                self.fieldset_to,
                self.fieldset_multiple_to,
                self.fieldset_user
            )

        return fieldsets

    def response_add(self, request, obj):
        if not obj._get_pk_val():
            request.POST.pop('_continue', None)

        return super(RegionalizePartnerQueryRuleAdmin, self).response_add(request, obj)

    @transaction.atomic
    def save_model(self, request, obj, form, change, *args, **kwargs):
        multiple_settlements_to = getattr(form, 'multiple_settlements_to', None)

        if multiple_settlements_to:
            for settlement in multiple_settlements_to:
                RegionalizePartnerQueryRule.objects.get_or_create(
                    partner=obj.partner,
                    settlement_from=obj.settlement_from,
                    country_from=obj.country_from,
                    settlement_to=settlement,
                    defaults={
                        'pseudo_region_from': obj.pseudo_region_from,
                        'user_settlement': obj.user_settlement,
                        'user_country': obj.user_country,
                        'user_pseudo_region': obj.user_pseudo_region,
                        'exclude': obj.exclude,
                    }
                )

        else:
            return super(RegionalizePartnerQueryRuleAdmin, self).save_model(
                request, obj, form, change, *args, **kwargs
            )

        return True

    def display_from(self, rpq):
        point = rpq.get_point_from()
        if point:
            return point.title
        else:
            return _(u"Все")

    display_from.short_description = _(u"Откуда")

    def display_to(self, rpq):
        point = rpq.get_point_to()
        if point:
            return point.title
        else:
            return _(u"Все")

    display_to.short_description = _(u"Куда")

    def display_user_point(self, rpq):
        point = rpq.get_user_point()
        if point:
            return point.title
        else:
            return _(u"Все")

    display_user_point.short_description = _(u"Положение пользователя")


admin.site.register(RegionalizePartnerQueryRule, RegionalizePartnerQueryRuleAdmin)


class QueryBlackListForm(forms.ModelForm):
    active_to = forms.DateField(
        initial=datetime.today() + timedelta(days=3),
        widget=AdminDateWidget
    )

    def clean(self):
        data = self.cleaned_data

        price_from = data.get("price_from", None)
        price_to = data.get("price_to", None)
        currency = data.get("currency", None)

        settlement_from = data.get("settlement_from", None)
        station_from = data.get("station_from", None)
        country_from = data.get("country_from", None)

        settlement_to = data.get("settlement_to", None)
        station_to = data.get("station_to", None)
        country_to = data.get("country_to", None)

        if (price_from or price_to) and not currency:
            raise forms.ValidationError(_(u"Пожалуйста, выберите валюту."))

        if currency and not (price_from or price_to):
            raise forms.ValidationError(_(u"Пожалуйста, укажите цену от и/или до."))

        field_from = settlement_from or station_from or country_from
        field_to = settlement_to or station_to or country_to

        if field_from and not field_to:
            raise forms.ValidationError(_(u'Заполните страну, город или аэропорт прибытия.'))

        if field_to and not field_from:
            raise forms.ValidationError(_(u'Заполните страну, город или аэропорт отправления.'))

        return data

    class Meta:
        model = QueryBlackList
        exclude = []


class QueryBlackListAdmin(RaspExportModelAdmin):
    form = QueryBlackListForm

    def formfield_for_dbfield(self, db_field, request, **kwargs):
        formfield = super(QueryBlackListAdmin, self).formfield_for_dbfield(db_field, request, **kwargs)

        if db_field.name == 'description':
            formfield.widget = forms.Textarea(attrs=formfield.widget.attrs)

        return formfield

    def display_from(self, qbl):
        froms = [f.L_popular_title() for f in filter(None, [qbl.country_from, qbl.settlement_from, qbl.station_from])]

        return ', '.join(froms)

    display_from.short_description = _(u"Откуда")

    def display_to(self, qbl):
        froms = [f.L_title() for f in filter(None, [qbl.country_to, qbl.settlement_to, qbl.station_to])]

        return ', '.join(froms)

    display_to.short_description = _(u"Куда")

    def display_full_flight_number(self, qbl):
        company_iata = None

        if getattr(qbl, "company", None):
            company_iata = qbl.company.iata

        return '-'.join(filter(None, [company_iata, qbl.flight_number]))

    display_full_flight_number.short_description = _(u"АК/Рейс")

    def display_partner(self, qbl):
        return qbl.partner.L_title() if qbl.partner else None

    display_partner.short_description = _(u"Партнер")

    list_display = (
        "__unicode__",
        "display_partner",
        "display_from",
        "display_to",
        "display_full_flight_number",
        "when_from", "when_to",
        "price_from", "price_to", "currency",
        "active_to", "active"
    )

    raw_id_fields = (
        "settlement_from", "station_from", "country_from",
        "settlement_to", "station_to", "country_to",
        "company"
    )

admin.site.register(QueryBlackList, QueryBlackListAdmin)


def _point_by_code(code):
    try:
        return Point.get_by_key(code)
    except (ValueError, ObjectDoesNotExist):
        pass

    code = code.lower()

    try:
        return Settlement.objects.get(iata__iexact=code)
    except Settlement.DoesNotExist:
        pass

    try:
        return Settlement.objects.get(sirena_id__iexact=code)
    except Settlement.DoesNotExist:
        pass

    try:
        return Station.get_by_code('iata', code)
    except Station.DoesNotExist:
        pass

    try:
        return Station.get_by_code('sirena', code)
    except Station.DoesNotExist:
        pass


class BlablacarTokenAdmin(RaspExportModelAdmin):
    list_display = ('token_title', 'expire_msk_dt', 'receive_msk_dt')
    readonly_fields = ('token', 'expire_msk_dt', 'receive_msk_dt')

    ordering = ('-expire_msk_dt', )

    actions = None

    def has_add_permission(self, request):
        return False

    def has_delete_permission(self, request, obj=None):
        return False

    def token_title(self, blablacar_token):
        return blablacar_token.token[:8]


admin.site.register(BlablacarToken, BlablacarTokenAdmin)
