# coding: utf-8

from __future__ import unicode_literals

import calendar
import contextlib

import re
from datetime import date
from itertools import chain, izip, repeat

from django import forms
from django.conf import settings
from django.core.files.base import ContentFile
from django.db import models
from django.db.models import fields
from django.db.models.signals import post_save
from django.dispatch import receiver
from django.template.loader import render_to_string
from django.utils import translation
from django.utils.safestring import mark_safe
from django.utils.translation import ugettext_lazy as _

from common.apps.train.models import PlacePriceRules
from common.db.mds.s3_storage import mds_s3_media_storage
from common.models.partner import DohopVendor, Partner
from common.models_utils.i18n import L_field, AbstractTranslate
from common.utils.fields import CodeCharField, TrimmedCharField, RegExpField, RegExpTextField
from travel.rasp.admin.lib.image import get_pilimage_content, svg2image


@receiver(post_save, sender=DohopVendor)
@receiver(post_save, sender=Partner)
def partner_post_save(instance, **kwargs):
    p = instance

    for pngfield, svgfield in [
        (p.logo_svg2png_ru, p.logo_svg_ru),
        (p.logo_svg2png_ua, p.logo_svg_ua),
        (p.logo_svg2png_tr, p.logo_svg_tr),
        (p.logo_svg2png_com, p.logo_svg_com),
    ]:
        if svgfield and not pngfield:
            try:
                svgfield.open()

                with contextlib.closing(svgfield) as f:
                    svg = f.read()

                img = svg2image(svg, size=(420, 132))
                img_content = get_pilimage_content(img)
                pngfield.save('unused_name', ContentFile(img_content))

            except Exception:
                pass


class StatisticsEntry(models.Model):
    partner = models.ForeignKey(Partner, verbose_name=_('партнер'))
    year = models.IntegerField(verbose_name=_('год'))
    month = models.IntegerField(verbose_name=_('месяц'), choices=((m, m) for m in range(1, 13)))
    day = models.IntegerField(verbose_name=_('день'), choices=((m, m) for m in range(1, 32)))
    price = models.FloatField(verbose_name=_('сумма заказа'))

    class Meta:
        ordering = ['year', 'month', 'day']

    @classmethod
    def for_month(cls, partner, year, month, **kwargs):
        now = date.today()
        is_current = now.year == year and now.month == month
        days_in_month = calendar.monthrange(year, month)[1] + 1 if not is_current else now.day

        if year > now.year or (year == now.year and month > now.month):
            return cls.objects.filter(partner=partner, year=year, month=month, **kwargs)

        for d in range(1, days_in_month):
            entries = cls.objects.filter(partner=partner, year=year, month=month, day=d, **kwargs)

            if not entries.count():
                cls.retrieve_for_day(partner, year, month, d, **kwargs)

        return cls.objects.filter(partner=partner, year=year, month=month, **kwargs)

    @classmethod
    def retrieve_for_month(cls, partner, year, month, **kwargs):
        """Собрать статистику за месяц"""
        days_in_month = calendar.monthrange(year, month)[1]

        for day in range(1, days_in_month + 1):
            cls.retrieve_for_day(partner, year, month, day, **kwargs)

    @classmethod
    def retrieve_for_day(cls, partner, year, month, day, **kwargs):
        """Собрать статистику за день"""

        import stats.sindbad as sindbad
        retrievers = {
            'sindbad': sindbad
        }

        code = partner.code if isinstance(partner, Partner) else partner
        partner_object = partner if isinstance(partner, Partner) else Partner.objects.get(code=code)

        if code not in retrievers:
            raise NotImplemented('Для партнера %s не реализован сбор статистики' % code)

        entries = retrievers[code].retrieve(year=year, month=month, day=day, **kwargs)

        # clean old
        cls.objects.filter(partner=partner, year=year, month=month, day=day).delete()

        for entry in entries:
            entry.partner = partner_object
            entry.save()


class CoachSchemaWidget(forms.Widget):
    def _media(self):
        return forms.Media(
            css={
                'all': (settings.STATIC_URL + 'markup/pages-desktop/coach-schema/_coach-schema.css',)
            },
            js=(
                settings.STATIC_URL +
                'markup/pages-desktop/coach-schema/_coach-schema.%s.pub.js' %
                translation.get_language(),
            )
        )

    media = property(_media)

    def render(self, name, value, attrs):
        schema = value if isinstance(value, CoachSchema) else CoachSchema()
        context = {
            'name': name,
            'value': unicode(schema),
            'image': schema.image,
            'width': schema.width or CoachSchema.WIDTH,
            'height': schema.height or CoachSchema.HEIGHT,
            'thumb': schema.thumb,
        }

        return mark_safe(render_to_string('admin/coach_schema_widget.html', context))


def grouper(n, iterable, fillvalue=None):
    """ grouper(3, 'ABCDEFG', 'x') --> ABC DEF Gxx"""
    return izip(*[chain(iterable, repeat(fillvalue, n - 1))] * n)


class CoachSchema(object):
    def __init__(self, seats=None, source='', image=None, width=None, height=None, thumb=True):
        self.source = source
        self.seats = seats or source and self.deserialize(source) or []

        self.width = width
        self.height = height
        self.image = image

        self.thumb = thumb

    WIDTH = 728
    HEIGHT = 195
    STRIP_RE = re.compile(r'\D')

    class SEAT:
        WIDTH = 23
        HEIGHT = 21
        PADDING = 1

    class ROW:
        PADDING = 3

    CAPACITIES = {
        'soft': 12,
        'suite': 20,
        'compartment': 40,
        'platzkarte': 56,
    }

    @classmethod
    def auto(cls, coach):
        """Автоматическа генерация схемы"""

        if not coach.seats:
            return None

        numbers = coach.seats['free']

        def parse_int(s):
            try:
                return int(cls.STRIP_RE.sub('', s))
            except ValueError:
                return None

        max_number = max(parse_int(number) for number in numbers) if numbers else 0

        guessed = cls.CAPACITIES.get(coach.klass.code, 0)

        required = max(max_number, guessed)

        required += required % 2  # Число мест должно быть четным

        groups = {}

        for n in range(1, required + 1):
            groups.setdefault((n - 1) / 2, set()).add(unicode(n))

        for n in numbers:
            i = (parse_int(n) - 1) / 2

            groups[i].add(n)

        groups = [
            sorted(items, key=lambda n: (parse_int(n), n))
            for g, items in sorted(groups.items(), key=lambda g: g[0])
        ]

        n_columns = (cls.WIDTH + cls.SEAT.PADDING) / (cls.SEAT.WIDTH + cls.SEAT.PADDING)
        n_rows = (len(groups) + n_columns - 1) / n_columns

        seats = []

        height = 0

        for row in range(n_rows):
            max_group = 0

            row_groups = groups[n_columns * row:(row + 1) * n_columns]

            max_group = max(len(g) for g in row_groups)

            height += max_group * (cls.SEAT.HEIGHT + cls.SEAT.PADDING) - cls.SEAT.PADDING

            for column, group in enumerate(row_groups):
                left = (cls.SEAT.WIDTH + cls.SEAT.PADDING) * column

                for i, number in enumerate(group):
                    top = height - cls.SEAT.HEIGHT - (cls.SEAT.HEIGHT + cls.SEAT.PADDING) * i

                    seats.append({
                        'number': number,
                        'coords': {
                            'x': left,
                            'y': top,
                        }
                    })

        return cls(seats, height=height, thumb=False)

    def check(self, free_seats):
        """Проверяет, есть-ли на схеме нужные места"""

        seats = set(seat['number'] for seat in self.seats)

        for seat in free_seats:
            if seat not in seats:
                return False

        return True

    LINE_RE = re.compile(r'^( *)(.*)')
    SPLIT_RE = re.compile(r'( *, *)')

    def parseInt(self, value):
        try:
            return int(value)
        except ValueError:
            try:
                return int(float(value))
            except ValueError:
                return None

    def deserialize(self, source):
        seats = []

        for line in source.split('\n'):
            match = self.LINE_RE.match(line)

            values = self.SPLIT_RE.split(match.group(2))

            if len(values) < 9:
                values.extend([''] * (9 - len(values)))

            seat = {
                'number': values[0],
                'coords': {
                    'x': self.parseInt(values[2]),
                    'y': self.parseInt(values[4]),
                },
                'thumb': {
                    'x': self.parseInt(values[6]),
                    'y': self.parseInt(values[8])
                }
            }

            if seat['coords']['x'] is not None or seat['coords']['y'] is not None:
                seats.append(seat)

        return seats

    def __unicode__(self):
        return self.source or u""


class CoachSchemaFieldDescriptor(object):
    def __init__(self, field):
        self.field = field

    def __get__(self, instance=None, owner=None):
        if instance is None:
            raise AttributeError(
                "The '%s' attribute can only be accessed from %s instances."
                % (self.field.name, owner.__name__))

        schema_source = instance.__dict__[self.field.name]

        if isinstance(schema_source, CoachSchema):
            return schema_source

        image = getattr(instance, self.field.image_field)

        without_image_schema = instance.__dict__[self.field.name] = CoachSchema(
            source=schema_source,
        )

        schema = None

        if image:
            try:
                schema = instance.__dict__[self.field.name] = CoachSchema(
                    source=schema_source, image=image.url, width=image.width, height=image.height,
                )
            except IOError:
                pass

        if schema is None:
            return without_image_schema
        else:
            return schema

    def __set__(self, instance, value):
        instance.__dict__[self.field.name] = value


class CoachSchemaField(models.Field):
    def __init__(self, *args, **kwargs):
        self.image_field = kwargs.pop('image_field', None)

        self.image = None

        super(CoachSchemaField, self).__init__(*args, **kwargs)

    def contribute_to_class(self, cls, name):
        super(CoachSchemaField, self).contribute_to_class(cls, name)
        setattr(cls, self.name, CoachSchemaFieldDescriptor(self))

    def get_prep_value(self, value):
        if value is None:
            return None

        return unicode(value)

    def formfield(self, **kwargs):
        defaults = {'widget': CoachSchemaWidget}
        defaults.update(kwargs)
        return super(CoachSchemaField, self).formfield(**defaults)

    def deconstruct(self):
        name, path, args, kwargs = super(CoachSchemaField, self).deconstruct()
        if self.image_field is not None:
            kwargs['image_field'] = self.image_field
        return name, path, args, kwargs

    def get_internal_type(self):
        return 'TextField'


class CoachService(models.Model):
    TANKER_L_FIELDS = ['name']

    L_name = L_field(_('Название'), max_length=100, add_local_field=True, default=None)

    def __unicode__(self):
        return self.L_name()

    class Meta:
        verbose_name = _('услуга')
        verbose_name_plural = _('услуги')


class CoachInfo(models.Model):
    SEAT_SELECTION_ALGO_CHOICES = (
        ('old', _('Старая версия')),
        ('two-level', _('Версия от 02.2015 с выбором двух нижних'))
    )

    name = TrimmedCharField(_('Название'), max_length=100, null=False, blank=False, default=None)
    image = models.ImageField(
        _('Изображение'),
        storage=mds_s3_media_storage,
        upload_to='data/coachinfo/image', null=True, blank=True,
        height_field='height',
        width_field='width',
    )
    width = models.IntegerField(_('Ширина'), editable=False, null=True)
    height = models.IntegerField(_('Высота'), editable=False, null=True)
    schema = CoachSchemaField(_('Схема'), db_column='schema', image_field='image')
    seat_selection_algo = models.CharField(_('Алгоритм выбора мест'), max_length=20,
                                           default='old', choices=SEAT_SELECTION_ALGO_CHOICES)
    upper_places = models.TextField(_('Верхние места (через запятую)'), default='', null=False, blank=True)
    side_places = models.TextField(_('Боковые места (через запятую)'), default='', null=False, blank=True)
    near_toilet_places = models.TextField(_('Места у туалета (через запятую)'), default='', null=False, blank=True)
    group_places = models.TextField(
        _('Группы мест: купе, отсек (каждая группа в новой строке, места в группе - через запятую)'),
        default='', null=False, blank=True)
    details = models.TextField(_('Дополнительная информация о вагоне и местах'), default='', null=False, blank=True)
    svg_schema = models.TextField(_('Размеченая схема вагона (содержание svg файла)'),
                                  default='', null=False, blank=True)
    place_price_rules = models.ManyToManyField(PlacePriceRules, blank=True, related_name='schemas',
                                               verbose_name=_('Правила расчета тарифов на места в вагоне'))
    two_storey = models.BooleanField(_('Двухэтажный вагон'), default=False)

    def __unicode__(self):
        return self.name

    class Meta:
        app_label = 'order'
        verbose_name = _('схема вагона')
        verbose_name_plural = _('схемы вагонов')


class ServiceClassNoteTranslate(AbstractTranslate):
    @classmethod
    def get_keys(cls):
        return set(ServiceClass.objects.values_list('note', flat=True))

    class Meta:
        ordering = ('value',)
        verbose_name = _('перевод примечания класс обслуживания')
        verbose_name_plural = _('переводы примечании классов обслуживания')
        app_label = 'order'


class ServiceClass(models.Model):
    owner = CodeCharField(_('Владелец вагона'), max_length=100, null=True)
    code = CodeCharField(_('Код'), max_length=100, null=False)
    is_brand = CodeCharField(_('Фирменный'), choices=[
        ('any', 'не важно'),
        ('yes', 'да'),
        ('no', 'нет'),
    ], max_length=100, null=False, default='any')

    name = CodeCharField(_('Название'), max_length=100, null=False)
    note = CodeCharField(_('Примечание'), max_length=100, null=True)
    L_note = ServiceClassNoteTranslate.get_L_method('note')

    services = models.ManyToManyField(CoachService, verbose_name=_('услуги'), blank=True)

    @classmethod
    def bind(cls, classes):
        owners = set(c.owner for c in classes)
        service_classes = set(c.service_class_code for c in classes)

        bindings = {}

        for c in cls.objects.filter(owner__in=list(owners), code__in=list(service_classes)):
            bindings.setdefault((c.owner, c.code), []).append(c)

        for c in classes:
            c.service_class = None

            for b in bindings.get((c.owner, c.service_class_code), []):
                if b.is_brand == 'any' or b.is_brand == ('yes' if c.is_brand else 'no'):
                    c.service_class = b
                    break

    def __unicode__(self):
        return "%s %s" % (self.owner, self.code)

    class Meta:
        verbose_name = _('класс обслуживания')
        verbose_name_plural = _('классы обслуживания')


CLASS_CHOICES = [
    ('compartment', _('купе')),
    ('suite', _('люкс')),
    ('sitting', _('сидячий')),
    ('platzkarte', _('плацкарт')),
    ('soft', _('мягкий')),
    ('common', _('общий')),
]


class CoachInfoBinding(models.Model):
    """Привязка схемы к вагону"""
    klass = fields.CharField(_('Категория вагона'), choices=CLASS_CHOICES, default='', blank=True, max_length=100)
    service_class = RegExpField(_('Класс обслуживания'), default='', blank=True, max_length=100)
    international_service_class = RegExpField(_('Международный класс обслуживания'), max_length=100, default='',
                                              blank=True)
    coach_subtype_code = TrimmedCharField(_('Подтип вагона'), max_length=100, default='', blank=True,
                                          help_text=_('Можно указать несколько через запятую.'))
    road = RegExpField(_('Дорога принадлежности вагона'), max_length=255, default='', blank=True,
                       help_text=_('или государство принадлежности'))
    train_number = RegExpTextField(_('Номер поезда'), default='', blank=True)
    coach_number = RegExpField(_('Номер вагона'), default='', blank=True, max_length=255)
    priority = fields.IntegerField(_('Приоритет'), null=False, blank=False, default=10)
    info = models.ForeignKey(CoachInfo, verbose_name=_('схема'), null=False, blank=False)
    direction_confirmed = models.BooleanField(_('Направление движения подтверждено'), default=False)

    # TODO: two_storey используется только в старой морде.
    # После закапывания страниц покупки на старой морде, его нужно удалить.
    two_storey = models.BooleanField(_('Двухэтажный вагон'), default=False)
    im_car_scheme_id = RegExpField(_('RailwayCarSchemeId из ИМ'), default='', blank=True, max_length=255)

    @classmethod
    def bind(cls, train):
        # Похоже этот метод не используется, в train-api для этих целей есть BestSchemaFinder
        objects = list(cls.objects.filter(priority__gt=0))

        for train_class in train.classes:
            for coach in train_class.coaches:
                coach.schema = cls.get(coach, objects) or CoachSchema.auto(coach)

    @classmethod
    def get(cls, coach, objects):
        choices = [(i.test(coach), i.priority, i) for i in objects]

        choices.sort(reverse=True)

        for conformance, _prior, i in choices:
            if not conformance:
                return None

            return i.info.schema

        return None

    def test(self, coach):
        # Какая то проверка на соответствие спецификации вагона, копать у Ильи
        # FIXME: Докопаться и написать сюда документацию
        if not coach.two_storey == self.two_storey:
            return 0

        if not coach.seats:
            return 0

        if not self.info.schema.check(coach.seats['free']):
            return 0

        if self.klass != coach.klass.code:
            return 0

        points = 1

        if self.service_class:
            if coach.klass.service_class_code and re.match(self.service_class, coach.klass.service_class_code,
                                                           re.U | re.I):
                points += 1
            else:
                return 0

        if self.train_number:
            if re.match(self.train_number, coach.klass.train.number, re.U | re.I):
                points += 1
            else:
                return 0

        if self.coach_number:
            if re.match(self.coach_number, coach.number, re.U | re.I):
                points += 1
            else:
                return 0

        return points

    def __unicode__(self):
        return "%s %s %s %s" % (self.klass, self.train_number, self.coach_number, self.priority)

    class Meta:
        verbose_name = _('привязка схемы вагона')
        verbose_name_plural = _('привязки схем вагонов')


class QueryBlackList(models.Model):
    CLASS_CHOICES = [
        ('economy', _('Эконом')),
        ('business', _('Бизнес')),
        ('first', _('Первый')),
    ]

    partner = models.ForeignKey(Partner, null=True, blank=True, verbose_name=_('партнер'))

    country_from = models.ForeignKey('www.Country', null=True, blank=True,
                                     related_name='query_black_list_country_from',
                                     verbose_name=_('страна отправления'))

    settlement_from = models.ForeignKey('www.Settlement', null=True, blank=True,
                                        related_name='query_black_list_settlement_from',
                                        verbose_name=_('город отправления'))

    station_from = models.ForeignKey('www.Station', null=True, blank=True,
                                     related_name='query_black_list_station_from',
                                     verbose_name=_('станция отправления'))

    country_to = models.ForeignKey('www.Country', null=True, blank=True,
                                   related_name='query_black_list_country_to',
                                   verbose_name=_('страна прибытия'))

    settlement_to = models.ForeignKey('www.Settlement', null=True, blank=True,
                                      related_name='query_black_list_settlement_to',
                                      verbose_name=_('город прибытия'))

    station_to = models.ForeignKey('www.Station', null=True, blank=True,
                                   related_name='query_black_list_station_to',
                                   verbose_name=_('станция прибытия'))

    company = models.ForeignKey('www.Company', null=True, blank=True,
                                related_name='query_black_list_company',
                                verbose_name=_('Перевозчик'))

    flight_number = fields.CharField(_('Номер рейса'), null=True, blank=True, max_length=10, db_index=True)

    when_from = fields.DateField(_('Дата рейса от'), null=True, blank=True, db_index=True)

    when_to = fields.DateField(_('Дата рейса до'), null=True, blank=True, db_index=True)

    price_from = models.FloatField(_('Цена от'), blank=True, null=True)

    price_to = models.FloatField(_('Цена до'), blank=True, null=True)

    currency = models.ForeignKey('currency.Currency', null=True, blank=True,
                                 related_name='query_black_list_price_currency',
                                 verbose_name=_('Валюта'))

    klass = models.CharField(_('Класс'), choices=CLASS_CHOICES, null=True, blank=True, max_length=10)

    description = fields.CharField(_('Причина бана'), null=False, blank=False, max_length=255)

    active_to = fields.DateField(_('Активно до'), null=False, blank=False, db_index=True)

    active = fields.BooleanField(_('Активно'), null=False, blank=False, db_index=True)

    def __unicode__(self):
        return _('Правило № %s') % self.id

    class Meta:
        verbose_name = _('Правило черного списка')
        verbose_name_plural = _('Правила черного списка')


class ActualDirection(models.Model):
    """
    Данная модель используется для фильтрации запросов к партнерам по станциям.
    Позволяет запрашивать партнера только на те направления, где у него есть рейсы.
    Заполняется при импорте.
    """

    partner = models.ForeignKey(Partner, null=False, blank=False, verbose_name=_(u"партнер"))
    station_from = models.ForeignKey(
        'www.Station', null=False, blank=False,
        verbose_name=_('начальная станция'),
        related_name='station_can_ask_from'
    )
    station_to = models.ForeignKey(
        'www.Station', null=False, blank=False,
        verbose_name=_('конечная станция'),
        related_name='station_can_ask_to'
    )
    settlement_from = models.ForeignKey(
        'www.Settlement', null=True, blank=True, default=None,
        verbose_name=_('начальный город'),
        related_name='settlement_can_ask_from'
    )
    settlement_to = models.ForeignKey(
        'www.Settlement', null=True, blank=True, default=None,
        verbose_name=_('конечный город'),
        related_name='settlement_can_ask_to'
    )

    supplier_to_partner = {
        'swdfactory': 'swdfactory',
        'ukrmintrans': 'ukrmintrans_bus',
    }
