#!/usr/bin/env python
# coding: utf-8
"""
RASP-3551: Мониторить число рейсов по поставщикам в большом импорте
https://jira.yandex-team.ru/browse/RASP-3551
"""

import travel.rasp.admin.scripts.load_project  # noqa

import copy
import logging
import os.path
from datetime import timedelta
from email.mime.text import MIMEText
from optparse import OptionParser
from StringIO import StringIO

from django.core.mail.message import EmailMultiAlternatives
from django.db import transaction
from django.conf import settings

from common.models.schedule import Supplier
from travel.rasp.library.python.common23.date import environment
from common.utils.metrics import task_progress_report
from common.utils.unicode_csv import UnicodeDictWriter
from travel.rasp.admin.importinfo.models import ImportReport
from travel.rasp.admin.importinfo.models.two_stage_import import TwoStageImportPackage
from travel.rasp.admin.lib import tmpfiles
from travel.rasp.admin.lib.logs import print_log_to_stdout, create_current_file_run_log, get_script_log_context, ylog_context


log = logging.getLogger(__name__)


def make_html_from_tsv(data, delimiter=u'\t', make_head=False, border=1):
    output = u"<table border=\"%d\">" % border
    for index, line in enumerate(data.splitlines()):
        tr = u"<tr>%s</tr>"
        td = u"<th>%s</th>"
        if index == 0 and make_head:
            td = u"<th>%s</th>"
        output += tr % (u"".join([td % v.strip() for v in line.split(delimiter)]))

    output += u"</table>"
    return output


fieldnames = (
    'percentage_diff',
    'supplier_title',
    'supplier_code',
    'current_route_count',
    'previous_route_count',
    'current_imported_dt',
    'previous_imported_dt',
)


@transaction.atomic
def run(save_report=False):
    subject = u'Изменения в импорте : Import Changes'
    mail = EmailMultiAlternatives(u'%s%s' % (settings.EMAIL_SUBJECT_PREFIX, subject),
                                  u'', settings.SERVER_EMAIL,
                                  [a[1] for a in settings.PROCESS_RCPNTS])

    attach_route_chages(mail, save=save_report)
    attach_tsi_autoimport_state(mail)

    mail.send()


@tmpfiles.clean_temp
def attach_route_chages(mail, save=False):
    today = environment.today()

    diff_dicts = []
    for supplier in Supplier.objects.filter(add_to_route_changes_report=True):
        prev_report = ImportReport.get_last_report(supplier)
        report = ImportReport.create_report(supplier, save=save)

        if not prev_report:
            continue

        diff_dict = ImportReport.build_diff_dict(previous_report=prev_report, current_report=report)
        if diff_dict:
            diff_dicts.append(diff_dict)

        ImportReport.objects.filter(supplier=supplier, imported__lt=today - timedelta(30 * 3)).delete()

    if not diff_dicts:
        mail.body += u'\nНет изменений в данных\n'
        return

    has_big_changes = any(dd['is_ciritical_diff'] for dd in diff_dicts)
    diff_dicts.sort(key=lambda dd: abs(dd['percentage_diff']), reverse=True)

    tmp_dir = tmpfiles.get_tmp_dir('check_route_changes')
    diff_csv_filepath_tab = os.path.join(tmp_dir, 'route_changes_tab.csv')

    with open(diff_csv_filepath_tab, 'wb') as f:
        writer = UnicodeDictWriter(f, fieldnames=fieldnames, extrasaction='ignore', delimiter='\t',
                                   encoding='utf-8')
        writer.writeheader()

        for diff_dict in diff_dicts:
            rowdict = copy.copy(diff_dict)
            rowdict['supplier_title'] = rowdict['supplier'].title
            rowdict['supplier_code'] = rowdict['supplier'].code
            rowdict['percentage_diff'] = u'{}%'.format(rowdict['percentage_diff'])
            rowdict['is_ciritical_diff'] = u'Yes' if rowdict['is_ciritical_diff'] else u'No'

            writer.writerow(rowdict)

    if has_big_changes:
        mail.subject = u'!!!Большие изменения в импорте : Big Import Changes'

    with open(diff_csv_filepath_tab) as f:
        data_tab = f.read().decode('utf-8')

    attachment = MIMEText(data_tab.encode('utf-16'), 'csv', 'utf-16')
    attachment.add_header('Content-Disposition', 'attachment', filename='route_changes_utf-16_tab.csv')
    mail.attach(attachment)


def attach_tsi_autoimport_state(mail):
    stream = StringIO()

    fieldnames = (
        'package_id', 'package_title',
        'last_import_date', 'last_mask_date',
        'supplier_id', 'supplier_code', 'supplier_title',
    )

    writer = UnicodeDictWriter(stream, fieldnames, delimiter='\t', encoding='utf-8')
    writer.writeheader()

    for package in TwoStageImportPackage.objects.filter(autoimport=True):
        writer.writerow({
            'package_id': package.id,
            'package_title': package.title,
            'last_import_date': package.last_import_date,
            'last_mask_date': package.last_mask_date,
            'supplier_id': package.supplier.id,
            'supplier_title': package.supplier.title,
            'supplier_code': package.supplier.code,
        })

    file_content = stream.getvalue().decode('utf-8')

    attachment = MIMEText(file_content.encode('utf-16'), 'csv', 'utf-16')
    attachment.add_header('Content-Disposition', 'attachment',
                          filename='tsi_autoimport_state_utf-16_tab.csv')
    mail.attach(attachment)


usage = u"Usage: python %prog [options]"

# Если файл запустили из консоли парсим параметры и запускаем скрипт
if __name__ == '__main__':
    with ylog_context(**get_script_log_context()), task_progress_report('check_route_changes'):
        optparser = OptionParser(usage=usage, description=__doc__)
        optparser.add_option('-v', '--verbose', action="store_true",
                             help=u"выводить лог на экран")
        optparser.add_option('-s', '--save-report', action="store_true",
                             help=u"Сохранять состояние рейсов в базу")
        options, args = optparser.parse_args()

        if options.verbose:
            print_log_to_stdout()

        create_current_file_run_log()

        run(save_report=options.save_report)
