#!/usr/bin/env python
# coding: utf-8

"""
RASP-6353: подчищать старые календари ниток
https://jira.yandex-team.ru/browse/RASP-6353
"""

import travel.rasp.admin.scripts.load_project  # noqa

import logging
from datetime import date, timedelta

from django.conf import settings
from django.db import connection, transaction

from common.models.schedule import RThread
from common.models_utils import model_iter
from common.utils.date import RunMask
from common.utils.progress import PercentageStatus
from travel.rasp.admin.lib.logs import print_log_to_stdout, create_current_file_run_log, get_script_log_context, ylog_context
from travel.rasp.admin.lib.tmpfiles import get_tmp_filepath, clean_temp


log = logging.getLogger(__name__)


# Сколько дней в прошлом оставлять в календаре
PAST_RANGE_SIZE = 8
TIS_PAST_RANGE_SIZE = 15


@clean_temp
@transaction.atomic
def clean_threads():
    log.info(u'Начинаем очистку календарей')

    today = date.today()

    threads = RThread.objects.exclude(route__supplier__code='tis') \
                     .filter(route__script_protected=False)

    tis_threads = RThread.objects.filter(supplier__code='tis') \
                         .filter(route__script_protected=False)

    start = today - timedelta(settings.DAYS_TO_PAST)
    end = today - timedelta(PAST_RANGE_SIZE)
    diff_mask = RunMask.range(start, end, today=today)

    tis_start = today - timedelta(settings.DAYS_TO_PAST)
    tis_end = today - timedelta(TIS_PAST_RANGE_SIZE)
    tis_diff_mask = RunMask.range(tis_start, tis_end, today=today)

    status = PercentageStatus(threads.count() + tis_threads.count(), log=log)

    tmp_file = get_tmp_filepath("clean_dates.tmp")

    with open(tmp_file, "w") as outfile:
        for t in model_iter(threads, chunksize=1000):
            try:
                mask = t.get_mask(today=today)
                clean_mask = mask.difference(diff_mask)

                print >>outfile, "\t".join([str(t.id), str(clean_mask)])
            except ValueError, e:
                log.error(' '.join(e.args))

            status.step()

        for t in model_iter(tis_threads, chunksize=1000):
            try:
                mask = t.get_mask(today=today)
                clean_mask = mask.difference(tis_diff_mask)

                print >>outfile, "\t".join([str(t.id), str(clean_mask)])
            except ValueError, e:
                log.error(' '.join(e.args))

            status.step()

    cursor = connection.cursor()

    cursor.execute("DROP TABLE IF EXISTS _new_year_days")
    cursor.execute("CREATE TABLE _new_year_days(thread_id integer, tz_year_days text)")
    cursor.execute("LOAD DATA LOCAL INFILE %s INTO TABLE _new_year_days", [tmp_file])
    cursor.execute("ALTER TABLE _new_year_days ADD KEY(thread_id)")
    cursor.execute("""UPDATE www_rthread t
                             INNER JOIN _new_year_days n ON t.id = n.thread_id
                         SET t.tz_year_days = n.tz_year_days""")
    cursor.execute("DROP TABLE _new_year_days")

    log.info(u'Очистка успешно закончена')


if __name__ == '__main__':
    with ylog_context(**get_script_log_context()):
        from optparse import OptionParser

        parser = OptionParser()
        parser.add_option('-v', '--verbose', action='store_true', dest='verbose')

        (options, args) = parser.parse_args()

        if options.verbose:
            print_log_to_stdout(log)

        create_current_file_run_log()

        clean_threads()
