# coding: utf-8
"""Скрипт еженедельного обновления сервисной базы"""

import travel.rasp.admin.scripts.load_project  # noqa

import logging
import os
import sys
from optparse import OptionParser
from cStringIO import StringIO
from unittest import TextTestRunner, TestLoader

from django.conf import settings

from common.data_api.file_wrapper.config import get_wrapper_creator
from common.settings.configuration import Configuration
from common.utils.metrics import task_progress_report
from travel.rasp.admin.lib.exceptions import FormattingException
from travel.rasp.admin.lib.maintenance.flags import flags
from travel.rasp.admin.lib.maintenance.scripts import job
from travel.rasp.admin.lib.logs import (
    print_log_to_stdout, create_current_file_run_log, get_script_log_context, ylog_context, get_current_file_run_log_path,
    copy_std_streams_to_file
)
from travel.rasp.admin.lib.mail import mail_process
from travel.rasp.admin.scripts import check_base_integrity
from travel.rasp.admin.scripts.prepare_all import make_action_list as make_prepare_all_action_list
from travel.rasp.admin.scripts.support_methods import run_action_list, StateSaver
from travel.rasp.admin.scripts.utils.file_wrapper.registry import FileType
from travel.rasp.admin.scripts.utils.file_wrapper.mds_utils import get_cron_log_key
from travel.rasp.admin.scripts.utils.file_wrapper.uploaders import ObjWrapperSyncer
from travel.rasp.admin.scripts.utils.lock import get_script_lock


warning_action = settings.SCRIPT_WARNING_ACTION

log = logging.getLogger(__name__)


before_prepare_all_scripts = (
    {'base_path': settings.SCRIPTS_PATH},
    [
        "== delete_data ==",
        (True, ['python', '-W', warning_action, 'remove_obsolete_af_schedule.py', '-v']),

        (False, ['python', '-W', warning_action, 'clean_old_dates.py', '-v']),
        (True, ['python', '-W', warning_action, 'clean_old_tariffs.py']),

        # RASP-1872, обновлять геобазу
        (False, ['python', '-W', warning_action, 'update_geobase.py']),

        # Уточнения координат городов
        (False, ['python', '-W', warning_action, 'settlement_coords.py'])
    ]
)


def get_import_scripts_list():
    tis_import_command = ['python', '-W', warning_action, 'schedule/tis_train/import_tis.py']

    if getattr(settings, 'APPLIED_CONFIG', None) in [Configuration.PRODUCTION, Configuration.TESTING]:
        tis_import_command.append('download-and-import')
    else:
        tis_import_command.append('import')

    return [(
        {'base_path': settings.SCRIPTS_PATH},
        [
            "== tis ==",
            (True, tis_import_command),

            '== red blue autoimports',
            (True, ['python', '-W', warning_action, 'schedule/bus/two_stage_autoimport.py']),
            (True, ['python', '-W', warning_action, 'schedule/red_autoimport.py']),
        ]
    )]


after_prepare_all_scripts = (
    {'base_path': settings.SCRIPTS_PATH},
    [
        '== 8 ==',
        (False, ['python', '-W', warning_action, 'www_station_rise_majority.py']),
        # нужно вызывать до inflect и preposition
        (False, ['python', '-W', warning_action, 'www_fix_station_types.py']),
        (False, ['python', '-W', warning_action, 'www_fill_preposition_in.py']),
        '== 9 ==',  # Сравниваем с предыдущим импортом
        (False, ['python', '-W', warning_action, 'check_route_changes.py', '--save-report']),
        (False, ['python', '-W', warning_action, 'schedule/remove_old_slices.py']),
        (False, ['python', '-W', warning_action, 'export/gen_t_types_by_geoid_for_mobile_morda.py']),

        (False, ['python', '-W', warning_action, 'geobase_consistency_reports.py']),

        (False, ['python', '-W', warning_action, 'notify_bus_suppliers.py']),

        (False, ['python', '-W', warning_action, 'i18n/collect_platform_translations.py']),

        u'== others ==',

        (False, ['python', '-W', warning_action, 'check_autoincrements.py']),

    ]
)


class DBIsBusyError(FormattingException):
    pass


class CriticalScriptFailedError(FormattingException):
    pass


def big_import(is_continue=False, exit_on_fail=True):
    # Поднимаем флаг идет большой импорт
    if flags['maintenance']:
        log.error(u'Идет работа с базой данных или предыдущий запуск скрипта завершился некореектно')
        if exit_on_fail:
            sys.exit(1)
        else:
            raise DBIsBusyError(u'Идет работа с базой данных или предыдущий запуск скрипта завершился некореектно')

    log.info(u"Выставляем флаг")
    flags['maintenance'] = job.BIG_IMPORT.flag_value

    log.info(u"Снимаем флаг переключения")
    flags['switch'] = False

    action_list = make_action_list()

    status = StateSaver('big_import_state.txt')

    try:
        failed_scripts = run_action_list(action_list, status, log, on_error, is_continue)
    except CriticalScriptFailedError:
        if exit_on_fail:
            sys.exit(1)
        else:
            raise

    message = u'Update finished successfully.'
    log.info(message)

    if failed_scripts:
        message += '\nFailed scripts:\n%s.' % "\n".join(failed_scripts)

    mail_process(u'Weekly database update', message)

    log.info(u"Run tests")
    text_result = StringIO()
    tests = TestLoader().loadTestsFromModule(check_base_integrity)
    runner = TextTestRunner(text_result)
    result = runner.run(tests)
    text_result = text_result.getvalue().decode('utf-8')  # encoding from stderr
    if not result.wasSuccessful():
        mail_process(u'Weekly database update, test results', text_result)

    log.info(u"Снимаем флаг большого импорта")
    flags['maintenance'] = False

    log.info(u"Done")


def on_error(bad_action):
    text = u"%s failed" % bad_action
    mail_process(u'Weekly database update failed!!!', text)
    raise CriticalScriptFailedError(u'Weekly database update failed!!!')


def make_action_list():
    prepare_all_action_list = make_prepare_all_action_list(short=False, full=True)

    return ([before_prepare_all_scripts] + get_import_scripts_list() + prepare_all_action_list +
            [after_prepare_all_scripts])


def run_big_import(options, log_file_path):
    if options.cron_run:
        log_key = get_cron_log_key('cron_schedule_{}'.format(os.path.basename(log_file_path)))
        file_wrapper = get_wrapper_creator(FileType.CRON_RUNS, key=log_key).get_file_wrapper(log_file_path)

        with ObjWrapperSyncer(file_wrapper, interval=settings.MDS_LOG_UPLOAD_INTERVAL):
            try:
                big_import(is_continue=options.is_continue)
            except Exception:
                log.exception('big_import failed')
                raise
    else:
        big_import(is_continue=options.is_continue)


if __name__ == '__main__':
    with get_script_lock(), \
            ylog_context(**get_script_log_context()), task_progress_report('big_import'):
        optparser = OptionParser()

        optparser.add_option('-v', '--verbose', action="store_true",
                             help=u"выводить лог на экран")
        optparser.add_option("--continue", dest="is_continue", action="store_true",
                             help=u"продолжить выпольнения с последнего шага")
        optparser.add_option("--cron-run", dest="cron_run", action="store_true",
                             help=u"запущен по крону")

        options, args = optparser.parse_args()

        if options.verbose:
            print_log_to_stdout()
            log_file_path = get_current_file_run_log_path()
            with copy_std_streams_to_file(log_file_path):
                run_big_import(options, log_file_path)
        else:
            log_file = create_current_file_run_log(capture_stdstreams=True)
            run_big_import(options, log_file.name)
