# coding: utf-8

import travel.rasp.admin.scripts.load_project  # noqa

import logging
import sys
import os.path
import getopt

from django.db import connection, transaction

from common.models.geo import DirectionMarker
from common.models.schedule import RThread
from common.models.transport import TransportType
from travel.rasp.admin.lib.maintenance.flags import flags
from travel.rasp.admin.lib.logs import print_log_to_stdout, create_current_file_run_log, get_script_log_context, ylog_context


log = logging.getLogger(__name__)


script_name = os.path.basename(__file__)

usage = """
Usage: python %(script_name)s [-v]
""" % {
    'script_name': script_name
}


class Helper(object):
    _station_markers = None

    @staticmethod
    def get_station_markers():
        u""" Словарик со всеми маркерами станции"""
        if Helper._station_markers is None:
            station_markers = {}
            markers = DirectionMarker.objects.filter(station__isnull=False).select_related()
            for marker in markers:
                station_markers.setdefault(marker.station, []).append(marker)
            Helper._station_markers = station_markers
        return Helper._station_markers


@transaction.atomic
def clean_binds(threads):
    if not threads:
        return

    threads_in = ','.join([str(t.id) for t in threads])
    cursor = connection.cursor()
    cursor.execute(('UPDATE www_rtstation SET arrival_subdir = NULL, '
                    '  departure_subdir = NULL, '
                    '  arrival_direction_id = NULL, '
                    '  departure_direction_id = NULL WHERE thread_id IN ({})').format(threads_in))


def bind_rtstation(index, rts, rtstations, popular_directions):
    """
    Привязываем станцию нитки к направлению
    https://wiki.yandex-team.ru/AndrejjZharinov/Raspisanija/NapravlenijaJelektrichek
    """
    is_from_subdir = None
    thread_number = rts.thread.route.route_uid
    # 1 Шаг
    station_markers = Helper.get_station_markers()
    station = rts.station

    # 2. если через станцию не проходит направлений, или станция не использует
    # направления, то ничего не делаем
    if not station.use_direction:
        return

    markers = station_markers.get(station, [])

    # Нет ни одного направления выходим
    if len(markers) == 0:
        return

    # 3. если через станцию проходит ровно одно направление, то:
    elif len(markers) == 1:
        our_marker = markers[0]
        our_direction = our_marker.direction
        # 1. в «направление отправления» (и прибытия) кладём ссылку на это направление
        rts.arrival_direction = our_direction
        rts.departure_direction = our_direction
        # Пытаемся получить следующую станцию
        next_is_good = False
        next_station = None
        try:
            next_station = rtstations[index + 1].station
        except IndexError:
            pass
        # Проверяем есть ли она в нашем направлении
        if next_station is not None:
            next_markers = station_markers.get(next_station, [])
            for mr in next_markers:
                if mr.direction == our_direction:
                    next_marker = mr
                    next_is_good = True

        # Следущая станция не лежит в нашем направлении или это конечная станция
        # Пробуем взять предыдущую станцию и определить поднаправление по ней
        prev_is_good = False
        prev_station = None
        if not next_is_good:
            try:
                prev_station = rtstations[index - 1].station
            except IndexError:
                pass
            # Проверяем есть ли она в нашем направлении
            if prev_station is not None:
                prev_markers = station_markers.get(prev_station, [])
                for mr in prev_markers:
                    if mr.direction == our_direction:
                        prev_marker = mr
                        prev_is_good = True

        # Логируем ошибки привязки
        if not next_is_good and next_station:
            log.error(u"%s: Станции %s %s и %s %s лежат на границах направлений,"
                      u" но ни одна из них не принадлежит обоим направлениям",
                      thread_number, station.id, station.title, next_station.id, next_station.title)

        if not prev_is_good and prev_station:
            log.error(u"%s: Станции %s %s и %s %s лежат на границах направлений,"
                      u" но ни одна из них не принадлежит обоим направлениям",
                      thread_number, station.id, station.title, prev_station.id, prev_station.title)

        if not next_is_good and not prev_is_good:
            log.error(u"%s: Не удалось привязать станцию %s %s, нет хороших соседних станций",
                      thread_number, station.id, station.title)
            return

        # 2. Определение поднаправления, и привязка
        # Привязываем по следующей станции
        if next_is_good:
            # Это поднаправление 'туда'
            if next_marker.order > our_marker.order:
                # Если у маркера следующей станции проставлено поднаправление 'туда'
                # берем его, иначе берем поднаправление 'туда' из направления
                departure_subdir = next_marker.title_from or our_direction.title_from
                arrival_subdir = our_marker.arrival_title_from or our_marker.title_from or our_direction.title_from
                is_from_subdir = True
            # Это поднаправление 'обратно'
            elif next_marker.order < our_marker.order:
                # Если у маркера следующей станции проставлено поднаправление 'обратно'
                # берем его, иначе берем поднаправление 'обратно' из направления
                departure_subdir = next_marker.title_to or our_direction.title_to
                arrival_subdir = our_marker.arrival_title_to or our_marker.title_to or our_direction.title_to
                is_from_subdir = False
            else:
                log.error(u"У нитки %s две станции привязанные к одному и томуже маркеру",
                          thread_number)
                return
        # Привязываем по предыдущей станции
        elif prev_is_good:
            # Это поднаправление 'туда'
            if prev_marker.order < our_marker.order:
                # Берем название поднаправления 'туда' из направления
                departure_subdir = our_marker.title_from or our_direction.title_from
                arrival_subdir = our_marker.arrival_title_from or departure_subdir
                is_from_subdir = True
            # Это поднаправление 'обратно'
            elif prev_marker.order > our_marker.order:
                # Берем название поднаправления 'обратно' из направления
                departure_subdir = our_marker.title_to or our_direction.title_to
                arrival_subdir = our_marker.arrival_title_to or departure_subdir
                is_from_subdir = False
            else:
                log.error(u"У нитки %s две станции привязанные к одному и томуже маркеру",
                          thread_number)
                return

        rts.arrival_subdir = arrival_subdir
        rts.departure_subdir = departure_subdir

    # 4.1 Если через станцию проходит более одного направления,
    #    и станция использует поднаправления:
    elif len(markers) > 1 and station.use_direction == 'subdir':
        our_markers = markers
        # 1. выбираем первое попавшееся навправление, проходящее через эту
        #    станцию и следующую или предыдущую станцию нитки

        # Пытаемся получить следующую станцию
        next_is_good = False
        next_station = None
        try:
            next_station = rtstations[index + 1].station
        except IndexError:
            pass

        # Проверяем есть ли с этой стацией общее направление
        if next_station is not None:
            common_directions = {}
            next_markers = station_markers.get(next_station, [])
            next_directions = dict([(mkr.direction, mkr) for mkr in next_markers])
            for mr in our_markers:
                if mr.direction in next_directions:
                    next_marker = next_directions[mr.direction]
                    our_marker = mr
                    our_direction = mr.direction
                    common_directions[our_direction] = (our_marker, next_marker)

            if common_directions:
                next_is_good = True
                for d in popular_directions:
                    if d in common_directions:
                        our_direction = d
                        our_marker, next_marker = common_directions[our_direction]
                        break

        # Следущая станция не имеет общих направлений с нашей или это конечная станция
        # Пробуем взять предыдущую станцию и определить поднаправление по ней
        prev_is_good = False
        prev_station = None
        if not next_is_good:
            try:
                prev_station = rtstations[index - 1].station
            except IndexError:
                pass
            # Проверяем есть ли с этой стацией общее направление
            if prev_station is not None:
                common_directions = {}
                prev_markers = station_markers.get(prev_station, [])
                prev_directions = dict([(mkr.direction, mkr) for mkr in prev_markers])
                for mr in our_markers:
                    if mr.direction in prev_directions:
                        prev_marker = prev_directions[mr.direction]
                        our_marker = mr
                        our_direction = mr.direction
                        common_directions[our_direction] = (our_marker, prev_marker)

                if common_directions:
                    prev_is_good = True
                    for d in popular_directions:
                        if d in common_directions:
                            our_direction = d
                            our_marker, prev_marker = common_directions[our_direction]
                            break

        # Логируем ошибки привязки
        if not next_is_good and next_station:
            log.error(u"%s: Станции %s %s и %s %s лежат на границах направлений,"
                      u" и у них нет ни одного общего",
                      thread_number, station.id, station.title, next_station.id, next_station.title)

        if not prev_is_good and prev_station:
            log.error(u"%s: Станции %s %s и %s %s лежат на границах направлений,"
                      u" и у них нет ни одного общего",
                      thread_number, station.id, station.title, prev_station.id, prev_station.title)

        if not next_is_good and not prev_is_good:
            log.error(u"%s: Не удалось привязать станцию %s %s, нет соседних станций с общими направлениями",
                      thread_number, station.id, station.title)
            return

        # 2. если в маршруте нитки текущая и следующая станция находится в том же порядке, что и в этом направлении,
        #    1. то в «поднаправление» кладём «туда»
        #    2. иначе в «поднаправление» кладём «обратно»
        # Привязываем по следующей станции

        # Для поднаправлений прибытия берем заголовок из маркера текущей станции

        if next_is_good:
            # Это поднаправление 'туда'
            if next_marker.order > our_marker.order:
                # Если у маркера следующей станции проставлено поднаправление 'туда'
                # берем его, иначе берем поднаправление 'туда' из направления
                departure_subdir = next_marker.title_from or our_direction.title_from
                arrival_subdir = our_marker.arrival_title_from or our_marker.title_from or our_direction.title_from
                is_from_subdir = True
            # Это поднаправление 'обратно'
            elif next_marker.order < our_marker.order:
                # Если у маркера следующей станции проставлено поднаправление 'обратно'
                # берем его, иначе берем поднаправление 'обратно' из направления
                departure_subdir = next_marker.title_to or our_direction.title_to
                arrival_subdir = our_marker.arrival_title_to or our_marker.title_to or our_direction.title_to
                is_from_subdir = False
            else:
                log.error(u"У нитки %s две станции привязанные к одному и томуже маркеру",
                          thread_number)
                return
        # Привязываем по предыдущей станции
        elif prev_is_good:
            # Это поднаправление 'туда'
            if prev_marker.order < our_marker.order:
                # Берем название поднаправления 'туда' из направления
                departure_subdir = our_marker.title_from or our_direction.title_from
                arrival_subdir = our_marker.arrival_title_from or departure_subdir
                is_from_subdir = True
            # Это поднаправление 'обратно'
            elif prev_marker.order > our_marker.order:
                # Берем название поднаправления 'обратно' из направления
                departure_subdir = our_marker.title_to or our_direction.title_to
                arrival_subdir = our_marker.arrival_title_to or departure_subdir
                is_from_subdir = False
            else:
                log.error(u"У нитки %s две станции привязанные к одному и томуже маркеру",
                          thread_number)
                return

        rts.arrival_subdir = arrival_subdir
        rts.departure_subdir = departure_subdir

        # 3. ссылки на направления (отправления и прибытия) не заполняем

    # 4.2 Если через станцию проходит более одного направления,
    #    и станция использует поднаправления:
    elif len(markers) > 1 and station.use_direction == 'dir':
        # 1. Определяем направление отправления
        # a. берём текущую станцию, предпоследнюю и последнюю станции
        #    маршрута следования данной нитки
        try:
            last_rts = rtstations[-1]
            pre_last_rts = rtstations[-2]
        except IndexError:
            log.error(u"В маршруте %s очень мало станций", thread_number)

        # Если одна из станций совпадает с нашей
        # Выходим т.к. мы не можем правильно выставить направление
        if last_rts == rts or pre_last_rts == rts:
            log.error(u"У конечных станций маршрута %s несколько направлений,"
                      u" не возможно определить направление",
                      thread_number)
        else:
            last_markers = station_markers.get(last_rts.station, [])
            last_directions = dict([(mkr.direction, mkr) for mkr in last_markers])
            pre_last_markers = station_markers.get(pre_last_rts.station, [])
            pre_last_directions = dict([(mkr.direction, mkr)
                                        for mkr in pre_last_markers])

            # Ищем общие направления для все 3-х станций
            common_directions = {}
            for mkr in markers:
                if (mkr.direction in last_directions and
                        mkr.direction in pre_last_directions):
                    pre_last_mkr = pre_last_directions[mkr.direction]
                    last_mkr = last_directions[mkr.direction]
                    common_directions[mkr.direction] = (mkr, pre_last_mkr, last_mkr)

            # b. выбираем направление,
            #    которое проходит через все эти три станции в том же порядке
            # c. если таких направлений несколько:
            #    выбираем первое попавшееся из других направлений,
            #    которые проходят через текущую станцию
            from_direction = None
            departure_direction = None
            for d in popular_directions:
                mkr, pre_last_mkr, last_mkr = common_directions.get(d, (None, None, None))
                if mkr is None:
                    continue
                # определяем направление
                # Это направление 'туда'
                if mkr.order < pre_last_mkr.order < last_mkr.order:
                    departure_direction = mkr.direction
                    from_direction = True
                    is_from_subdir = True
                    break
                # Это направление 'обратно'
                elif mkr.order > pre_last_mkr.order > last_mkr.order:
                    departure_direction = mkr.direction
                    from_direction = False
                    is_from_subdir = False
                    break

            #    если других направлений не осталось, то ругаемся в лог
            if not departure_direction:
                log.error(u"%s: Для станции %s %s не смогли выбрать из нескольких"
                          u" направлений направление отправления",
                          thread_number, station.id, station.title)

            # d. в «направление отправления» кладём ссылку на это направление
            # это может быть None на этом этапе

            rts.departure_direction = departure_direction

            # Если нашли общие направления
            if departure_direction:
                # e. определяем поднаправление в выбранном направлении (*)
                rts.departure_subdir = (departure_direction.title_from
                                        if from_direction
                                        else departure_direction.title_to)
                # Берем следующую станцию она обязательно есть
                # т.к. мы нашли предпоследнюю и последнюю станции
                subdir = None
                next_station = rtstations[index + 1].station
                # Если следующая станция лежит в нашем направлении
                # пробуем взять название поднаправления из ее маркера
                for marker in station_markers.get(next_station, []):
                    if marker.direction == departure_direction:
                        subdir = ((marker.title_from or departure_direction.title_from)
                                  if from_direction
                                  else (marker.title_to or departure_direction.title_to))
                rts.departure_subdir = subdir or rts.departure_subdir

        # 2. Для прибытия:
        # a. берём первую и вторую станции маршрута следования данной нитки,
        #    текущую станцию
        try:
            first_rts = rtstations[0]
            second_rts = rtstations[1]
        except IndexError:
            log.error(u"В маршруте %s очень мало станций", thread_number)
        # Если одна из станций совпадает с нашей
        # Выходим т.к. мы не можем правильно выставить направление
        if first_rts == rts or second_rts == rts:
            log.error(u"У первых станций маршрута %s несколько направлений,"
                      u" не возможно определить направление",
                      thread_number)
        else:
            first_markers = station_markers.get(first_rts.station, [])
            first_directions = dict((mkr.direction, mkr) for mkr in first_markers)
            second_markers = station_markers.get(second_rts.station, [])
            second_directions = dict((mkr.direction, mkr) for mkr in second_markers)

            # Ищем общие направления для все 3-х станций
            common_directions = {}
            for mkr in markers:
                if (mkr.direction in first_directions and
                        mkr.direction in second_directions):
                    second_mkr = second_directions[mkr.direction]
                    first_mkr = first_directions[mkr.direction]
                    common_directions[mkr.direction] = (first_mkr, second_mkr, mkr)

            # b. выбираем направление,
            #    которое проходит через все эти три станции в том же порядке
            # c. если таких направлений несколько:
            #    выбираем первое попавшееся из других направлений,
            #    которые проходят через текущую станцию
            arrival_direction = None
            from_direction = None
            for d in popular_directions:
                first_mkr, second_mkr, mkr = common_directions.get(d, (None, None, None))
                if mkr is None:
                    continue
                # определяем направление
                # Это направление 'туда'
                if first_mkr.order < second_mkr.order < mkr.order:
                    arrival_direction = mkr.direction
                    is_from_subdir = True
                    from_direction = True
                    break
                # Это направление 'обратно'
                elif first_mkr.order > second_mkr.order > mkr.order:
                    arrival_direction = mkr.direction
                    from_direction = False
                    is_from_subdir = False
                    break

            #    если других направлений не осталось, то ругаемся в лог
            if not arrival_direction:
                log.error(u"%s: Для станции %s %s не смогли выбрать из нескольких"
                          u" направлений направление отправления",
                          thread_number, station.id, station.title)

            # d. в «направление прибытия» кладём ссылку на это направление
            # это может быть None на этом этапе

            rts.arrival_direction = arrival_direction

            # Если нашли общие направления
            if arrival_direction:
                # e. определяем поднаправление в выбранном направлении (*)
                rts.arrival_subdir = (arrival_direction.title_from
                                      if from_direction
                                      else arrival_direction.title_to)
                # Берем следующую станцию маршрута
                # т.к. мы нашли предпоследнюю и последнюю станции
                subdir = ((mkr.arrival_title_from or mkr.title_from or arrival_direction.title_from)
                          if from_direction
                          else (mkr.arrival_title_to or mkr.title_to or arrival_direction.title_to))
                rts.arrival_subdir = subdir or rts.arrival_subdir

        # Если одно из направлений не определилось берем данные из другого
        if not rts.arrival_direction:
            rts.arrival_direction = rts.departure_direction
            rts.arrival_subdir = rts.departure_subdir
        if not rts.departure_direction:
            rts.departure_direction = rts.arrival_direction
            rts.departure_subdir = rts.arrival_subdir

    # Сохряняем изменения
    rts.is_from_subdir = is_from_subdir
    rts.save()


def bind_thread(thread):
    """Привязка стаций нитки к направлениям"""
    rtstations = list(thread.rtstation_set.all().select_related())
    log.info(u"Привязываем нитку %s %s", thread.number, thread.title)
    station_markers = Helper.get_station_markers()
    popular_directions = {}
    for rts in rtstations:
        station = rts.station
        markers = station_markers.get(station, [])
        directions = [marker.direction for marker in markers]
        for d in directions:
            try:
                popular_directions[d] += 1
            except KeyError:
                popular_directions[d] = 1

    popular_directions = popular_directions.items()
    popular_directions.sort(key=lambda x: x[1], reverse=True)
    # Направления встречающиеся чаще всего отсортированные по частоте
    popular_directions = [pd[0] for pd in popular_directions]
    for index, rts in enumerate(rtstations):
        bind_rtstation(index, rts, rtstations, popular_directions)


def run(supplier_code=None, route_uid=None):
    # Привязываем все электрички к направлениям
    partial = flags['partial_preparation']
    threads = RThread.objects.filter(t_type_id=TransportType.SUBURBAN_ID)
    if supplier_code:
        threads = threads.filter(supplier__code=supplier_code)
    elif route_uid:
        threads = threads.filter(route__route_uid=route_uid)
    elif partial:
        threads = threads.filter(changed=True)
    else:
        threads = threads

    clean_binds(threads)

    for thread in threads:
        bind_thread(thread)


# Если файл запустили из консоли парсим параметры и запускаем импорт
if __name__ == '__main__':
    with ylog_context(**get_script_log_context()):
        error = False
        route_uid = None
        supplier_code = None
        try:
            options, args = getopt.gnu_getopt(
                sys.argv[1:], 'vs:r:', ['verbose', 'supplier=', 'route_uid=']
            )
            for option in options:
                key, value = option
                if key in ('-v', '--verbose'):
                    print_log_to_stdout(log)
                elif key in ('-s', '--supplier'):
                    supplier_code = value
                elif key in ('-r', '--route_uid'):
                    route_uid = value

        except getopt.GetoptError, e:
            error = unicode(e)

        create_current_file_run_log()

        if error:
            log.error(u'Проблема запуска скрипта: %s', error)
            print error
            print usage
            sys.exit(1)

        run(supplier_code, route_uid)

