#!/usr/bin/env python
# coding: utf-8
""" Коды станций поставщиков """

import travel.rasp.admin.scripts.load_project  # noqa

import argparse
import codecs
import json
import logging
import sys
from collections import defaultdict
from itertools import groupby
from operator import attrgetter

from common.data_api.file_wrapper.config import get_wrapper_creator
from travel.rasp.admin.importinfo.models.mappings import StationMapping
from travel.rasp.admin.lib.logs import print_log_to_stdout, create_current_file_run_log, get_script_log_context, ylog_context
from travel.rasp.admin.scripts.utils.file_wrapper.registry import FileType, FILE_PATH

log = logging.getLogger(__name__)


VENDOR_MARKER = '_vendor_'
LEGACY_MARKER = 'legacy'


def get_code_titles(mappings):
    code_titles = defaultdict(list)
    for mapping in mappings:
        code = mapping.code.split(VENDOR_MARKER, 1)[-1]
        if LEGACY_MARKER not in code:
            code_titles[code].append(mapping.title)
    return code_titles


def main():
    result = {}
    result_suppliers = result['suppliers'] = []

    for supplier_id, supplier_mappings in groupby(
        StationMapping.objects.filter(code__contains=VENDOR_MARKER).order_by('supplier_id', 'station_id'),
        key=attrgetter('supplier_id')
    ):
        code_station_id_titles = defaultdict(lambda: defaultdict(set))
        result_station_codes = []

        for station_id, station_mappings in groupby(supplier_mappings, key=attrgetter('station_id')):
            code_titles = get_code_titles(station_mappings)

            if not code_titles:
                continue

            if len(code_titles) > 1:
                log.warning(u'Пропускаем станцию %s поставщика %s так как ей соответствует %s кодов',
                            station_id, supplier_id, len(code_titles))
                continue

            code, titles = code_titles.items()[0]
            code_station_id_titles[code][station_id].update(titles)

        for code, station_id_titles in code_station_id_titles.items():
            if len(station_id_titles) > 1:
                log.warning(u'Пропускаем код %r поставщика %s так как ему соответствует %s станций',
                            code, supplier_id, len(station_id_titles))
                continue

            station_id, titles = station_id_titles.items()[0]
            result_station_codes.append({'code': code, 'titles': list(titles), 'station_id': station_id})

        if result_station_codes:
            result_suppliers.append({
                'id': supplier_id,
                'station_codes': result_station_codes
            })

    with open(FILE_PATH[FileType.BUS_STATION_CODES], 'w') as outfile:
        json.dump(result, codecs.getwriter('utf-8')(outfile), ensure_ascii=False)

    file_wrapper = get_wrapper_creator(FileType.BUS_STATION_CODES).get_file_wrapper(FILE_PATH[FileType.BUS_STATION_CODES])
    file_wrapper.upload()


if __name__ == '__main__':
    with ylog_context(**get_script_log_context()):
        parser = argparse.ArgumentParser(description=__doc__)
        parser.add_argument('-v', '--verbose', action='store_true', help=u'выводить лог на экран')
        args = parser.parse_args()

        if args.verbose:
            print_log_to_stdout()

        create_current_file_run_log()

        main()
