#!/usr/bin/env python
# coding: utf-8

""" Экспорт городов с пригородным сообщением (RASP-4422) """

import travel.rasp.admin.scripts.load_project  # noqa

import os.path

from django.conf import settings
from django.db.models import Avg
from django.utils import translation
from lxml import etree

from common.data_api.file_wrapper.config import get_wrapper_creator
from common.models.geo import Settlement
from common.models.transport import TransportType
from travel.rasp.admin.lib.logs import create_current_file_run_log, get_script_log_context, ylog_context
from travel.rasp.admin.scripts.utils.file_wrapper.registry import FileType, FILE_DIR_PATH


FILENAMES_BY_NATIONAL_VERSION_AND_LANG = {
    ('ru', 'ru'): ('suburban_cities_ru_ru.xml', 'suburban_cities.xml'),
    ('ua', 'uk'): ('suburban_cities_ua_uk.xml',),
    ('ua', 'ru'): ('suburban_cities_ua_ru.xml',),
    ('tr', 'tr'): ('suburban_cities_tr_tr.xml',),
}


def generate_xml(settlements, national_version, lang):
    with translation.override(lang):
        root_el = etree.Element('suburbancities')

        for settlement in settlements:
            city_el = etree.Element(
                'city',
                id=str(settlement.id),
                title=settlement.L_title(),
                importance=str(settlement.majority_id),
            )

            if settlement.region:
                city_el.set('region', settlement.region.L_title())

            country = get_settlement_real_country(settlement, national_version)

            if country:
                city_el.set('country', country.L_title())

            if hasattr(settlement, 'median_lat'):
                city_el.set('lat', '%.6f' % settlement.median_lat)
                city_el.set('lon', '%.6f' % settlement.median_lon)

            root_el.append(city_el)

        return etree.tostring(root_el, xml_declaration=True, encoding='utf-8', pretty_print=True)


def get_settlements():
    settlements = list(
        Settlement.objects.filter(
            station__rtstation__thread__t_type_id=TransportType.SUBURBAN_ID,
            suburban_zone__isnull=False
        )
        .order_by('title', 'title_en')
        .distinct(),
    )

    for settlement in settlements:
        # Координаты города - усредненные координаты станций.
        # Будем надеяться, что город не находится на 180-ом меридиане.
        median_coords = settlement.station_set.filter(
            latitude__isnull=False).aggregate(Avg('latitude'), Avg('longitude'))

        if median_coords['latitude__avg']:
            settlement.median_lat = median_coords['latitude__avg']
            settlement.median_lon = median_coords['longitude__avg']

    return settlements


def get_settlement_real_country(settlement, national_version):
    real_country = settlement.translocal_country(national_version=national_version)

    return real_country or settlement.country


if __name__ == '__main__':
    with ylog_context(**get_script_log_context()):
        create_current_file_run_log()

        settlements = get_settlements()

        for (national_version, lang), filenames in FILENAMES_BY_NATIONAL_VERSION_AND_LANG.iteritems():
            xml = generate_xml(settlements, national_version, lang)

            for filename in filenames:
                with open(os.path.join(settings.MEDIA_ROOT, 'data', 'export', filename), 'w') as f:
                    f.write(xml)

        dir_wrapper = get_wrapper_creator(FileType.MEDIA_DATA_EXPORT).get_file_wrapper(FILE_DIR_PATH[FileType.MEDIA_DATA_EXPORT])
        dir_wrapper.upload()
