# coding: utf-8

import travel.rasp.admin.scripts.load_project  # noqa

import logging
import sys
import gzip
import os.path
import getopt
from datetime import timedelta, datetime

from django.conf import settings

from common.models.geo import Station, ExternalDirection
from common.models.schedule import RThread
from common.utils.date import daterange
from travel.rasp.admin.lib.maintenance.flags import get_current_job
from travel.rasp.admin.lib.logs import print_log_to_stdout, create_current_file_run_log, get_script_log_context, ylog_context
from travel.rasp.admin.scripts.support_methods import delete_old_export_files


script_name = os.path.basename(__file__)
PREFIX = '.in_process'

log = logging.getLogger(__name__)

usage = "Usage: python2.4 %(script_name)s [-v]" % {'script_name': script_name}


STATION_EXPORT_FILE = os.path.join(settings.EXPORT_PATH, "export_nearest_stations.tsv.gz")
DIRECTION_EXPORT_FILE = os.path.join(settings.EXPORT_PATH, "export_nearest_directions.tsv.gz")
ROUTES_EXPORT_TEMPLATE = os.path.join(settings.EXPORT_PATH, "export_nearest_routes_%Y_%m_%d.tsv.gz")


def run():
    job, job_text = get_current_job()
    if job:
        log.error(u"Нельзя запускать экспорт: %s", job_text)
        sys.exit(1)

    log.info(u"Export Stations")
    station_file = gzip.open(STATION_EXPORT_FILE + PREFIX, 'w')
    for station in Station.objects.all().select_related('settlement'):
        sz = station.settlement and station.settlement.suburban_zone
        if not sz and station.region:
            try:
                capital = station.region.settlement_set.all().order_by('majority__id')[0]
                sz = capital.suburban_zone
            except IndexError:
                sz = None
        record = list()
        record.append(u"%d" % station.id)
        record.append(u"%s" % (station.short_title or station.title))
        record.append(u"%s" % (
            station.externaldirectionmarker_set.filter(external_direction__suburban_zone=sz) and
            station.externaldirectionmarker_set.filter(external_direction__suburban_zone=sz)[0].external_direction_id or
            u"")
        )
        record.append(u"%s" % (station.settlement and station.settlement._geo_id or u""))
        record.append(u"%s" % (sz and sz.title or u""))
        station_file.write(u"\t".join(record).encode('utf8') + '\n')

    station_file.close()
    os.rename(STATION_EXPORT_FILE + PREFIX, STATION_EXPORT_FILE)
    log.info(u"Export Directions")
    direction_file = gzip.open(DIRECTION_EXPORT_FILE + PREFIX, 'w')
    for direction in ExternalDirection.objects.all():
        record = list()
        record.append(u"%s" % direction.id)
        record.append(u"%s" % direction.title)
        direction_file.write(u"\t".join(record).encode('utf8') + "\n")
    direction_file.close()
    os.rename(DIRECTION_EXPORT_FILE + PREFIX, DIRECTION_EXPORT_FILE)

    log.info(u"Export routes")
    # 1 Это Москва
    threads = (RThread.objects.filter(route__t_type__code='suburban', route__supplier__code='af',
                                      rtstation__station__region=1, rtstation__isnull=False)
                      .exclude(type__code=u"cancel").distinct())

    for day in daterange(datetime.today(), datetime.today() + timedelta(3)):
        routes_file_name = day.strftime(ROUTES_EXPORT_TEMPLATE)
        log.info(u"Генерируем %s", routes_file_name)
        routes_file = gzip.open(routes_file_name + PREFIX, 'w')
        for thread in threads:
            record = list()
            record.append(thread.number)
            if thread.runs_at(day):
                rtstations = thread.rtstation_set.all()
                for rtstation in rtstations:
                    record.append(u"%d" % rtstation.station_id)
                    # Прибытие
                    record.append(rtstation.arrival is not None and
                                  (datetime.combine(day, thread.start_time) +
                                   timedelta(minutes=rtstation.arrival)
                                   ).strftime("%Y%m%d%H%M") or u"")
                    # Отправление
                    record.append(rtstation.departure is not None and
                                  (datetime.combine(day, thread.start_time) +
                                   timedelta(minutes=rtstation.departure)
                                   ).strftime("%Y%m%d%H%M") or u"")
                routes_file.write(u"\t".join(record).encode('utf8') + '\n')

        routes_file.close()
        os.rename(routes_file_name + PREFIX, routes_file_name)

    # Удаляем старые файлы экспорта
    log.info(u"Удаляем старые файлы экспорта")
    delete_old_export_files(ROUTES_EXPORT_TEMPLATE, log=log)


# Если файл запустили из консоли парсим параметры и запускаем импорт
if __name__ == '__main__':
    with ylog_context(**get_script_log_context()):
        error = False
        try:
            options, args = getopt.gnu_getopt(sys.argv[1:], 'v', ['verbose'])
            for option in options:
                key, value = option
                if key in ('-v', '--verbose'):
                    print_log_to_stdout(log)

        except getopt.GetoptError, e:
            error = unicode(e)

        create_current_file_run_log()

        if error:
            log.error(u'Проблема запуска скрипта: %s', error)
            print error
            print usage
            sys.exit(1)
        run()
