# -*- coding: utf-8 -*-
"""
Список станций пригородных зон [RASP-7889]
"""

import travel.rasp.admin.scripts.load_project  # noqa

import time
import os
import logging
from xml.etree import ElementTree

from django.conf import settings

from common.models.geo import SuburbanZone, Station, StationCode, Region, Settlement
from travel.rasp.admin.lib.logs import create_current_file_run_log, get_script_log_context, ylog_context


log = logging.getLogger(__name__)


def get_suburban_zone_xml(suburban_zone):
    tree = ElementTree.Element('citystations')
    markers = suburban_zone.markers()
    stations = Station.objects.in_bulk(marker.station_id for marker in markers).values()
    export_stations_list(tree, stations, markers)
    xml = ElementTree.tostring(tree, 'UTF-8')
    return xml


def export_stations_list(tree, stations, markers=None, city=None):
    id_to_code = StationCode.id_to_code_dict_for('esr', stations)
    visited_stations = {}

    regions = Region.objects.in_bulk(station.region_id for station in stations)
    settlements = Settlement.objects.in_bulk(station.settlement_id for station in stations)

    for station in stations:
        try:
            esr_code = id_to_code[station.id]
        except KeyError:
            continue

        if esr_code in visited_stations:
            continue

        if not station.has_suburban_traffic():
            continue

        title = station.L_title()
        popular_title = station.L_popular_title(fallback=False)

        title = append_station_title_prefix(station, title)

        direction = get_station_direction(station, markers, city)

        branch = ElementTree.Element(
            'station',
            esr=esr_code,
            title=title or u'',
            popular_title=popular_title or u'',
            importance=str(station.majority_id),
            region=regions[station.region_id].title if station.region_id in regions else '',
            direction=direction.title if direction else ''
        )

        if station.settlement_id in settlements:
            branch.set('city', settlements[station.settlement_id].title)

        if station.latitude:
            branch.set('lat', '%.6f' % station.latitude)
            branch.set('lon', '%.6f' % station.longitude)

        if station.short_title:
            branch.set('short_title', station.short_title)

        tree.append(branch)
        visited_stations[esr_code] = True


def append_station_title_prefix(station, title):
    if station.t_type_id == 2:
        return u'аэропорт ' + title
    else:
        return title


def get_station_direction(station, markers, city):
    if markers:
        for marker in markers:
            if marker.station_id == station.pk:
                return marker.external_direction

    else:
        try:
            return (station.externaldirectionmarker_set
                           .filter(external_direction__suburban_zone__settlement=city)
                           .select_related('external_direction')[0]
                           .external_direction)
        except IndexError:
            pass

    return None


def run():
    start = time.time()
    prefix = '.inprocess'

    log.info(u'Start')

    for zone in SuburbanZone.objects.all():
        xml = get_suburban_zone_xml(zone)

        filename = os.path.join(settings.EXPORT_SUBURBAN_PATH, 'suburban_zone_%d.xml' % zone.id)
        export = open(filename + prefix, 'w')
        export.write(xml)
        export.close()

        os.rename(filename + prefix, filename)

    log.info(u"Finish %f", time.time() - start)


if __name__ == '__main__':
    with ylog_context(**get_script_log_context()):
        create_current_file_run_log()
        try:
            run()
        except Exception:
            log.exception(u"failed")
