# coding: utf8
from __future__ import unicode_literals, division, print_function

"""
RASPADMIN-793 Генерация файла для мобильной морды
В файле должны быть geoid и актуальные виды транспорта
"""

import travel.rasp.admin.scripts.load_project  # noqa

import json
import logging
import os.path
from collections import defaultdict

from django.conf import settings

from common.data_api.file_wrapper.config import get_wrapper_creator
from common.models.geo import Settlement, Station, Station2Settlement, StationMajority, CityMajority
from common.models.transport import TransportType
from common.utils.iterrecipes import chunker
from travel.rasp.admin.lib.logs import print_log_to_stdout, create_current_file_run_log, get_script_log_context, ylog_context
from travel.rasp.admin.scripts.utils.file_wrapper.registry import FileType, FILE_PATH


log = logging.getLogger(__name__)


TRAIN_AND_SUBURBAN_IDS = {TransportType.TRAIN_ID, TransportType.SUBURBAN_ID}


# https://st.yandex-team.ru/SUBURBAN-238
FORCE_PLANE_REGIONS = {
    1,      # Московская,
    10174,  # Ленинградская
    11162,  # Свердловская
    11316,  # Новосибирская
    11079,  # Нижегородская
    11029,  # Ростовская
    11131,  # Самарская
    10995,  # Краснодарский край
    977,    # Крым
}


def generate_t_types_by_geoid():
    cities_query = Settlement.hidden_manager.filter(_geo_id__isnull=False).select_related('region')

    geoid_by_city_ids = {}
    capitals_for_regions = {}
    force_plane_settlements = set()
    for id_, geoid, majority, region_geoid in cities_query.values_list('id', '_geo_id', 'majority', 'region___geo_id'):
        geoid_by_city_ids[id_] = geoid

        if majority in [CityMajority.CAPITAL_ID, CityMajority.REGION_CAPITAL_ID]:
            capitals_for_regions[region_geoid] = geoid

        if region_geoid in FORCE_PLANE_REGIONS:
            force_plane_settlements.add(id_)

    log.info(u'Всего нужно обработать %s населенных пунктов', len(geoid_by_city_ids))

    t_type_ids_by_city_ids = {}
    for chunk_city_ids in chunker(geoid_by_city_ids.keys(), chunk_size=3000):
        t_type_ids_by_city_ids.update(get_t_type_ids_by_city_ids(chunk_city_ids))
    log.info(u'Населенных пунктов с известным расписанием %s', len(t_type_ids_by_city_ids))

    add_extra_t_type_ids(t_type_ids_by_city_ids)
    add_suburban(t_type_ids_by_city_ids)

    result = {
        geoid_by_city_ids[city_id]: {
            'train': has_train(t_type_ids),
            'suburban': has_suburban(t_type_ids),
            'plane': city_id in force_plane_settlements or has_plane(t_type_ids),
            'bus': has_bus(t_type_ids),
            'water': has_water(t_type_ids),
        }
        for city_id, t_type_ids in t_type_ids_by_city_ids.iteritems() if t_type_ids
    }

    # https://st.yandex-team.ru/SUBURBAN-238
    for region_geo_id, capital_geo_id in capitals_for_regions.items():
        capital_t_types = result.get(capital_geo_id)
        if capital_t_types:
            result[region_geo_id] = capital_t_types

    return result


def get_extra_t_type_ids_by_city_ids():
    station_ids = set(Station2Settlement.objects.all().values_list('station_id', flat=True))
    visible_t_type_ids_by_station_ids = {
        id_: t_type_id
        for id_, t_type_id in Station.hidden_manager
                                     .filter(id__in=station_ids, majority__id__lte=StationMajority.IN_TABLO_ID)
                                     .values_list('id', 't_type_id')
    }

    extra_t_type_ids_by_city_ids = defaultdict(set)
    city_and_station_query = Station2Settlement.objects.filter(station_id__in=visible_t_type_ids_by_station_ids.keys())\
                                                       .values_list('settlement_id', 'station_id')
    for city_id, station_id in city_and_station_query:
        extra_t_type_ids_by_city_ids[city_id].add(visible_t_type_ids_by_station_ids[station_id])

    return extra_t_type_ids_by_city_ids


def get_t_type_ids_by_city_ids(city_ids):
    t_type_ids_by_city_ids = defaultdict(set)
    city_and_station_query = Station.hidden_manager.filter(settlement_id__in=city_ids) \
                                                   .filter(majority__id__lte=StationMajority.IN_TABLO_ID) \
                                                   .values_list('settlement_id', 't_type_id')
    for city_id, t_type_id in city_and_station_query:
        t_type_ids_by_city_ids[city_id].add(t_type_id)

    return t_type_ids_by_city_ids


def add_extra_t_type_ids(t_type_ids_by_city_ids):
    extra_t_type_ids_by_city_ids = get_extra_t_type_ids_by_city_ids()

    for city_id in t_type_ids_by_city_ids:
        t_type_ids_by_city_ids[city_id] |= extra_t_type_ids_by_city_ids[city_id]


def add_suburban(t_type_ids_by_city_ids):
    city_ids_with_train = [city_id for city_id, t_type_ids in t_type_ids_by_city_ids.iteritems()
                           if TransportType.TRAIN_ID in t_type_ids]
    log.info(u'Нужно отделить электрички от поездов для %s населенных пунктов', len(city_ids_with_train))

    city_ids_with_suburban_threads = set(Settlement.objects.filter(
        station__rtstation__thread__t_type=TransportType.SUBURBAN_ID
    ).values_list('id', flat=True).distinct())

    city_ids_with_train_threads = set(Settlement.objects.filter(
        station__rtstation__thread__t_type=TransportType.TRAIN_ID
    ).values_list('id', flat=True).distinct())

    for city_id in city_ids_with_train:
        if city_id not in city_ids_with_train_threads:
            t_type_ids_by_city_ids[city_id] -= {TransportType.TRAIN_ID}

        if city_id in city_ids_with_suburban_threads:
            t_type_ids_by_city_ids[city_id] |= {TransportType.SUBURBAN_ID}


def has_train(t_type_ids):
    return TransportType.TRAIN_ID in t_type_ids


def has_suburban(t_type_ids):
    return TransportType.SUBURBAN_ID in t_type_ids


def has_plane(t_type_ids):
    return TransportType.PLANE_ID in t_type_ids


def has_bus(t_type_ids):
    return TransportType.BUS_ID in t_type_ids


def has_water(t_type_ids):
    return TransportType.WATER_ID in t_type_ids


def main():
    log.info(u'Начинаем генерацию t_types_by_geoid.json')
    result = generate_t_types_by_geoid()

    file_path = FILE_PATH[FileType.T_TYPES_BY_GEOID]
    with open(file_path, 'w') as fp:
        json.dump(result, fp)

    file_wrapper = get_wrapper_creator(FileType.T_TYPES_BY_GEOID).get_file_wrapper(file_path)
    file_wrapper.upload()

    log.info(u'Успешно завершили генерацию t_types_by_geoid.json')


if __name__ == '__main__':
    with ylog_context(**get_script_log_context()):
        import argparse

        parser = argparse.ArgumentParser()
        parser.add_argument("-v", "--verbose", dest="verbose", action="store_true")
        args = parser.parse_args()

        create_current_file_run_log()
        if args.verbose:
            print_log_to_stdout()

        main()
