# coding: utf-8
from __future__ import unicode_literals

import travel.rasp.admin.scripts.load_project  # noqa

import argparse
import re
import logging

from django.db import connection

from common.models.schedule import RThread, TrainPurchaseNumber
from common.models.transport import TransportType
from travel.rasp.admin.lib.logs import print_log_to_stdout, create_current_file_run_log, get_script_log_context, ylog_context


log = logging.getLogger(__name__)

SUBURBAN_NUMBER_TAIL_RE = re.compile('/?70(\d\d)')
TRAIN_NUMBER_TAIL_RE = re.compile('^(?P<prefix>70|8)(?P<tail>\d\d)\D*$')


class TrainPurchaseNumberCreator(object):
    def __init__(self, logger, truncate_table=True):
        self.log = logger
        self.truncate_table = truncate_table

    def run(self):
        self.log.info('Ищем дубликаты электричек-экспрессов и поездов')
        numbers = []
        for suburban in RThread.objects.filter(t_subtype__has_train_tariffs=True, t_type=TransportType.SUBURBAN_ID):
            trains = self.get_matched_trains(suburban)
            numbers.extend(self.get_train_numbers(suburban, trains))

        self.log.info('Очистка номеров экспресс для электричек')
        if self.truncate_table:
            cursor = connection.cursor()
            cursor.execute('TRUNCATE TABLE www_trainpurchasenumber')
        else:
            TrainPurchaseNumber.objects.all().delete()

        self.log.info('Вставка новых номеров')
        TrainPurchaseNumber.objects.bulk_create(numbers, batch_size=1000)
        count_suburbans = len({n.thread for n in numbers})
        self.log.info('Добавлено %s номеров экспресс для %s электричек', len(numbers), count_suburbans)

    def get_matched_trains(self, suburban):
        trains = RThread.objects.filter(
            t_type=TransportType.TRAIN_ID,
            tz_start_time=suburban.tz_start_time,
        )
        trains = filter(lambda t: (
            t.path_cached[0].station_id == suburban.path_cached[0].station_id and
            t.path_cached[-1].station_id == suburban.path_cached[-1].station_id and
            t.path_cached[-1].tz_arrival == suburban.path_cached[-1].tz_arrival
        ), trains)
        return trains

    def get_train_numbers(self, suburban, trains):
        if not trains:
            return
        suburban_tails = SUBURBAN_NUMBER_TAIL_RE.findall(suburban.number)
        for number in {t.number for t in trains}:
            train_tail_match = TRAIN_NUMBER_TAIL_RE.match(number)
            if train_tail_match and train_tail_match.group('tail') in suburban_tails:
                yield TrainPurchaseNumber(number=number, thread=suburban)
            else:
                self.log.warning('Электричка %s (uid=%s) не прошла проверку соответсвия с номером %s',
                                 suburban.number, suburban.uid, number)


if __name__ == '__main__':
    with ylog_context(**get_script_log_context()):
        parser = argparse.ArgumentParser()

        parser.add_argument('-v', '--verbose', action='store_true', help='increase output verbosity')

        args = parser.parse_args()

        if args.verbose:
            print_log_to_stdout(log)

        create_current_file_run_log()

        TrainPurchaseNumberCreator(log).run()
