#!/usr/bin/env python2
# coding: utf-8

u"""
Позволяет заливать и выгружать в tanker значения полей L_field.

Пример использования:

Заливает в танкер значения поля title из StationMajority
python scripts/i18n/dbtanker.py upload generic www_stationmajority_title

Просмотреть список доступных ключей
python scripts/i18n/dbtanker.py list generic

Загрузить из танкера переводы всех keysets из generic
python scripts/i18n/dbtanker.py download generic '*'

Загрузить в танкер переводы всех keysets из generic
python scripts/i18n/dbtanker.py upload generic '*'


Для работы метода generic необходимо в модель добавить TANKER_L_FIELDS,
например, TANKER_L_FIELDS = [ 'title' ]. TANKER_KEY_FIELD определяет поле являющееся
ключем в танкере
"""

import travel.rasp.admin.scripts.load_project  # noqa

import sys

from django.apps import apps
from django.utils.encoding import smart_str, force_unicode

from common.models.disclaimers import StaticText
from common.models.geo import Station
from common.models.schedule import Company
from common.models_utils import i18n
from common.models_utils.i18n import RouteLTitle
from common.utils.tanker import Tanker
from travel.rasp.admin.lib.logs import get_script_log_context, ylog_context


class Generic(object):
    class KeySet(object):
        def __init__(self, model, field):
            self.name = '%s_%s_%s' % (model._meta.app_label, model._meta.model_name, field)
            self.model = model
            self.field = field
            self.count = model.objects.all().count()

    def __init__(self):
        self.keysets = []

        for model in apps.get_models():
            if hasattr(model, 'TANKER_L_FIELDS'):
                for field in model.TANKER_L_FIELDS:
                    keyset = self.KeySet(model, field)

                    self.keysets.append(keyset)

    def select(self, patterns):
        from fnmatch import fnmatch

        error = False

        selected = []

        for keyset_pattern in patterns:
            matched = False

            for keyset in self.keysets:
                if fnmatch(keyset.name, keyset_pattern):
                    selected.append(keyset)

                    if keyset not in selected:
                        selected.append(keyset)

                    matched = True

            if not matched:
                print >>sys.stderr, "Unknown keyset", keyset_pattern
                error = True

        if error:
            sys.exit(1)

        selected = sorted(selected)

        print "Selected keysets:"

        for keyset in selected:
            print keyset.name, keyset.count

        return selected


def keyset_generic_list(args):
    generic = Generic()

    for keyset in generic.keysets:
        print keyset.name, keyset.count


def keyset_generic_upload(args):
    generic = Generic()

    selected = generic.select(args.keyset_patterns)

    for keyset in selected:
        i18n.tankerupload(args.tanker, keyset.model, keyset.model.objects.all(), keyset.field, keyset.name)


def keyset_generic_download(args):
    generic = Generic()

    selected = generic.select(args.keyset_patterns)

    for keyset in selected:
        i18n.tankerdownload(args.tanker, keyset.model.objects.all(), keyset.field, keyset.name)


def keyset_route_titles_upload(args):
    RouteLTitle.tankerupload(args.tanker, args.t_type)


def keyset_route_titles_download(args):
    RouteLTitle.tankerdownload(args.tanker)


def keyset_companies_upload(args):
    i18n.tankerupload(args.tanker, Company, Company.objects.all(), 'title', 'companies_title')


def keyset_companies_download(args):
    i18n.tankerdownload(args.tanker, Company.objects.all(), 'title', 'companies_title')


STATICTEXT_TANKER_WHITELIST = """\
train_schedule_disclaimer
autobus_schedule_disclaimer
autobus_station_schedule_disclaimer
sea_schedule_disclaimer
train_schedule_disclaimer
roughly_route_time
station_incomplete_schedule_disclaimer
autobus_schedule_disclaimer
autobus_station_schedule_disclaimer
sea_schedule_disclaimer
suburban_schedule_disclaimer
train_station_not_our_country_disclaimer
e-ticket
alldays_schedule_disclaimer
no_direct_routes_disclaimer
currency_national_disclaimer
belarus_search_disclaimer
bus_station_incomplete_schedule_disclaimer
settlement_incomplete_schedule_disclaimer
direction_help
order_partner
ufs_limit_message
currency_buypage_disclaimer
currency_bus_disclaimer
currency_disclaimer
currency_national_disclaimer
technical_stop_disclaimer""".split()


def keyset_staticpages_statictext_upload(args):
    texts = StaticText.objects.filter(code__in=STATICTEXT_TANKER_WHITELIST)

    keys = {}

    for text in texts:
        translations = {}

        for lang in StaticText.LANGUAGES_TANKER:
            content = getattr(text, 'content_' + lang)

            if content:
                translations[lang] = {
                    'context': text.title,
                    'status': 'approved',
                    'form': content,
                }

        if translations:
            keys[text.code] = {
                'info': {
                    'is_plural': False
                },
                'translations': translations
            }

    args.tanker.upload('static_texts', keys, StaticText.LANGUAGES_TANKER)


def keyset_staticpages_statictext_download(args):
    print smart_str(
        u'Downloading %s[%s]' % (
            force_unicode('static_texts'),
            ','.join(force_unicode(lang) for lang in StaticText.LANGUAGES_TANKER),
        )
    )

    keys = args.tanker.download('static_texts', StaticText.LANGUAGES_TANKER)

    texts = StaticText.objects.all()

    updated = 0

    for text in texts:
        fields_updated = 0

        if text.code in keys:
            translations = keys[text.code]['translations']

            for lang in StaticText.LANGUAGES_TANKER:
                if lang not in translations:
                    continue

                translated_content = translations[lang]['form']

                content = getattr(text, 'content_' + lang)

                if translated_content and translated_content != content:
                    setattr(text, 'content_' + lang, translated_content)

                    updated += 1
                    fields_updated += 1

        if fields_updated:
            text.save()

    print smart_str(u'Updated %d objects' % updated)


def keyset_stations_upload(args):
    if args.scope == 'country':
        stations = Station.objects.filter(country__id=args.scope_object_id)
        keyset = 'stations_country_%d' % args.scope_object_id

    elif args.scope == 'region':
        stations = Station.objects.filter(region__id=args.scrope_object_id)
        keyset = 'stations_region_%d' % args.scope_object_id

    else:
        assert False

    Station.tankerupload(args.tanker, stations, keyset)


def keyset_stations_download(args):
    Station.tankerdownload(args.tanker)


if __name__ == '__main__':
    with ylog_context(**get_script_log_context()):
        from django.utils import translation

        translation.activate('ru')

        import argparse

        superparser = argparse.ArgumentParser(description=__doc__, formatter_class=argparse.RawTextHelpFormatter)

        superparser.add_argument('--tanker', '-t', action='store', default='stable', help=u'Tanker instance'),

        superparser.add_argument('--token', '-u', action='store', default=None, help=u'Project token'),

        command_subparsers = superparser.add_subparsers(dest='command')

        upload_parser = command_subparsers.add_parser('upload')
        download_parser = command_subparsers.add_parser('download')
        list_parser = command_subparsers.add_parser('list')

        upload_subparsers = upload_parser.add_subparsers(dest='keyset')
        download_subparsers = download_parser.add_subparsers(dest='keyset')
        list_subparsers = list_parser.add_subparsers(dest='keyset')

        # route_titles
        upload_subparsers.add_parser('route_titles') \
            .add_argument('-t', '--t-type', choices=['all', 'bus', 'suburban'], default='all',
                          dest='t_type')

        download_subparsers.add_parser('route_titles')

        # companies
        upload_subparsers.add_parser('companies')
        download_subparsers.add_parser('companies')

        # staticpages_statictext
        upload_subparsers.add_parser('staticpages_statictext')
        download_subparsers.add_parser('staticpages_statictext')

        # stations
        stations_parser = upload_subparsers.add_parser('stations')
        stations_parser.add_argument('scope', choices=['country', 'region'])
        stations_parser.add_argument('scope_object_id', type=int)

        download_subparsers.add_parser('stations')

        # generic
        list_subparsers.add_parser('generic')

        upload_subparsers.add_parser('generic') \
            .add_argument('keyset_patterns', nargs='+')

        download_subparsers.add_parser('generic') \
            .add_argument('keyset_patterns', nargs='+')

        # start
        args = superparser.parse_args()

        method = 'keyset_%s_%s' % (args.keyset.replace('-', '_'), args.command)

        args.tanker = Tanker(tanker=args.tanker, token=args.token, project='rasp')

        globals()[method](args)
