#!/usr/bin/env python2
# coding: utf-8

import travel.rasp.admin.scripts.load_project  # noqa

import sys
import logging
import argparse

from django.utils import translation
from django.db.models import Q

from common.models.geo import Station
from common.utils.tanker import Tanker
from travel.rasp.admin.lib.logs import create_current_file_run_log, print_log_to_stdout, get_script_log_context, ylog_context


log = logging.getLogger(__name__)


def get_tanker_key_to_query_set():
    return {
        'db:www_staiton:kiev_region_all': {
            'langs': ['ru', 'uk'],
            'queryset': Station.objects.filter(region=20544, hidden=False).exclude(
                Q(title__icontains=u'#техзуп') |
                Q(title__icontains=u'#тех') |
                Q(title__icontains=u'# тех')
            ),
            'L_fields': [
                {
                    'name': 'L_title',
                    'context': lambda s: u'Название станции. ' + s.L_full_geography()
                }
            ],
            'model': Station,
            'upload_filled': True
        }
    }


def upload_keyset(tanker, keyset):
    translation.activate('ru')

    keyset_settings = get_tanker_key_to_query_set()[keyset]
    query_set = keyset_settings['queryset']
    langs = keyset_settings['langs']
    l_fields = keyset_settings['L_fields']
    model = keyset_settings['model']
    upload_filled = keyset_settings['upload_filled']

    for l_field_info in l_fields:
        l_field = getattr(model, l_field_info['name'])

        keyset_name = keyset + '_' + l_field.field_name + '_' + ','.join(langs)

        keys = {}

        for obj in query_set:
            translations = {}

            context = l_field_info['context'](obj)
            if l_field.add_local_field:
                context += u' %s=%s' % (l_field.field_name, getattr(obj, l_field.field_name))

            for lang in langs:
                form = getattr(obj, '%s_%s' % (l_field.field_name, lang))

                if not form and lang == 'ru' and l_field.add_local_field:
                    form = getattr(obj, l_field.field_name)

                if form:
                    translations[lang] = {
                        "form": form,
                    }

            has_all_translations = len(translations) == len(langs)

            if has_all_translations and not upload_filled:
                continue

            keys[unicode(obj.id)] = {
                "info": {
                    "is_plural": False,
                    "context": context
                },

                "translations": translations
            }

        tanker.upload(keyset_name, keys, langs)

    log.info(u'Загрузили keyset=%s в танкер', keyset)


def download_keyset(tanker, keyset):
    translation.activate('ru')

    keyset_settings = get_tanker_key_to_query_set()[keyset]
    query_set = keyset_settings['queryset']
    langs = keyset_settings['langs']
    l_fields = keyset_settings['L_fields']
    model = keyset_settings['model']

    for l_field_info in l_fields:
        l_field = getattr(model, l_field_info['name'])

        keyset_name = keyset + '_' + l_field.field_name + '_' + ','.join(langs)

        keys = tanker.download(keyset_name, langs)

        objects = query_set.filter(pk__in=keys.keys())

        for obj in objects:
            fields_updated = 0

            key = keys[unicode(obj.id)]

            for lang in langs:
                override = getattr(obj, '%s_%s_override' % (l_field.field_name, lang), False)

                if not override:
                    form = (key['translations'][lang]['form'] or u'').strip()

                    old_form = (getattr(obj, '%s_%s' % (l_field.field_name, lang)) or u'').strip()

                    if form and old_form != form:
                        setattr(obj, '%s_%s' % (l_field.field_name, lang), form)

                        log.info(u'Update %(model)s: %(pk)s: %(field_name)s "%(old_form)s" -> %(new_form)s', {
                            'model': model.__name__,
                            'pk': obj.pk,
                            'field_name': '%s_%s' % (l_field.field_name, lang),
                            'old_form': old_form,
                            'new_form': form
                        })

                        fields_updated += 1

            if fields_updated:
                obj.save()

    log.info(u'Обновили переводы из keyset=%s', keyset)


def main(args):
    parser = argparse.ArgumentParser(formatter_class=argparse.RawTextHelpFormatter)

    parser.add_argument('--tanker', '-t', action='store', default='testing', help=u'Tanker instance')
    parser.add_argument('--branch', '-b', action='store', default='master', help=u'Tanker branch')
    parser.add_argument('--verbose', '-v', action='store_true', default=False, help=u'Verbose')

    parser.add_argument('action', action='store', help=u'Action', nargs=1,
                        choices=['upload', 'download'])

    parser.add_argument('keysets', action='store', help=u'Keyset', nargs='+',
                        choices=list(get_tanker_key_to_query_set().keys()))

    args = parser.parse_args(args)

    if args.verbose:
        print_log_to_stdout()

    if args.action[0] == 'upload':
        action = upload_keyset
    elif args.action[0] == 'download':
        action = download_keyset

    tanker = Tanker(tanker=args.tanker, branch=args.branch, project='rasp')

    for keyset in args.keysets:
        action(tanker, keyset)


if __name__ == '__main__':
    with ylog_context(**get_script_log_context()):
        create_current_file_run_log()
        main(sys.argv[1:])
