#!/usr/bin/env python
# coding: utf-8

from __future__ import absolute_import, print_function, unicode_literals

import travel.rasp.admin.scripts.load_project  # noqa

import json
from collections import Counter

from common.models.geo import Station
from common.utils.geo import GeoPoint
from travel.rasp.admin.lib.logs import print_log_to_stdout, create_current_file_run_log, get_script_log_context, ylog_context
from travel.rasp.admin.www.models.geo import RoutePath


class ArcInfo(object):
    def __init__(self, d):
        data = json.loads(d) if d else None

        self.start_point = GeoPoint(data[0][1], data[0][0])
        self.end_point = GeoPoint(data[-1][1], data[-1][0])

        self.data = data


class PathInfo(object):
    STATUS_TO_STRING = {
        0: "STATUS_REMOVED",
        1: "STATUS_CONFIRMED",
        2: "STATUS_CHANGED",
    }

    def __init__(self, p):
        self.route_path = p

        self.arc_direct = ArcInfo(p.data_direct) if p.data_direct else None
        self.arc_back = ArcInfo(p.data_back) if p.data_back else None

        self.status_direct = self.STATUS_TO_STRING[p.status_direct]
        self.status_back = self.STATUS_TO_STRING[p.status_back]


class Checker(object):
    def __init__(self, route_path_objects):
        data = {}
        stations_ids = set()

        for route_path in route_path_objects:
            key = (route_path.station_from_id, route_path.station_to_id)

            data.setdefault(key, []).append(PathInfo(route_path))

            stations_ids.add(route_path.station_from_id)
            stations_ids.add(route_path.station_to_id)

        self.data = data
        self.stations = Station.objects.in_bulk(list(stations_ids))

    def check(self):
        direct_statuses = Counter()
        back_statuses = Counter()

        unknown_stations = set()
        reversed_arcs = []

        extra = Counter()

        for (station_from_id, station_to_id), path_infos in self.data.items():
            assert len(path_infos) == 1

            if station_from_id == station_to_id:
                extra['station_from_id == station_to_id'] += 1

            if station_from_id > station_to_id:
                extra['station_from_id > station_to_id'] += 1

            station_from = self.stations.get(station_from_id)
            station_to = self.stations.get(station_to_id)

            if station_from is None:
                unknown_stations.add(station_from_id)

            if station_to is None:
                unknown_stations.add(station_to_id)

            for path_info in path_infos:
                direct_statuses[path_info.status_direct] += 1
                back_statuses[path_info.status_back] += 1

                if path_info.route_path.for_two_directions:
                    extra['for_two_directions'] += 1

                if station_from and station_to:
                    for arc in [path_info.arc_direct]:
                        start_to_from = arc.start_point.distance(station_from)
                        start_to_to = arc.start_point.distance(station_to)
                        end_to_from = arc.end_point.distance(station_from)
                        end_to_to = arc.end_point.distance(station_to)

                        if start_to_from > start_to_to and end_to_to > end_to_from:
                            reversed_arcs.append((station_from_id, station_to_id))

                            print(station_from_id, station_to_id, start_to_from, start_to_to, end_to_from, end_to_to,
                                  path_info.route_path.for_two_directions)

        print("Статусы прямых направлений:", direct_statuses)
        print("Статусы обратных направлений:", back_statuses)

        print("Неизвестные станции:", unknown_stations)
        print("Обратное направление:", len(reversed_arcs))

        for key, value in sorted(extra.items()):
            print(key, value)


if __name__ == '__main__':
    with ylog_context(**get_script_log_context()):
        create_current_file_run_log()

        from optparse import OptionParser

        parser = OptionParser()
        parser.add_option("-v", "--verbose",
                          dest="verbose", action="store_true",
                          help="Print log messages to STDOUT")

        (options, args) = parser.parse_args()

        if options.verbose:
            print_log_to_stdout()

        checker = Checker(RoutePath.objects.all())

        checker.check()
