# coding: utf-8

import travel.rasp.admin.scripts.load_project  # noqa

import logging
import sys

from django.conf import settings
from django.core.management import call_command
from django.db import connection

from common.utils.dump import TIMESTAMP_TABLE_NAME, list_tables, load_dump_url
from common.utils.metrics import task_progress_report
from travel.rasp.admin.lib.logs import create_current_file_run_log, print_log_to_stdout, get_script_log_context, ylog_context
from travel.rasp.admin.lib.maintenance.flags import flags


log = logging.getLogger(__name__)

EXIT_CODE_FAIL = 1


def update_from_tmpdb():
    loaded_dump_url = load_dump_url(settings.LOAD_LATEST_DB_TMP_DATABASE)
    if not loaded_dump_url:
        log.error(u'Свежий дамп отсутствует или не консистентен')
        return EXIT_CODE_FAIL

    if flags['maintenance']:
        log.error(u"Идет работа с базой данных")
        return EXIT_CODE_FAIL

    log.info(u'Обновляем базу из дампа %s', loaded_dump_url)

    cursor = connection.cursor()
    qn = connection.ops.quote_name

    db_name = qn(connection.get_db_name())
    tmp_db_name = qn(settings.LOAD_LATEST_DB_TMP_DATABASE)

    run_sql(cursor, 'DROP DATABASE {}'.format(db_name))
    run_sql(cursor, 'CREATE DATABASE {} DEFAULT CHARSET utf8'.format(db_name))

    tables = list_tables(settings.LOAD_LATEST_DB_TMP_DATABASE)
    for table in tables:
        if table == TIMESTAMP_TABLE_NAME:
            continue

        table = qn(table)

        run_sql(cursor, 'RENAME TABLE {from_}.{table} TO {to}.{table}'.format(
            from_=tmp_db_name, to=db_name, table=table
        ))

    run_sql(cursor, 'DROP DATABASE {}'.format(tmp_db_name))

    log.info(u'Обновление завершено')
    set_up_databse()
    log.info(u'База готова к использованию')


def set_up_databse():
    log.info(u'Подготавливаем базу к работе')
    from django.db import connections

    for c in connections.all():
        c.close()

    flags['maintenance'] = 0

    log.info(u'Запускаем обновление базы')
    call_command('update')

    if settings.LOAD_LATEST_DB_GRANT_SUPERUSER:
        from django.contrib.auth.models import User
        User.objects.all().update(is_superuser=True)

    from travel.rasp.admin.admin.maintenance_settings.models import Script
    Script.objects.filter(code__in=('load_fresh_dump_to_tmpdb', 'update_from_tmpdb'))\
          .update(on_work=True, on_service=True)

    log.info(u'Подготовка завершена')


def run_sql(cursor, sql, args=()):
    if args:
        log.info(u'%s %s', sql, args)
        cursor.execute(sql, args)
    else:
        log.info(u'%s', sql)
        cursor.execute(sql)


run_functions = {
    'load-to-tmpdb': lambda args: call_command(
        'load_db_dump', settings.LOAD_LATEST_DB_TMP_DATABASE,
        db_type=args.db_type if args.db_type != 'any' else None, interactive=False,
        full=settings.LOAD_LATEST_DB_FULL
    ),
    'update-from-tmpdb': lambda args: update_from_tmpdb()
}


if __name__ == '__main__':
    with ylog_context(**get_script_log_context()), task_progress_report('load_latest_db'):
        create_current_file_run_log()

        import argparse

        parser = argparse.ArgumentParser()

        parser.add_argument('-v', '--verbose', action='store_true')
        subparsers = parser.add_subparsers(dest='action')

        load_to_tmpdb_parser = subparsers.add_parser('load-to-tmpdb')
        load_to_tmpdb_parser.add_argument('-t', '--db-type', choices=('any', 'work_db', 'service_db'),
                                          default='any')

        subparsers.add_parser('update-from-tmpdb')

        args = parser.parse_args()

        if args.verbose:
            print_log_to_stdout()

        if not settings.LOAD_LATEST_DB_ALLOWED:
            log.info(u'Загрузка и обновление базы на этом хосте не разрешена')
            sys.exit(EXIT_CODE_FAIL)

        if not settings.LOAD_LATEST_DB_TMP_DATABASE:
            log.error(u'Имя временной базы данных не задано')
            sys.exit(EXIT_CODE_FAIL)

        sys.exit(run_functions[args.action](args) or 0)
