# coding: utf-8

from __future__ import unicode_literals

import travel.rasp.admin.scripts.load_project  # noqa

import logging
import os
import sys
from shutil import copyfile

from django.conf import settings

from common.data_api.file_wrapper.config import get_wrapper_creator
from common.data_api.file_wrapper.registry import FileType
from common.db import maintenance
from common.db.switcher import switcher
from travel.rasp.library.python.common23.date import environment
from common.utils.metrics import task_progress_report
from travel.rasp.admin.lib.logs import create_current_file_run_log, print_log_to_stdout, get_script_log_context, ylog_context
from travel.rasp.admin.lib.maintenance.flags import get_flag, set_flag
from travel.rasp.admin.lib.maintenance.scripts import job
from travel.rasp.admin.lib.mysqlutils import dump_db_by_alias

log = logging.getLogger(__name__)

DUMP_FORMAT = 'dump_{timestamp:%Y%m%d%H%M%S}_{prefix}_{db_name}_{db_type}.sql.gz'
SCHEMA_FORMAT = 'schema_{timestamp:%Y%m%d%H%M%S}_{prefix}_{db_name}_{db_type}.sql.gz'


def run(export_folder, db_type=None, prefix=None, skip_flag_checking=False, save_filename=False, schema_only=False):
    prefix = prefix or ''
    db_type = db_type or settings.INSTANCE_ROLE.role

    check_flag = not skip_flag_checking

    if check_flag:
        if get_flag('maintenance', db_type):
            log.error('Идет работа с базой данных, нельзя делать дамп')
            return 1

        set_flag('maintenance', job.MAKE_DUMP.flag_value, db_type)

    fileformat = DUMP_FORMAT
    if schema_only:
        fileformat = SCHEMA_FORMAT

    db_name = switcher.get_db_alias(db_type)
    fname = fileformat.format(
        db_type=db_type,
        timestamp=environment.now(),
        prefix=prefix,
        db_name=db_name,
    )

    dump_path = os.path.join(export_folder, fname)

    log.info('Сохраняем дамп %s', dump_path)

    dump_db_by_alias(db_type, dump_path, schema_only=schema_only)

    if schema_only:
        upload_mysql_dump_schema(dump_path, export_folder)
    else:
        attrs = {
            'original_name': dump_path,
            'environment': settings.APPLIED_CONFIG,
            'db_type': db_type
        }
        file_wrapper = get_wrapper_creator(FileType.MYSQL_DUMP, attrs=attrs).get_file_wrapper(dump_path)
        file_wrapper.upload()

    if check_flag:
        set_flag('maintenance', False, db_type)

    if save_filename:
        maintenance.update_conf({'{}_dump'.format(db_type): fname})


def upload_mysql_dump_schema(file_name, export_folder):
    target = os.path.join(export_folder, 'schema_dump.gz')
    copyfile(file_name, target)

    attrs = {
        'original_name': file_name,
        'environment': settings.APPLIED_CONFIG
    }
    file_wrapper = get_wrapper_creator(FileType.MYSQL_SCHEMA, attrs=attrs).get_file_wrapper(target)
    file_wrapper.upload()


if __name__ == '__main__':
    with ylog_context(**get_script_log_context()), task_progress_report('make_fresh_dump'):
        create_current_file_run_log()

        import argparse

        parser = argparse.ArgumentParser()

        parser.add_argument('-v', '--verbose', action='store_true')
        parser.add_argument('-t', '--db-type', choices=('work_db', 'service_db'))
        parser.add_argument('-p', '--prefix')
        parser.add_argument('--skip-flag-checking', action='store_true')
        parser.add_argument('--save-filename', action='store_true')
        parser.add_argument('--schema-only', action='store_true')
        parser.add_argument('export_folder', nargs='?', default=settings.EXPORT_PATH)

        args = parser.parse_args()

        if args.verbose:
            print_log_to_stdout(log)

        sys.exit(run(
            export_folder=args.export_folder,
            db_type=args.db_type, prefix=args.prefix,
            skip_flag_checking=args.skip_flag_checking,
            save_filename=args.save_filename,
            schema_only=args.schema_only
        ) or 0)
