#!/usr/bin/env python
# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import travel.rasp.admin.scripts.load_project  # noqa

from datetime import timedelta

from django.conf import settings
from django.core.mail import send_mail
from django.core.urlresolvers import reverse

from travel.rasp.library.python.common23.date import environment
from common.utils.date import uni_strftime
from travel.rasp.admin.importinfo.models.two_stage_import import TwoStageImportPackage
from travel.rasp.admin.lib.logs import get_script_log_context, ylog_context


SENDER = "Яндекс.Расписания <rasp-info@yandex-team.ru>"

SUBJECT = "Срок размещения расписания на Яндекс.Расписаниях истекает {human_end_date}"
MESSAGE = ""
HTML_MESSAGE = ("Здравствуйте. {human_end_date} истекает срок размещения Ваших данных на Я.Расписаниях."
                " Если Вы хотите продолжить сотрудничество с нашим сервисом,"
                " то Вам необходимо прислать на адрес rasp-info@yandex-team.ru"
                " или ответственному контент-менеджеру обновленный файл в форматах .yarm или .xml. <br />"
                " <b>В файле должны быть проставлены корректные дни хождения (максимум на 3 месяца вперед, начиная с текущей даты).</b> <br />"
                " Если у Вас нет актуального файла, то Вы можете запросить его у нас, а после этого отправить скорректированный вариант. <br />"
                " <b>Сообщения \"изменений нет\" и подобные им без данных в файле в дальнейшем рассматриваться не будут.</b> <br />"
                "__ <br />"
                "Как прислать расписание: <br />"
                "https://rasp.yandex.ru/info/format/")


SUBJECT_FOR_CONTENT_MANAGER = ("{supplier_description} Срок размещения расписания"
                               " на Яндекс.Расписаниях истекает {human_end_date}")
MESSAGE_FOR_CONTENT_MANAGER = ("Напоминание об истечении срока размещения на сервисе отправлено поставщику."
                               " {supplier_description}. Дата импорта {import_date},"
                               " дата протухания {human_end_date}.")

NUMBER_OF_REMAINING_DAYS_TO_START_SEND_EMAILS = 15
ADMIN_WORK_URL = "https://{}".format(settings.PROD_WORK_URL)
ADMIN_SERVICE_URL = "https://{}".format(settings.PROD_SERVICE_URL)


def send_notifications():
    today = environment.today()

    for package in TwoStageImportPackage.objects.filter(notify_supplier=True):
        end_date = get_package_end_date(package)
        if not need_to_send_email(package, today, end_date):
            continue

        human_end_date = uni_strftime("%d %B %Y", end_date)
        subject = SUBJECT.format(human_end_date=human_end_date)

        html_message = HTML_MESSAGE.format(human_end_date=human_end_date)

        subject_for_content_manager = SUBJECT_FOR_CONTENT_MANAGER.format(
            human_end_date=human_end_date,
            supplier_description=get_supplier_description(package, short=True)
        )
        message_for_content_manager = MESSAGE_FOR_CONTENT_MANAGER.format(
            supplier_description=get_supplier_description(package),
            import_date=(uni_strftime("%d %B %Y", package.last_import_date)
                         if package.last_import_date else "неизвестна"),
            human_end_date=human_end_date,
        )

        if settings.SEND_MAIL_TO_PARTNERS:
            send_mail(subject, MESSAGE, SENDER, [package.supplier_email], html_message=html_message)
            if need_to_send_email_to_content_manager(package):
                send_mail(subject_for_content_manager, message_for_content_manager,
                          SENDER, [package.content_manager.email])
        else:
            for __, email in settings.ADMINS:
                subject = "Test: " + subject
                subject_for_content_manager = "Test: " + subject_for_content_manager
                send_mail(subject, MESSAGE, SENDER, [email], html_message=html_message)
                send_mail(subject_for_content_manager, message_for_content_manager, SENDER, [email])


def need_to_send_email(package, today, end_date):
    if not package.notify_supplier:
        return False

    if not package.supplier_email:
        return False

    if not package.last_import_date:
        return False

    if not end_date:
        return False

    return end_date - today < timedelta(days=NUMBER_OF_REMAINING_DAYS_TO_START_SEND_EMAILS)


def get_package_end_date(package):
    if package.last_mask_date:
        return package.last_mask_date

    if not package.last_import_date:
        return

    return package.last_import_date + timedelta(days=package.tsisetting.max_forward_days)


def need_to_send_email_to_content_manager(package):
    return package.notify_content_manager


def get_supplier_description(package, short=False):
    if short:
        return "Пакет {}, поставщик {}.".format(package.title, package.supplier.title)

    relative_url = reverse("admin:importinfo_twostageimportpackage_change", args=[package.id])
    work_url = ADMIN_WORK_URL + relative_url
    service_url = ADMIN_SERVICE_URL + relative_url
    return "Пакет {}, поставщик {}, ссылка на пакет в рабочей базе {} ссылка на пакет в сервисной базе {}".format(
        package.title, package.supplier.title, work_url, service_url
    )


if __name__ == "__main__":
    with ylog_context(**get_script_log_context()):
        send_notifications()
