# coding: utf-8
# Скрипт собирает файлы данных, полученных по WSDL, в архив
# и перемещает их в директорию с логами.

from __future__ import unicode_literals

import travel.rasp.admin.scripts.load_project  # noqa

import logging
import os.path
import shutil
import tarfile
from datetime import datetime, timedelta

from django.utils.encoding import force_bytes
from django.conf import settings

from travel.rasp.library.python.common23.date import environment
from common.utils.metrics import task_progress_report
from travel.rasp.admin.lib import tmpfiles
from travel.rasp.admin.lib.logs import create_current_file_run_log, print_log_to_stdout, get_script_log_context, ylog_context
from travel.rasp.admin.scripts.utils.import_file_storage import (parse_date, get_stored_supplier_codes, get_all_temporary_dirs_by_dt,
                                               get_type_path)


log = logging.getLogger(__name__)


LEFT_TABLO_DAYS = 1
TAR_FORMAT = '%Y%m%d.tar.gz'


class OldFilesCleaner(object):
    def __init__(self):
        self.today = environment.now()

        # Удаляем файлы и архивы старше месяца
        self.tablo_point_of_no_return_dt = self.today - timedelta(days=30)
        self.schedule_point_of_no_return_dt = self.today - timedelta(days=30)
        self.archive_path = os.path.join(settings.LOG_PATH, 'archive/tablo')

    def clean(self):
        log.info('Start cleaning')

        self.prepare()

        self.compress_old_tablo_data()
        self.clean_old_tablo()
        self.clean_old_schedule()

        log.info('Done')

    def prepare(self):
        if not os.path.exists(self.archive_path):
            os.makedirs(self.archive_path)

    @tmpfiles.clean_temp
    def compress_old_tablo_data(self):
        today = environment.today()
        to_compress_date = today - timedelta(LEFT_TABLO_DAYS)
        type_path = get_type_path('tablo')

        dirs = []

        for supplier_code in get_stored_supplier_codes('tablo'):
            for dt, path in get_all_temporary_dirs_by_dt('tablo', supplier_code).items():
                dirs.append((dt, path))

        to_compress_by_date = {}
        for dt, path in dirs:
            if dt.date() <= to_compress_date:
                to_compress_by_date.setdefault(dt.date(), []).append(path)

        for date_, files in to_compress_by_date.items():
            if not files:
                break

            tarfilepath = os.path.join(self.archive_path, date_.strftime(TAR_FORMAT))
            log.info('Создаем архив %s', tarfilepath)

            old_archive = None
            if os.path.exists(tarfilepath):
                log.info('Архив уже был создан, пополняем его данными')
                old_archive = tmpfiles.get_tmp_filepath('archive.tar.gz')
                shutil.copy(tarfilepath, old_archive)

            with tarfile.open(tarfilepath, 'w:gz') as new_tar:
                if old_archive:
                    log.info('Копируем данные из предыдущего архива в пустой новый архив')
                    with tarfile.open(old_archive) as old_tar:
                        for tarinfo in old_tar.getmembers():
                            if tarinfo.isfile():
                                fobj = old_tar.extractfile(tarinfo)
                                new_tar.addfile(tarinfo, fobj)
                            else:
                                new_tar.addfile(tarinfo)

                for path in files:
                    log.info('Добавляем в архив %s и удаляем с диска', os.path.relpath(path, type_path))
                    new_tar.add(force_bytes(path), force_bytes(os.path.relpath(path, type_path)))
                    shutil.rmtree(force_bytes(path), ignore_errors=False)

                new_tar.close()

    def clean_old_tablo(self):
        def get_date_from_tarfile(filepath):
            try:
                return datetime.strptime(os.path.basename(filepath), TAR_FORMAT)
            except ValueError:
                return datetime.fromtimestamp(os.stat(filepath).st_mtime)

        for tarfile_name in os.listdir(self.archive_path):
            tarfilepath = os.path.join(self.archive_path, tarfile_name)
            dt = get_date_from_tarfile(tarfilepath)
            if dt is not None and dt <= self.tablo_point_of_no_return_dt:
                log.info('Удаляем старых архив %s', tarfilepath)
                os.remove(tarfilepath)

    def clean_old_schedule(self):
        temporary_dir = get_type_path('schedule')
        if os.path.exists(temporary_dir):
            for supplier_dirname in os.listdir(temporary_dir):
                supplier_dir = os.path.join(temporary_dir, supplier_dirname)
                if not os.path.isdir(supplier_dir):
                    continue

                for package_id_or_date_subdir in os.listdir(supplier_dir):
                    subdir_date = parse_date(package_id_or_date_subdir)
                    if subdir_date:
                        self.check_and_delete(supplier_dir, package_id_or_date_subdir, subdir_date)
                    else:
                        package_data_path = os.path.join(supplier_dir, package_id_or_date_subdir)
                        if not os.path.isdir(package_data_path):
                            continue

                        for subdir in os.listdir(package_data_path):
                            subdir_date = parse_date(subdir)
                            self.check_and_delete(package_data_path, subdir, subdir_date)

    def check_and_delete(self, base_dir, subdir, subdir_date):
        dirpath = os.path.join(base_dir, subdir)
        if subdir_date is not None and subdir_date <= self.schedule_point_of_no_return_dt:
            log.info('Удаляем файлы расписаний %s', dirpath)
            shutil.rmtree(force_bytes(dirpath), ignore_errors=True)


if __name__ == '__main__':
    with ylog_context(**get_script_log_context()), task_progress_report('rotate_temporary_data'):
        import argparse

        create_current_file_run_log()

        parser = argparse.ArgumentParser()
        parser.add_argument('-v', '--verbose', action='store_true')
        args = parser.parse_args()
        if args.verbose:
            print_log_to_stdout()

        OldFilesCleaner().clean()
