#!/usr/bin/env python
# coding: utf-8

import travel.rasp.admin.scripts.load_project  # noqa

import logging
import subprocess
from optparse import OptionParser

from django.utils.encoding import smart_unicode
from django.conf import settings

from travel.rasp.admin.lib.logs import print_log_to_stdout, create_current_file_run_log


log = logging.getLogger(__name__)


IO_TIMEOUT = 600
CONNECTION_TIMEOUT = 10


def rsync(config):
    log.info(u"Запускаем rsync для конфига %s", config)

    processes = []

    try:
        sync_list, hosts = settings.RSYNC_CONFIGS[config]
    except KeyError:
        log.error(u"Такого конфига %s для rsync нет", config)
        return

    options = ['-aL', '--delay-updates', '--delete-after', '--delete-excluded']
    options += ['--timeout={}'.format(IO_TIMEOUT), '--contimeout={}'.format(CONNECTION_TIMEOUT)]

    for share, path, exclude in sync_list:
        command = [settings.RSYNC] + options + [path]

        if exclude:
            for path in exclude:
                command.append('--exclude=' + path)

        for host in hosts:
            target = "rsync://%s/%s" % (host, share)
            args = command + [target]
            log.info(u' '.join(args))
            process = subprocess.Popen(args, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
            processes.append(process)

    has_errors = False
    has_notices = False

    # Ждем завершение процессов
    for process in processes:
        process.wait()

        stderr = process.stderr.read()
        stdout = process.stdout.read()

        if stderr:
            log.error(u"stderr:\n%s", smart_unicode(stderr))
            has_errors = True

        if stdout:
            log.info(u"stdout:\n%s", smart_unicode(stdout))
            has_notices = True

    if has_errors:
        log.error(u"Запуск rsync %s прошел с ошибками", config)
    elif has_notices:
        log.warning(u"Запуск rsync %s прошел с замечаниями", config)
    else:
        log.info(u"Запуск rsync %s прошел успешно", config)


usage = u"Usage: python %prog [options] [configs]"


if __name__ == '__main__':
    create_current_file_run_log()

    optparser = OptionParser(usage=usage, description=__doc__)
    optparser.add_option('-v', '--verbose', action="store_true",
                         help=u"выводить лог на экран")

    options, configs = optparser.parse_args()

    if options.verbose:
        print_log_to_stdout()

    for config in configs:
        rsync(config)
