# coding: utf8

from __future__ import absolute_import, unicode_literals

import string
import logging
from time import strptime
from datetime import date

from common.models.geo import Country
from common.models.schedule import RunMask
from travel.rasp.library.python.common23.date import environment
from travel.rasp.admin.lib.import_bounds import import_bounds
from travel.rasp.admin.lib.mask_builder.bounds import MaskBounds
from travel.rasp.admin.lib.mask_builder.standard_builders import (mask_from_day_condition, daily_mask, empty_mask, even_mask, odd_mask,
                                                mask_from_days_of_week)
from travel.rasp.admin.lib.mask_builder.ycal_builders import ycal_workdays_mask, ycal_weekends_and_holidays_mask
from travel.rasp.admin.scripts.schedule.af_processors.parse_utils import get_yesno_param, get_date_param


log = logging.getLogger(__name__)


def make_runmask(thread_el, bounds, country, today):
    today = today or environment.today()

    include_masks = []
    exclude_masks = []

    include_masks.append(get_dates_mask(thread_el, bounds, today))
    include_masks.append(get_daily_mask(thread_el, bounds, today))
    include_masks.append(get_even_mask(thread_el, bounds, today))
    include_masks.append(get_odd_mask(thread_el, bounds, today))
    include_masks.append(get_days_of_week_mask(thread_el, bounds, today))
    include_masks.append(get_workdays_mask(thread_el, bounds, country, today))
    include_masks.append(get_weekends_mask(thread_el, bounds, country, today))

    exclude_masks.append(get_noweekends_mask(thread_el, bounds, country, today))

    runmask = RunMask(today=today)

    for mask in include_masks:
        runmask |= mask

    for mask in exclude_masks:
        runmask = runmask.difference(mask)

    return runmask


def get_dates_mask(thread_el, bounds, today):
    strdays = filter(None, map(string.strip, thread_el.get('dates', '').split(';')))
    dates = [date(*strptime(strday, '%Y-%m-%d')[:3]) for strday in strdays]

    def day_condition(day):
        return day in dates

    return mask_from_day_condition(bounds, day_condition, today)


def get_daily_mask(thread_el, bounds, today):
    daily = get_yesno_param(thread_el, 'daily')

    if daily:
        return daily_mask(bounds, today)
    else:
        return empty_mask(today)


def get_even_mask(thread_el, bounds, today):
    even = get_yesno_param(thread_el, 'even')

    if even:
        return even_mask(bounds, today)
    else:
        return empty_mask(today)


def get_odd_mask(thread_el, bounds, today):
    odd = get_yesno_param(thread_el, 'odd')

    if odd:
        return odd_mask(bounds, today)
    else:
        return empty_mask(today)


def get_workdays_mask(thread_el, bounds, country, today):
    workdays = get_yesno_param(thread_el, 'workdays')

    if workdays:
        return ycal_workdays_mask(bounds, country, today)
    else:
        return empty_mask(today)


def get_weekends_mask(thread_el, bounds, country, today):
    weekend = get_yesno_param(thread_el, 'weekend')

    if weekend:
        return ycal_weekends_and_holidays_mask(bounds, country, today)
    else:
        return empty_mask(today)


def get_noweekends_mask(thread_el, bounds, country, today):
    noweekend = get_yesno_param(thread_el, 'noweekend')

    if noweekend:
        return ycal_weekends_and_holidays_mask(bounds, country, today)
    else:
        return empty_mask(today)


def get_days_of_week_mask(thread_el, bounds, today):
    # 0, 7 - Sunday
    days = thread_el.get('days')
    if days:
        days = days.replace('0', '7')
        return mask_from_days_of_week(bounds, days, today)
    else:
        return empty_mask(today)


def compute_bounds(thread_el, schedule_plan=None):
    first_date, last_date = import_bounds()

    period_start = get_date_param(thread_el, 'period_start')
    period_end = get_date_param(thread_el, 'period_end')

    # Даты указанные Сашей, важнее чем даты в графике хождения поездов
    if schedule_plan:
        graph_start_date = schedule_plan.start_date
        graph_end_date = schedule_plan.end_date
    else:
        graph_start_date = None
        graph_end_date = None

    period_start = period_start or graph_start_date or first_date
    period_end = period_end or graph_end_date or last_date
    period_start = period_start >= first_date and period_start or first_date
    period_end = period_end <= last_date and period_end or last_date

    return MaskBounds(period_start, period_end)


def get_calendar_country(thread_el):
    calendar_geobase_country_id = thread_el.get('calendar_geobase_country_id')

    if calendar_geobase_country_id:
        return Country.objects.get(_geo_id=calendar_geobase_country_id)

