# -*- coding: utf-8 -*-
from __future__ import absolute_import

import logging

from django.db import transaction
from django.db import IntegrityError

from common.models.schedule import Route, RThreadType, RThread
from travel.rasp.admin.scripts.schedule.utils.route_loader import CompactThreadNumberBuilder
from travel.rasp.admin.scripts.support_methods import full_create_thread, full_create_route


log = logging.getLogger(__name__)

basic_type = RThreadType.objects.get(code='basic')


@transaction.atomic
def process_thread(thread, filename):
    if thread.changemode == u"insert":
        insert_thread(thread, filename)
    elif thread.changemode == u"replace":
        replace_thread(thread, filename)
    else:
        log.error(u"Неизвестный changemode '%s'", thread.changemode)


def insert_thread(thread, filename):
    route = make_route(thread)
    check_route(route, filename)
    try:
        route = full_create_route(route, log, True)
    except IntegrityError:
        log.info(u"Маршрут %s уже добавлен", route.route_uid)
        added_route = Route.objects.get(route_uid=route.route_uid)
        thread_number_builder = CompactThreadNumberBuilder([added_route])
        for thread in route.threads:
            thread_number_builder.build_for(thread)
            thread.route = added_route
            try:
                full_create_thread(thread, log, True)
            except IntegrityError:
                log.warning(u"Нитка %s уже добавлена", thread.uid)
        route = added_route

    return route


def replace_thread(thread, filename):
    # Оставляем в нитке все тоже самое но заменяем время старта и станции
    try:
        old_thread = RThread.objects.get(uid=thread.uid, type=basic_type)
    except RThread.DoesNotExist:
        log.error(u"Не нашли основную нитку с uid %s", thread.uid)
        return

    route = old_thread.route
    log.info(u"Удаляем нитку %s", old_thread.uid)
    thread.year_days = old_thread.year_days
    thread.template_text = old_thread.template_text
    old_thread.delete()

    thread.type = basic_type
    thread.route = route
    thread.changed = True
    thread.uid = None
    CompactThreadNumberBuilder([route]).build_for(thread)
    full_create_thread(thread, log, True)

    log.info(u"Заменили нитку %s", thread.uid)
    return thread.route


def make_route(thread):
    route = Route()
    thread.route = route
    route.threads = [thread]

    route.supplier = thread.supplier
    route.t_type = thread.t_type
    route.route_uid = thread.gen_route_uid(use_start_station=True)

    return route


def check_route(route, filename):
    if not route.threads:
        log.error(u"Ни одной нитки у маршрута %s", route.route_uid)
        return False

    for thread in route.threads:
        if not thread.rtstations:
            log.error(u"Ни одной станции у маршрута %s", thread.number)
            return False

        if thread.rtstations[-1].tz_arrival > 300:
            log.warning(u"Время в пути больше 300 минут %s %s",
                        thread.number, filename)
    return True
