# coding: utf-8

from __future__ import absolute_import, unicode_literals


import logging
import operator
from functools import reduce

from django.db import transaction
from django.utils import six

from common.apps.facility.models import SuburbanThreadFacility
from common.models.schedule import RThread
from travel.rasp.library.python.common23.date import environment
from common.utils.date import RunMask
from travel.rasp.admin.lib.mask_builder.ycal_builders import CountryRequiredError
from travel.rasp.admin.scripts.schedule.af_processors.af_multiparam_mask_parser import get_calendar_country
from travel.rasp.admin.scripts.schedule.af_processors.suburban.affected_threads_finder import set_mask, AffectedThreadsByBasicThreadBuilder
from travel.rasp.admin.scripts.schedule.af_processors.suburban.exceptions import AfSuburbanProcessorError
from travel.rasp.admin.scripts.schedule.af_processors.suburban.utils.facility import build_pre_facilities, clone_facilities_to_change_thread, check_intersection


log = logging.getLogger(__name__)


@transaction.atomic
def apply_changemode_change_facilities(thread, thread_el, today, **kwargs):
    """
    Обновляем удобства нитки, затем клонируем их в нитки изменения.
    """

    today = today or environment.today()

    if thread.uid:
        try:
            threads = [RThread.objects.get(uid=thread.uid)]
        except RThread.DoesNotExist:
            log.error('Не нашли нитку по uid %s', thread.uid)
            return
        set_mask(threads[0], today)
    else:
        threads = AffectedThreadsByBasicThreadBuilder(thread, thread_el, today).get_basic_threads()

    country = get_calendar_country(thread_el)

    for thread in threads:
        log.info('Обновляем удобства у нитки %s', thread.uid)
        try:
            pre_facilities = build_pre_facilities(thread_el, thread.schedule_plan, country, today)
        except CountryRequiredError:
            raise AfSuburbanProcessorError('Необходимо указать calendar_geobase_country_id в нитке,'
                                           ' для построения масок удобств')

        check_intersection(thread, pre_facilities, today)
        _update_facilities(thread, pre_facilities, today)


def _clear_old_facilities(thread, today, total_new_mask):
    for old_thread_facility in thread.suburbanthreadfacility_set.all():
        mask = RunMask(old_thread_facility.year_days, today=today)
        mask -= total_new_mask
        if mask:
            old_thread_facility.year_days = six.text_type(mask)
            old_thread_facility.save()
        else:
            old_thread_facility.delete()


@transaction.atomic
def _update_facilities(thread, pre_facilities, today):
    total_new_mask = reduce(operator.or_, [pf.mask for pf in pre_facilities.values()])

    _clear_old_facilities(thread, today, total_new_mask)

    old_thread_facilities = {
        frozenset(tf.facilities.all()): tf
        for tf in thread.suburbanthreadfacility_set.prefetch_related('facilities')
    }

    for pre_facility in pre_facilities.values():
        if pre_facility.facilities and pre_facility.mask:
            key = frozenset(pre_facility.facilities)

            if key in old_thread_facilities:
                old_thread_facility = old_thread_facilities[key]
                new_mask = RunMask(old_thread_facility.year_days, today=today) | pre_facility.mask
                old_thread_facility.year_days = six.text_type(new_mask)
                old_thread_facility.save()
            else:
                thread_facility = SuburbanThreadFacility.objects.create(thread=thread, year_days=pre_facility.mask)
                for facility in pre_facility.facilities:
                    thread_facility.facilities.add(facility)

    for change_thread in thread.thread_changes.all():
        clone_facilities_to_change_thread(thread, change_thread)
