# coding: utf-8

from __future__ import unicode_literals

import logging
import operator
from functools import reduce

from django.db import transaction
from django.utils import six

from common.apps.facility.models import SuburbanFacility, SuburbanThreadFacility
from travel.rasp.admin.lib.mask_builder.afmask_builders import AfMaskBuilder
from travel.rasp.admin.scripts.schedule.af_processors.af_multiparam_mask_parser import compute_bounds, make_runmask
from travel.rasp.admin.scripts.schedule.af_processors.suburban.exceptions import AfSuburbanProcessorError


log = logging.getLogger(__name__)


class PreSuburbanThreadFacility(object):
    def __init__(self, mask, facilities):
        """
        :type mask: RunMask
        :type facilities: list[SuburbanFacility]
        """
        self.mask = mask
        self.facilities = facilities


def build_pre_facilities(thread_el, schedule_plan, country, today):
    pre_facilities = {}

    af_mask_builder = AfMaskBuilder()
    for facility_el in thread_el.findall('./facilities/facility_period'):
        bounds = compute_bounds(facility_el, schedule_plan)
        weektemplate = facility_el.get('weektemplate', '').strip()
        if weektemplate:
            mask = af_mask_builder.build(bounds, weektemplate, country, today)
        else:
            mask = make_runmask(facility_el, bounds, country, today)

        try:
            facilities = [
                SuburbanFacility.objects.get(code=c.strip()) for c in facility_el.get('facilities').split(',')
            ]
        except SuburbanFacility.DoesNotExist:
            raise AfSuburbanProcessorError('Не нашли некоторых удобств в базе "{}"'.format(
                facility_el.get('facilities'))
            )

        key = frozenset(facilities)
        if key in pre_facilities:
            pre_facilities[key].mask |= mask
        else:
            pre_facilities[key] = PreSuburbanThreadFacility(mask, facilities)

    return pre_facilities


@transaction.atomic
def clone_facilities_to_change_thread(thread, change_thread):
    change_thread.suburbanthreadfacility_set.all().delete()
    for thread_facility in thread.suburbanthreadfacility_set.all().prefetch_related('facilities'):
        change_thread_facility = SuburbanThreadFacility.objects.create(thread=change_thread,
                                                                       year_days=thread_facility.year_days)
        change_thread_facility.facilities = list(thread_facility.facilities.all())


CHECK_INTERSECTION_WARNING = 'Удобства содержат дни, которых нет в основной нитке или в нитках изменениях: %s'


def check_intersection(thread, pre_facilities, today):
    if not pre_facilities:
        return
    pre_facilities_total_mask = reduce(operator.or_, [pf.mask for pf in pre_facilities.values()])
    thread_total_mask = reduce(
        operator.or_, [thread.get_mask(today)] + [tc.get_mask(today) for tc in thread.thread_changes.all()]
    )

    diff_days = (pre_facilities_total_mask - thread_total_mask).dates()

    if diff_days:
        log.warning(CHECK_INTERSECTION_WARNING, ', '.join(six.text_type(dd) for dd in diff_days))
