# coding: utf-8

import logging

import requests
from django.conf import settings
from django.utils.translation import ugettext_noop as N_, ugettext_lazy as _
from lxml import etree

from common.cysix.builder import ChannelBlock, GroupBlock, ThreadBlock, StoppointBlock, ScheduleBlock
from common.models.geo import Country
from common.utils.caching import cache_method_result
from cysix.tsi_converter import CysixTSIConverterFactory, CysixTSIConverterFileProvider
from travel.rasp.admin.lib.geocoder import geocode_to_rasp_geo_objects
from travel.rasp.admin.scripts.schedule.utils import RaspImportError
from travel.rasp.admin.scripts.schedule.utils.file_providers import PackageFileProvider, XmlPackageFileProvider


log = logging.getLogger(__name__)


URL = u'https://allticketsfor.me/proxy/public/application/yandex/export'

USER = '@atfm'
PASSWORD = '#32167'


class AllticketsforMeCysixFactory(CysixTSIConverterFactory):
    def get_raw_download_file_provider(self):
        return AllticketsforMeRawDownloadFileProvider(self.package)

    def get_raw_package_file_provider(self):
        return XmlPackageFileProvider(self.package)

    def get_converter_file_provider(self, raw_file_provider):
        return AllticketsforMeCysixFileProvider(self.package, raw_file_provider)


class AllticketsforMeCysixFileProvider(CysixTSIConverterFileProvider):
    def convert_data(self, filepath):
        c = Converter(self.raw_file_provider)

        c.convert(filepath)


class ConvertError(RaspImportError):
    pass


class Converter(object):
    """
    Так как они не полностью соблюдают соглашения общего, то делаем предположение и проверку,
    что группы, временные зоны, и т.п. не меняются.
    """

    def __init__(self, provider):
        self.provider = provider

        self.stations = dict()

    def convert(self, filepath):
        schedule_xml_file = self.provider.get_schedule_files()[0]
        channel_el = etree.parse(schedule_xml_file).getroot()
        self.check_tree(channel_el)

        cysix_channel_el = self.build_channel_element(channel_el)

        with open(filepath, 'w') as f:
            f.write(etree.tostring(cysix_channel_el, xml_declaration=True, encoding='utf-8',
                    pretty_print=True))

    def assert_true(self, expr, message):
        if not expr:
            raise ConvertError(message)

    def check_tree(self, channel_el):
        error_message = _(u'Неожиданные значения в channel')

        self.assert_true(channel_el.get('timezone') == 'local', error_message)
        self.assert_true(channel_el.get('t_type') == 'train', error_message)
        self.assert_true(channel_el.get('carrier_code_system') == 'vendor', error_message)
        self.assert_true(channel_el.get('city_code_system') == 'vendor', error_message)
        self.assert_true(channel_el.get('vehicle_code_system') == 'vendor', error_message)

    def build_channel_element(self, channel_el):
        channel_block = ChannelBlock(
            'train', station_code_system='vendor',
            carrier_code_system='vendor',
            vehicle_code_system='vendor',
            timezone='local'
        )

        group_el = self.get_group_el(channel_el)

        group_block = self.build_group(channel_block, group_el)

        channel_block.add_group_block(group_block)

        return channel_block.get_element()

    def get_group_el(self, channel_el):
        group_els = channel_el.findall('./group')
        if len(group_els) != 1:
            raise ConvertError(_(u'Расчитываем на то что, у них всегда одна группа'))

        error_message = _(u'Неожиданные значения в group')

        group_el = group_els[0]

        self.assert_true(group_el.get('timezone') == 'local', error_message)
        self.assert_true(group_el.get('code') == 'example', error_message)
        self.assert_true(group_el.get('title') == u'Расписание', error_message)

        return group_el

    def build_group(self, channel_block, group_el):
        context = {}
        context['cities'] = self.extract_cities(group_el)
        context['terminals'] = self.extract_terminals(group_el)
        context['carriers'] = self.extract_carriers(group_el)
        context['vehicles'] = self.extract_vehicles(group_el)
        context['fares'] = self.extract_fares(group_el)

        group_block = GroupBlock(channel_block, u'Расписание', u'example')
        group_block.context = context

        self.build_threads(group_el, group_block)

        return group_block

    def extract_cities(self, group_el):
        cities = {}
        for city_el in group_el.findall('./cities/city'):
            cities[city_el.get('code')] = city_el.attrib

        return cities

    def extract_terminals(self, group_el):
        terminals = {}
        for terminal_el in group_el.findall('./terminals/terminal'):
            terminals[terminal_el.get('code')] = terminal_el.attrib

        return terminals

    def extract_carriers(self, group_el):
        carriers = {}
        for carrier_el in group_el.findall('./carriers/carrier'):
            carriers[carrier_el.get('code')] = carrier_el.attrib

        return carriers

    def extract_vehicles(self, group_el):
        vehicles = {}
        for vehicle_el in group_el.findall('./vehicles/vehicle'):
            vehicles[vehicle_el.get('code')] = vehicle_el.attrib

        return vehicles

    def extract_fares(self, group_el):
        fares = {}
        for fare_el in group_el.findall('./fares/fare'):
            prices = list()
            for price_el in fare_el.findall('./price'):
                city_code_from = price_el.find('./stop_from').get('city_code')
                city_code_to = price_el.find('./stop_to').get('city_code')

                prices.append(
                    ((city_code_from, city_code_to), price_el)
                )

            fares[fare_el.get('code')] = {
                'prices': prices,
                'fare_el': fare_el
            }

        return fares

    def build_threads(self, group_el, group_block):
        for thread_el in group_el.findall('./threads/thread'):
            self.build_thread(thread_el, group_block)

    def build_thread(self, thread_el, group_block):
        thread_block = ThreadBlock(
            group_block,
            title=thread_el.get('title'),
            number=thread_el.get('number'),
        )

        thread_block.vehicle = group_block.add_vehicle(
            thread_el.get('vehicle_title'), thread_el.get('vehicle_code')
        )
        thread_block.carrier = group_block.add_carrier(
            thread_el.get('carrier_title'), thread_el.get('carrier_code')
        )

        self.build_stoppoints(thread_el, thread_block, group_block)
        self.build_fare(thread_el, thread_block, group_block)

        self.build_schedules(thread_el, thread_block, group_block)

        self.build_raw_data(thread_el, thread_block, group_block)

        group_block.add_thread_block(thread_block)

    def build_stoppoints(self, thread_el, thread_block, group_block):
        thread_els = thread_el.findall('./stoppoints/stoppoint')
        for index, stoppoint_el in enumerate(thread_els):
            is_middle_station = 0 < index < (len(thread_els) - 1)

            station_block = self.get_station(stoppoint_el, group_block)

            stoppoint = StoppointBlock(
                thread_block, station_block,
                arrival_time=stoppoint_el.get('arrival_time'),
                departure_time=stoppoint_el.get('departure_time'),
                in_station_schedule='0'
            )

            if is_middle_station:
                if stoppoint_el.get('is_transfer') != u'1':
                    log.warning(N_(u'У рейса %s промежуточная станция не является пересадкой'),
                                thread_el.get('title'))

                stoppoint.is_combined = u'1'
                stoppoint.is_searchable_to = u'0'
                stoppoint.is_searchable_from = u'0'

            thread_block.add_stoppoint_block(stoppoint)

        return thread_block.stoppoints

    def build_schedules(self, thread_el, thread_block, group_block):
        expected_attrs = ('times', 'days')

        for schedule_el in thread_el.findall('./schedules/schedule'):
            for key in schedule_el.attrib:
                if key not in expected_attrs:
                    raise ConvertError(_(u'Неожиданный аттрибут в <schedule> %s'), key)

            thread_block.add_schedule_block(ScheduleBlock(
                thread_block, **schedule_el.attrib
            ))

    def build_fare(self, thread_el, thread_block, group_block):
        fare_info = group_block.context['fares'].get(thread_el.get('fare_code'))
        if not fare_info:
            return

        prices = fare_info['prices']

        fare_block = group_block.add_local_fare()

        def get_station_block(city_code):
            for stoppoint_block in thread_block.stoppoints:
                if stoppoint_block.station.city_code == city_code:
                    return stoppoint_block.station

        min_prices = {}

        common_currency = None

        for (city_code_from, city_code_to), price_el in prices:
            station_from = get_station_block(city_code_from)
            station_to = get_station_block(city_code_to)

            if station_from is None or station_to is None:
                log.error(N_(u'Не получилось прообразовать цены для %s %s:'
                             u' не нашли станцию'),
                          thread_el.get('title'), thread_el.get('number'))
                continue

            price = price_el.get('price')
            if not price:
                log.info(N_(u'Пустая цена в fare для %s %s'),
                         thread_el.get('title'), thread_el.get('number'))
                continue

            try:
                price = float(price)
            except (TypeError, ValueError) as e:
                log.error(N_(u'Не получилось прообразовать цены для %s %s:'
                             u' ошибка получения цены %s %s'),
                          thread_el.get('title'), thread_el.get('number'),
                          price, e)
                continue

            currency = price_el.get('currency')
            if not common_currency:
                common_currency = currency
            if common_currency != currency:
                log.error(N_(u'Не получилось прообразовать цены для %s %s:'
                             u' все валюты в одном fare должны совпадать'),
                          thread_el.get('title'), thread_el.get('number'))
                return

            other_price = min_prices.get((station_from, station_to))
            if other_price is None or price < other_price:
                min_prices[station_from, station_to] = price

        for (station_from, station_to), price in min_prices.items():
            fare_block.add_price_block(
                u'{:.2f}'.format(price), common_currency,
                station_from, station_to,
                is_min_price=u'1'
            )

        thread_block.set_fare_block(fare_block)

    def build_raw_data(self, thread_el, thread_block, group_block):
        data = etree.tounicode(thread_el, pretty_print=True)
        fare_info = group_block.context['fares'].get(thread_el.get('fare_code'))
        if fare_info:
            data += u'\n' + etree.tounicode(fare_info['fare_el'], pretty_print=True)

        thread_block.set_raw_data(data)

    def get_station(self, stoppoint_el, group_block):
        city_code = stoppoint_el.get('city_code')
        city_title = stoppoint_el.get('city_title')
        terminal_code = stoppoint_el.get('terminal_code')
        terminal_name = stoppoint_el.get('terminal_name')

        # Почемуто есть такие терминалы
        if city_code != u'0' and city_code in group_block.context['cities']:
            city = group_block.context['cities'][city_code]
            country_code = city['country_code']
        elif terminal_code in group_block.context['terminals']:
            terminal = group_block.context['terminals'][terminal_code]
            country_code = terminal['country_code']
        else:
            country_code = u''

        title_parts = []
        geocode_title_parts = []

        if city_title:
            title_parts.append(city_title)
            geocode_title_parts.append(city_title)

        if terminal_name and city_title != terminal_name:
            title_parts.append(terminal_name)
            if city_title.lower() != terminal_name.lower():
                geocode_title_parts.append(terminal_name)

        if country_code:
            title_parts.append(country_code)

            try:
                country = Country.objects.get(code=country_code)
            except Country.DoesNotExist:
                geocode_title_parts.append(country_code)
            else:
                geocode_title_parts.append(country.title_en or country.title)

        code = u'c' + city_code + u'-t' + terminal_code
        if not title_parts:
            log.error(u'Станция с кодом %s не имеет названия', code)
        station_block = group_block.add_station(u' / '.join(title_parts), code)

        station_block.city_code = city_code
        station_block.geocode_title = u' '.join(geocode_title_parts)

        return station_block

    @cache_method_result
    def check_geocoded_title(self, geocoded_title):
        result = geocode_to_rasp_geo_objects(geocoded_title)
        if result.rasp_geoocode_objects:
            log.info(u'geocoder+++: %s', geocoded_title)
        else:
            log.info(u'geocoder---: %s', geocoded_title)


class AllticketsforMeRawDownloadFileProvider(PackageFileProvider):
    filename = u'schedule.xml'

    def retrieve_data(self, url):
        response = requests.get(
            url, auth=(USER, PASSWORD), timeout=settings.SCHEDULE_IMPORT_TIMEOUT,
            stream=True, verify=False
        )

        return response.iter_content(self.CHUNK_SIZE)

    def get_schedule_files(self):
        filepath = self.get_package_filepath(self.filename)
        filepath = self.download_file(URL, filepath)

        return [filepath]
