# -*- coding: utf-8 -*-

"""
Особенности формата и данных:

1. Временная зона указана в schedule.
   Нужно каждое расписание складывать в отдельную нитку.
"""

import logging
from datetime import timedelta

from lxml import etree
from common.cysix.builder import ChannelBlock, GroupBlock, StationBlock, ThreadBlock, StoppointBlock, ScheduleBlock
from cysix.base import safe_parse_xml
from cysix.tsi_converter import CysixTSIConverterFactory, CysixTSIConverterFileProvider

from travel.rasp.admin.scripts.schedule.utils import RaspImportError
from travel.rasp.admin.scripts.schedule.utils.file_providers import XmlPackageFileProvider, PackageFileProvider
from travel.rasp.admin.scripts.utils.to_python_parsers import parse_date


log = logging.getLogger(__name__)


SCHEDULE_URL = 'http://avokzal.kz/schedules/schedule.xml'

TIMEZONE = 'Asia/Almaty'


class AvokzalKzCysixFactory(CysixTSIConverterFactory):
    def get_raw_download_file_provider(self):
        return AvokzalKzRawDownloadFileProvider(self.package)

    def get_raw_package_file_provider(self):
        return XmlPackageFileProvider(self.package)

    def get_converter_file_provider(self, raw_file_provider):
        return AvokzalKzCysixFileProvider(self.package, raw_file_provider)


class AvokzalKzCysixFileProvider(CysixTSIConverterFileProvider):
    def convert_data(self, filepath):
        c = Converter(self.raw_file_provider)

        c.convert(filepath)


class ConvertError(RaspImportError):
    pass


class Converter(object):
    def __init__(self, provider):
        self.provider = provider

        self.stations = dict()

    def convert(self, filepath):
        files = self.provider.get_schedule_files()

        if not files:
            raise ConvertError(u'Не нашли файл с расписанием.')

        raw_filepath = files[0]

        log.info(u"Парсим файл %s", raw_filepath)

        root_el = self.get_root_element(raw_filepath)

        channel_el = self.build_channel_element(root_el)

        with open(filepath, 'w') as f:
            f.write(etree.tostring(channel_el, xml_declaration=True, encoding='utf-8', pretty_print=True))

    def build_channel_element(self, root_el):
        channel_block = ChannelBlock(
            'bus',
            station_code_system='vendor',
            carrier_code_system='local',
            vehicle_code_system='local',
            timezone=TIMEZONE,
        )

        self.group_block = GroupBlock(channel_block, title='all', code='all')

        self.build_stations(root_el)

        self.build_threads(root_el)

        channel_block.add_group_block(self.group_block)

        return channel_block.get_element()

    def build_stations(self, root_el):
        for station_el in root_el.findall('./stopslist/stoppoint'):
            code = station_el.get('vendor_id', u'').strip()
            title = station_el.get('name', u'').strip()

            station_block = StationBlock(self.group_block, title, code)

            station_block.add_legacy_station(title, code)

            if code not in self.stations:
                self.stations[code] = station_block

                self.group_block.add_station_block(station_block)

    def build_threads(self, root_el):
        for thread_el in root_el.findall('./thread'):
            try:
                raw_thread = RawThread(thread_el)

            except RawThreadConstructError, e:
                log.error(u'%s', e)

            for raw_schedule in raw_thread.schedules:
                try:
                    self.build_thread(raw_thread, raw_schedule)

                except ConvertError, e:
                    log.error(u'Не получилось сконвертировать часть расписания для %s %s. %s',
                              raw_thread.number, raw_thread.title, e)

    def build_thread(self, raw_thread, raw_schedule):
        thread_block = ThreadBlock(self.group_block, raw_thread.title, raw_thread.number,
                                   timezone=raw_schedule.timezone)

        for stop in raw_thread.stoppoints:
            stop_block = self.get_station_block(stop.code)

            thread_block.add_stoppoint_block(StoppointBlock(
                thread_block,
                stop_block,
                arrival_shift=stop.arrival_shift,
                is_nonstop=stop.is_nonstop
            ))

        thread_block.add_schedule_block(ScheduleBlock(
            thread_block,
            raw_schedule.days,
            times=raw_schedule.times,
            period_start_date=raw_schedule.start_date.strftime('%Y-%m-%d'),
            period_end_date=raw_schedule.end_date.strftime('%Y-%m-%d')
        ))

        thread_block.set_raw_data(etree.tostring(raw_thread.thread_el, pretty_print=True,
                                  xml_declaration=False, encoding=unicode))

        self.group_block.add_thread_block(thread_block)

    def get_station_block(self, code):
        try:
            return self.stations[code]

        except KeyError:
            raise ConvertError(u'Нет станции с кодом {}.'.format(code))

    def get_root_element(self, filepath):
        try:
            return safe_parse_xml(filepath).getroot()

        except etree.LxmlError, e:
            raise ConvertError(u'Ошибка разбора xml-файла {}. {}'.format(filepath, e))


class RawThreadConstructError(ConvertError):
    pass


class RawThread(object):
    def __init__(self, thread_el):
        self.thread_el = thread_el

        self.title = thread_el.get('title', u'').strip()
        self.number = thread_el.get('number', u'').strip()

        self.stoppoints = self.get_stoppoints(thread_el)

        if not self.stoppoints:
            raise RawThreadConstructError(
                u'Нет станций следования для {} {}.'.format(self.number, self.title))

        self.schedules = self.get_schedules(thread_el)

    def get_schedules(self, thread_el):
        schedules = [RawSchedule(s_el) for s_el in thread_el.findall('./schedules/schedule')]

        return [s for s in schedules if not s.empty]

    def get_stoppoints(self, thread_el):
        return [RawStoppoint(s_el) for s_el in thread_el.findall('./stations/station')]


class RawSchedule(object):
    def __init__(self, schedule_el):
        self.start_date = get_date(schedule_el.get('period_start_date', u'').strip())
        self.end_date = get_date(schedule_el.get('period_end_date', u'').strip())

        if self.end_date is not None:
            self.end_date += timedelta(days=1)

        self.days = schedule_el.get('days', u'').strip()
        self.times = schedule_el.get('times', u'').strip().replace(u',', u';')

        self.timezone = schedule_el.get('timezone', u'').strip()

        self.empty = False

        if self.start_date is None:
            self.empty = True

        if self.end_date is None:
            self.empty = True

        if not self.empty and self.start_date >= self.end_date:
            self.empty = True


def get_date(date_str):
    return parse_date(date_str, default=None)


class RawStoppoint(object):
    def __init__(self, stoppoint_el):
        self.code = stoppoint_el.get('vendor_id', u'').strip()

        self.arrival_shift = stoppoint_el.get('arrival_time', u'').strip()

        self.is_nonstop = stoppoint_el.get('woArrival', u'').strip()


class AvokzalKzRawDownloadFileProvider(PackageFileProvider):
    filename = u'schedule.xml'

    def get_schedule_files(self):
        download_params = SCHEDULE_URL
        filepath = self.get_package_filepath(self.filename)
        filepath = self.download_file(download_params, filepath)

        return [filepath]
