# -*- coding: utf-8 -*-

import copy
import json
import logging

from lxml import etree
from django.utils import html

from common.cysix.builder import ChannelBlock, GroupBlock, ThreadBlock, ScheduleBlock, StoppointBlock
from cysix.tsi_converter import CysixTSIConverterFactory, CysixTSIConverterFileProvider
from travel.rasp.admin.lib.xmlutils import get_sub_tag_text
from travel.rasp.admin.scripts.schedule.utils import RaspImportError
from travel.rasp.admin.scripts.schedule.utils.file_providers import XmlPackageFileProvider, PackageFileProvider


log = logging.getLogger(__name__)

AVPERM_URL = u'http://avperm.ru/sheduleforya/permkray.xml'

CURRENCY = u'RUR'

SECONDS_IN_MINUTE = 60


class AvPermCysixFactory(CysixTSIConverterFactory):
    def get_raw_download_file_provider(self):
        return AvPermRawDownloadFileProvider(self.package)

    def get_raw_package_file_provider(self):
        return XmlPackageFileProvider(self.package)

    def get_converter_file_provider(self, raw_file_provider):
        return AvPermCysixFileProvider(self.package, raw_file_provider)


class AvPermCysixFileProvider(CysixTSIConverterFileProvider):
    def convert_data(self, filepath):
        c = Converter(self.raw_file_provider)

        c.convert(filepath)


class ConvertError(RaspImportError):
    pass


class Converter(object):
    def __init__(self, provider):
        self.provider = provider

    def convert(self, filepath):
        files = self.provider.get_schedule_files()

        if not files:
            raise ConvertError(u'Не нашли файл с расписанием.')

        raw_filepath = files[0]

        log.info(u"Парсим файл %s", raw_filepath)

        root_element = self._get_etree_from_file(raw_filepath)

        channel_el = self.build_channel_element(root_element)

        with open(filepath, 'w') as f:
            f.write(etree.tostring(channel_el, xml_declaration=True, encoding='utf-8', pretty_print=True))

    def build_channel_element(self, root_element):
        channel_block = ChannelBlock('bus', station_code_system='vendor', carrier_code_system='vendor',
                                     vehicle_code_system='vendor', timezone='start_station')
        group_block = GroupBlock(channel_block, u'Расписание пермского автовокзала', code='1')
        channel_block.add_group_block(group_block)

        for route_el in root_element.findall('./route'):
            route_title = get_value(route_el, 'route_title')
            route_code = get_value(route_el, 'route_code')

            is_transit = u'1' == get_value(route_el, 'tranzit')

            if is_transit:
                log.info(u'Пропускаем транзитный рейс "%s" "%s"', route_code, route_title)
                continue

            stations = self.get_sorted_stations(route_el)

            route_start_list = get_value(route_el, 'start_time').split()
            route_end_list = get_value(route_el, 'end_time').split()

            if len(route_start_list) != len(route_end_list):
                log.error(u'Пропускаем рейс "%s" "%s". '
                          u'Не совпадает количество времен начала и завершения рейса.',
                          route_code, route_title)
                continue

            for route_start_time, route_end_time in zip(route_start_list, route_end_list):
                thread_block = self.build_thread_block(route_el, route_title, group_block, route_start_time)

                fare_block = group_block.add_local_fare()
                thread_block.set_fare_block(fare_block)

                self.build_stoppoints(route_el, stations, group_block, thread_block, fare_block, route_end_time)

                group_block.add_thread_block(thread_block)

        return channel_block.get_element()

    def build_thread_block(self, route_el, route_title, group_block, route_start_time):
        thread_block = ThreadBlock(group_block, route_title)

        self.add_raw_data(thread_block, route_el)
        self.add_carrier(group_block, thread_block, route_el)
        self.add_vehicle(group_block, thread_block, route_el)

        days = get_value(route_el, 'days_of_week')

        if not days:
            days = get_value(route_el, 'days')

        days = u';'.join(days.split())

        period_start_date = get_value(route_el, 'period_start')
        period_end_date = get_value(route_el, 'period_end')

        thread_block.add_schedule_block(ScheduleBlock(thread_block, days, times=route_start_time,
                                                      period_start_date=period_start_date,
                                                      period_end_date=period_end_date))

        return thread_block

    fare_data_from_route_keys = ['id_shablon', 'start_time', 'route_code', 'code_awt', 'day_sale']

    def build_stoppoints(self, route_el, stations, group_block, thread_block, fare_block, route_end_time):
        first_station_block = None

        data_from_route = dict()
        for key in self.fare_data_from_route_keys:
            data_from_route[key] = get_value(route_el, key)

        for i, station_el in enumerate(stations):
            is_first_station = (i == 0)
            is_last_station = (i + 1 == len(stations))

            station_title = get_value(station_el, 'station_title')
            station_code = get_value(station_el, 'station_code')

            code = '%s_%s' % (station_code, group_block.code)

            station_block = group_block.add_station(station_title, code)
            station_block.add_legacy_station(station_title, station_code)

            stoppoint_block = StoppointBlock(thread_block, station_block)

            self.add_arrival(stoppoint_block, station_el,
                             is_first_station, is_last_station, route_end_time)

            stoppoint_block.distance = get_value(station_el, 'kilom_from_start')

            thread_block.add_stoppoint_block(stoppoint_block)

            if is_first_station:
                first_station_block = station_block

            else:
                price = get_value(station_el, 'station_price')

                try:
                    price_value = float(price)

                except ValueError:
                    price_value = None

                if price_value and price_value > 0.:
                    order_data = copy.copy(data_from_route)
                    order_data['station_title'] = station_title
                    order_data['station_price'] = price

                    data = json.dumps(order_data, ensure_ascii=False, encoding='utf8')
                    fare_block.add_price_block(
                        '%.2f' % price_value,
                        CURRENCY,
                        first_station_block,
                        station_block,
                        data=data
                    )

    def add_raw_data(self, thread_block, route_el):
        raw_data = etree.tostring(route_el, pretty_print=True, xml_declaration=False, encoding=unicode)
        thread_block.set_raw_data(raw_data)

    def add_carrier(self, group_block, thread_block, route_el):
        carrier_title = get_value(route_el, 'company_title')
        carrier_code = get_value(route_el, 'company_code')

        if carrier_title and carrier_code:
            carrier_block = group_block.add_carrier(carrier_title, carrier_code)
            thread_block.carrier = carrier_block

    def add_vehicle(self, group_block, thread_block, route_el):
        vehicle_title = get_value(route_el, 'vehicle_title')
        vehicle_code = get_value(route_el, 'vehicle_code')

        if vehicle_title and vehicle_code:
            vehicle_block = group_block.add_vehicle(vehicle_title, vehicle_code)
            thread_block.vehicle = vehicle_block

    def add_arrival(self, stoppoint_block, station_el, is_first_station, is_last_station, route_end_time):
        arrival_shift = get_value(station_el, 'minutes_from_start')

        if not is_first_station and arrival_shift == u'0':
            arrival_shift = u''

        if not arrival_shift and is_last_station:
            stoppoint_block.arrival_time = route_end_time

            return

        try:
            stoppoint_block.arrival_shift = int(arrival_shift) * SECONDS_IN_MINUTE

        except ValueError:
            pass

    def _get_etree_from_file(self, filepath):
        with open(filepath) as f:
            data = f.read()
            return etree.fromstring(data)

    def get_sorted_stations(self, route_el):
        stations = route_el.findall('./station')
        def sort_key(s):
            distance = float(get_value(s, 'kilom_from_start'))
            price = float(get_value(s, 'station_price'))
            time = float(get_value(s, 'minutes_from_start'))
            return (distance, price, time)
        stations.sort(key=sort_key)
        return stations

def get_value(main_el, sub_tag_name):
    value = get_sub_tag_text(main_el, sub_tag_name, silent=True, default=u"").strip()

    if value == u'_':
        value = u''

    return value


class AvPermRawDownloadFileProvider(PackageFileProvider):
    filename = u'permkray.xml'

    def get_schedule_files(self):
        download_params = AVPERM_URL
        filepath = self.get_package_filepath(self.filename)
        filepath = self.download_file(download_params, filepath)

        return [filepath]
