# coding: utf-8

import logging
import os.path
import urllib2
import gzip

from django.conf import settings
from lxml import etree

from common.cysix.builder import ChannelBlock, GroupBlock, ThreadBlock, StoppointBlock, ScheduleBlock
from common.utils.safe_xml_parser import safe_parse_xml
from cysix.tsi_converter import CysixTSIConverterFactory, CysixTSIConverterFileProvider
from travel.rasp.admin.lib.download_file import download_file
from travel.rasp.admin.lib.exceptions import SimpleUnicodeException
from travel.rasp.admin.lib.http_timeout_handler import get_http_timeout_handler_class
from travel.rasp.admin.scripts.schedule.utils.file_providers import XmlPackageFileProvider, PackageFileProvider
from travel.rasp.admin.scripts.utils.import_file_storage import get_schedule_temporary_date_filepath


log = logging.getLogger(__name__)


SCHEDULE_GZ_URL = 'http://ticketbus.by/yandex/beltranscom.gz'
USER = 'yandex'
PASSWORD = 'WhyRmv386Sd'


http_handler = get_http_timeout_handler_class(settings.SCHEDULE_IMPORT_TIMEOUT)()
auth_handler = urllib2.HTTPBasicAuthHandler(urllib2.HTTPPasswordMgrWithDefaultRealm())
auth_handler.add_password(None, 'ticketbus.by', USER, PASSWORD)

opener = urllib2.build_opener(http_handler, auth_handler)
urlopen = opener.open


class BeltranscomCysixFactory(CysixTSIConverterFactory):
    def get_raw_package_file_provider(self):
        return XmlPackageFileProvider(self.package)

    def get_raw_download_file_provider(self):
        log.info(u'Качаем данные по %s', SCHEDULE_GZ_URL)

        return BeltranscomHttpFileProvider(self.package)

    def get_converter_file_provider(self, raw_file_provider):
        return BeltranscomConverterFileProvider(self.package, raw_file_provider)


class BeltranscomHttpFileProvider(PackageFileProvider):
    def get_schedule_files(self):
        filepath = get_schedule_temporary_date_filepath('beltranscom.xml', self.package)

        return self.download_schedule_files(filepath)

    def download_schedule_files(self, filepath):
        if os.path.exists(filepath):
            log.info(u'Файл %s уже скачан берем его', filepath)
        else:
            log.info(u'Скачиваем Файл от поставщикака в %s', filepath)
            self.get_beltranscom_xml(filepath)

        return [filepath]

    def get_beltranscom_xml(self, filepath):
        gz_tmp_filepath = get_schedule_temporary_date_filepath('beltranscom.xml.gz', self.package)
        gz_tmp_filepath = download_file(SCHEDULE_GZ_URL, gz_tmp_filepath, chunk_size=1024, auth=(USER, PASSWORD))

        filepath = get_schedule_temporary_date_filepath('beltranscom.xml', self.package)
        with gzip.open(gz_tmp_filepath) as gz, open(filepath, 'wb') as fp:
            fp.write(gz.read())

        return filepath


class BeltranscomConverterFileProvider(CysixTSIConverterFileProvider):
    def __init__(self, package, raw_file_provider):
        self.package = package
        self.raw_file_provider = raw_file_provider

        super(BeltranscomConverterFileProvider, self).__init__(self.package, self.raw_file_provider)

    def convert_data(self, filepath):
        raw_file = self.raw_file_provider.get_schedule_files()[0]

        c = Converter(raw_file)
        c.convert(filepath)


class ConvertError(SimpleUnicodeException):
    pass


class Converter(object):
    def __init__(self, raw_file):
        self.raw_file = raw_file
        self.raw_root_element = None

    def convert(self, filepath):
        self._reed_raw_data()

        channel_block = ChannelBlock('bus', station_code_system='vendor', carrier_code_system='local',
                                     vehicle_code_system='local', timezone='local')

        group_title = u'Белтранском'
        group_code = 'beltranscom'
        group_block = GroupBlock(channel_block, group_title, group_code)

        stations = self.build_stations(group_block)

        self.build_threads(group_block, stations)

        channel_block.add_group_block(group_block)

        channel_el = channel_block.get_element()

        channel_el = add_legacy_stations(channel_el)

        with open(filepath, 'w') as f:
            f.write(etree.tostring(channel_el, xml_declaration=True, encoding='utf-8', pretty_print=True))

    def _reed_raw_data(self):
        self.raw_root_element = safe_parse_xml(self.raw_file)

    def build_threads(self, group_block, stations):
        for raw_thread_el in self.raw_root_element.findall('./thread'):
            try:
                thread_block = self.build_thread(group_block, raw_thread_el, stations)
                group_block.add_thread_block(thread_block)

            except ConvertError, e:
                log.error(e.msg)

    def build_thread(self, group_block, raw_thread_el, stations):
        code = raw_thread_el.get('route_code', u'').strip()
        number = clean_number(raw_thread_el.get('number', u'').strip())
        title = raw_thread_el.get('title', u'').strip()
        thread_name = u'{}: {}'.format(code, title)
        thread_block = ThreadBlock(group_block, title=title, number=number)

        self.add_stoppoints_to_thread(thread_block, raw_thread_el, thread_name, stations)
        self.add_schedule_to_thread(thread_block, raw_thread_el)
        self.add_raw_data_to_thread(thread_block, raw_thread_el)

        if not thread_block._schedules:
            raise ConvertError(u"Пропускаем нитку '{}'. Не удалось найти расписание.".format(thread_name))

        if not thread_block._stoppoints:
            raise ConvertError(u"Пропускаем нитку '{}'. Не удалось найти станции следования.".format(thread_name))

        return thread_block

    def add_stoppoints_to_thread(self, thread_block, raw_thread_el, thread_name, stations):
        for raw_stoppoint in raw_thread_el.findall('./stations/station'):
            code = raw_stoppoint.get('vendor_id', u'').strip()
            arrival = raw_stoppoint.get('arrival_time', u'').strip()
            departure = raw_stoppoint.get('departure_time', u'').strip()

            station = stations.get(code, None)

            if station is None:
                raise ConvertError(
                    u"Пропускаем нитку '{}'. Не удалось найти станцию с кодом '{}'.".format(thread_name, code)
                )

            stoppoint_block = StoppointBlock(thread_block, station, arrival_shift=arrival, departure_shift=departure)
            thread_block.add_stoppoint_block(stoppoint_block)

    def add_schedule_to_thread(self, thread_block, raw_thread_el):
        for raw_schedule_el in raw_thread_el.findall('./schedules/schedule'):

            days = raw_schedule_el.get('days', u'').strip()

            # https://st.yandex-team.ru/RASPADMIN-23
            if days in [u'f', u'F']:
                days = u'через день'

            schedule_attributes = (
                'period_start_date',
                'period_end_date',
                'period_start_time',
                'period_end_time',
                'period_int',
                'canceled',
                'times',
            )

            schedule_dict = dict()

            for attr in schedule_attributes:
                value = raw_schedule_el.get(attr, u'').strip()

                if value:
                    schedule_dict[attr] = value

            schedule = ScheduleBlock(thread_block, days, **schedule_dict)
            thread_block.add_schedule_block(schedule)

    def add_raw_data_to_thread(self, thread_block, raw_thread_el):
        raw_data = etree.tostring(raw_thread_el, pretty_print=True, xml_declaration=False, encoding=unicode)
        thread_block.set_raw_data(raw_data)

    def build_stations(self, group_block):
        stations = dict()

        for s_el in self.raw_root_element.findall('./stopslist/stoppoint'):
            code = s_el.get('vendor_id', u'').strip()
            title = s_el.get('name', u'').strip()
            station_block = group_block.add_station(title, code)
            stations[code] = station_block

        return stations


def add_legacy_stations(channel_el):
    for stations_el in channel_el.findall('.//stations'):
        for station_el in stations_el.findall('./station'):
            legacy_station_el = etree.Element('legacy_station')
            legacy_station_el.set('code', station_el.attrib.get('code', ''))
            legacy_station_el.set('title', station_el.attrib.get('title', ''))
            legacy_station_el.set('type', 'raw')

            station_el.append(legacy_station_el)

    return channel_el


def clean_number(number):
    if number.isdigit() and len(number) <= 4:
        return number
    else:
        return u''
