# coding: utf-8

import re
import os.path
from StringIO import StringIO
from lxml import etree

import pexpect
from django.conf import settings

from common.cysix.builder import ChannelBlock, GroupBlock, ThreadBlock, ScheduleBlock, StationBlock, StoppointBlock
from common.models.geo import Station
from common.importinfo.models.bus import BuscomuaStationCode
from common.utils.caching import cache_method_result
from cysix.tsi_converter import CysixTSIConverterFactory, CysixTSIConverterFileProvider
from cysix.two_stage_import.importer import CysixTwoStageImporter
from travel.rasp.admin.lib.logs import get_current_file_logger
from travel.rasp.admin.lib.xmlutils import copy_lxml_element
from travel.rasp.admin.scripts.schedule.utils.file_providers import XmlPackageFileProvider, PackageFileProvider
from travel.rasp.admin.scripts.schedule.utils import RaspImportError
from travel.rasp.admin.scripts.utils.import_file_storage import get_schedule_temporary_date_filepath


log = get_current_file_logger()


SERVER_FILENAME = '4yandex.xml.v0.9.1'
SSH_HOST = "bus2yandex@sm.bus.com.ua"
SSH_PASSWORD = '+380673766849'

UKR_I = u'І'
UKR_E = u'Є'


MIN_GOOD_K_REG_VALUE = 50  # k_reg - Коэффициент регулярности рейса в %


class BuscomuaFactory(CysixTSIConverterFactory):
    def get_raw_download_file_provider(self):
        return BuscomuaSSHFileProvider(self.package)

    def get_raw_package_file_provider(self):
        return XmlPackageFileProvider(self.package)

    def get_converter_file_provider(self, raw_file_provider):
        return BuscomuaCysixFileProvider(self.package, raw_file_provider)

    @cache_method_result
    def get_two_stage_importer(self):
        return BuscomuaTwoStageImporter(self)


class BuscomuaCysixFileProvider(CysixTSIConverterFileProvider):
    def convert_data(self, filepath):
        buscomua_file = self.raw_file_provider.get_schedule_files()[0]

        tree = etree.parse(buscomua_file)

        channel_block = ChannelBlock('bus', station_code_system='vendor', vehicle_code_system='local', timezone="local")

        for bus_station_el in tree.findall('.//BusStation'):
            group_block = self.build_group(channel_block, bus_station_el)
            channel_block.add_group_block(group_block)

        channel_block.to_xml_file(filepath)

    def build_group(self, channel_block, bus_station_el):
        self.bus_station_el = bus_station_el
        self.group_block = GroupBlock(channel_block, bus_station_el.get("name", u"").strip(),
                                      bus_station_el.get("station_code", u"").strip())

        self._current_group_points = {}
        self._fill_points()

        self.parse_routes()

        return self.group_block

    def _fill_points(self):
        for point in self.bus_station_el.findall('.//point'):
            self._current_group_points[point.get('station_code', u"").strip()] = point

    def parse_routes(self):
        for route_el in self.bus_station_el.findall('.//route'):
            self.parse_route(route_el)

    time_re = re.compile(u"^\d\d:\d\d$")

    def parse_route(self, route_el):
        number = route_el.get('route_code', u"").strip()
        title = route_el.get('route_title', u"").strip()

        k_reg = get_k_reg(route_el)

        if k_reg < MIN_GOOD_K_REG_VALUE:
            log.info(u'Для рейса %s %s слишком низкий коэффициент регулярности %s. Пропускаем.',
                     number, title, k_reg)

            return

        vehicle_title = route_el.get('vehicle_title', u"").strip()
        vehicle_recommended_title = get_vehicle_recommended_title(vehicle_title)
        vehicle_block = self.group_block.add_local_vehicle(vehicle_title, vehicle_recommended_title)

        thread_block = ThreadBlock(self.group_block, title, number)
        thread_block.vehicle = vehicle_block

        thread_block.add_schedule_block(self.build_schedule(thread_block, route_el))

        for station_el in route_el.findall(".//station"):
            try:
                distance = unicode(float(station_el.get("distans", u"").strip()))
            except ValueError:
                distance = u""

            departure_time = station_el.get("departure").strip()
            if not self.time_re.match(departure_time):
                departure_time = u""

            arrival_time = station_el.get("arrival").strip()
            if not self.time_re.match(arrival_time):
                arrival_time = u""

            station_block = self.build_station_block(station_el)

            stoppoint_block = StoppointBlock(thread_block, station_block, distance=distance,
                                             departure_time=departure_time, arrival_time=arrival_time)

            thread_block.add_stoppoint_block(stoppoint_block)

        raw_data = u"\n\n".join([
            etree.tounicode(copy_lxml_element(self.bus_station_el, top_element_only=True), pretty_print=True),
            etree.tounicode(copy_lxml_element(route_el), pretty_print=True)
        ])

        thread_block.set_raw_data(raw_data)

        self.group_block.add_thread_block(thread_block)

    def build_schedule(self, thread_block, route_el):
        dates = []
        for departure_el in route_el.findall('.//departure'):
            dates.append(departure_el.get("date", u"").strip())

        days = u";".join(dates)

        return ScheduleBlock(thread_block, days)

    def build_station_block(self, station_el):
        station_code = station_el.get("station", u"").strip()
        point_el = self._current_group_points.get(station_code)

        city_code = station_el.get('city', u"").strip()
        title = u""
        real_title = u""

        if point_el is not None:
            city_code = point_el.get("city_code", u"").strip() or city_code
            title = point_el.get('station_title', u"")

            real_title = point_el.get('city_title_ru', point_el.get('station_title', u"")).lower()
            real_title = u" ".join(map(unicode.capitalize, real_title.split()))

        busstation_code = self.group_block.code

        legacy_code = u"_".join([busstation_code, city_code, station_code])
        code = u"_".join(filter(None, [city_code, station_code]))

        station_block = StationBlock(self.group_block, title, code, recommended_title=real_title)
        station_block.add_legacy_station(title, legacy_code)

        self.group_block.add_station_block(station_block)

        station_block.set_json_data({
            "city_code": city_code,
            "station_code": station_code,
            "server_code": busstation_code
        })

        return station_block


def get_vehicle_recommended_title(vehicle_title):
    if not vehicle_title:
        return None

    vehicle_recommended_title = vehicle_title

    # кавычки
    vehicle_recommended_title = re.sub(u'(".*?")', r' \1 ', vehicle_recommended_title)

    vehicle_recommended_title = u' '.join(vehicle_recommended_title.split())

    # vehicle_recommended_title = u" ".join(vehicle_recommended_title.replace(u'(', u' (').split())
    vehicle_recommended_title = re.sub(u'(.+?)\((.+?)\)$', r'\1 (\2)', vehicle_recommended_title)

    vehicle_recommended_title = (vehicle_recommended_title
                                 .replace(u'#', u'')
                                 .replace(u'_', u' ')
                                 .replace(u' -', u'-')
                                 .replace(u'- ', u'-')
                                 .replace(UKR_I, u'И')
                                 .replace(UKR_I.lower(), u'и')
                                 .replace(UKR_E, u'Е')
                                 .replace(UKR_E.lower(), u'е'))

    # отделить цифры от всего, что больше 3 символов
    vehicle_recommended_title = re.sub(u'(\w{3,}?)((?<=\D)\d+)',
                                       r'\1 \2', vehicle_recommended_title, flags=re.U)

    # отделить разные языки (тут только русский от английского)
    vehicle_recommended_title = re.sub(u'([a-zA-Z]{3,})([А-Яа-я]{3,})',
                                       r'\1 \2', vehicle_recommended_title)

    vehicle_recommended_title = re.sub(u'([А-Яа-я]{3,})([a-zA-Z]{3,})',
                                       r' \1 \2', vehicle_recommended_title)

    parts = vehicle_recommended_title.split()

    # отделить то, что CamalCase пробелами, но только если каждвя часть > 3 символов
    new_first_part = re.sub(u'((?<=[a-zа-я])[A-ZА-Я]|(?<!\A)[A-ZА-Я](?=[a-zа-я]))', r' \1', parts[0])
    if all([len(part) > 3 for part in new_first_part.split()]):
        parts[0] = new_first_part

    vehicle_recommended_title = u' '.join(parts)

    # Capitalize all words what more whan 3 simbols
    vehicle_recommended_title = re.sub(u'([^\W\d_]{3,})', lambda match: match.group(1).capitalize(),
                                       vehicle_recommended_title, flags=re.U)

    # если первая часть названия не более 3 символов, то они все должны быть заглавными
    parts = vehicle_recommended_title.split(u' ')
    if len(parts[0]) <= 3:
        parts[0] = parts[0].upper()
    vehicle_recommended_title = u' '.join(parts)

    parts = vehicle_recommended_title.split(u'-')
    if len(parts[0]) <= 3:
        parts[0] = parts[0].upper()
    return u'-'.join(parts)


class BuscomuaTwoStageImporter(CysixTwoStageImporter):
    def post_import_routes(self):
        log.info(u"Обновляем коды городов")
        finder = self.factory.get_station_finder()

        for path in self.get_good_paths():
            for cysix_station in path:
                try:
                    station = finder.find_by_supplier_station(cysix_station)

                    self.__set_buscomua_code(station, cysix_station)
                except (Station.DoesNotExist, Station.MultipleObjectsReturned):
                    pass

    def __set_buscomua_code(self, station, cysix_station):
        data = cysix_station.get_json_data()
        if not data or not (data['city_code'] and data['server_code'] and data['station_code']):
            return

        try:
            old_code = BuscomuaStationCode.objects.get(city_code=data["city_code"],
                                                       station_code=data["station_code"],
                                                       server_code=data["server_code"])
            if old_code.station != station:
                another_station = old_code.station
                log.error(u"Несколько станций имеют один и тот же код %s %s: %s %s, %s %s",
                          data['city_code'], data['station_code'],
                          another_station.id, another_station.title,
                          station.id, station.title)
        except BuscomuaStationCode.DoesNotExist:
            BuscomuaStationCode.objects.create(city_code=data["city_code"],
                                               station_code=data["station_code"],
                                               server_code=data["server_code"],
                                               station=station)


class BuscomuaSSHFileProvider(PackageFileProvider):
    def get_schedule_files(self):
        filepath = get_schedule_temporary_date_filepath('buscomua.xml', self.package)

        return [self.download_schedule_file(filepath)]

    def download_schedule_file(self, filepath):
        if os.path.exists(filepath):
            log.info(u"Файл %s уже скачан берем его", filepath)
            return filepath

        log.info(u"Качаем файл %s:~/%s в %s", SSH_HOST, SERVER_FILENAME, filepath)

        pexpect_log = StringIO()

        good = False

        try:
            child = pexpect.spawn("scp", ['-o', 'UserKnownHostsFile=/dev/null', '-o', 'StrictHostKeyChecking=no',
                                          '-v',
                                          '%s:~/%s' % (SSH_HOST, SERVER_FILENAME), filepath],
                                  timeout=settings.SCHEDULE_IMPORT_TIMEOUT)
            child.logfile = pexpect_log
            child.expect('password: ')
            child.sendline(SSH_PASSWORD)

            result = child.expect(['Permission denied', 'Authentication succeeded'])
            if result == 0:
                log.error(u"Не верные логин или пароль")
                child.terminate(True)
            elif result == 1:
                child.expect('Exit status 0')
                good = True
        except (pexpect.EOF, pexpect.TIMEOUT):
            log.error(u"Процесс завершился не корректно")

        if not good:
            log.error(u"\n%s", pexpect_log.getvalue().replace(SSH_PASSWORD, '*************'))
            raise RaspImportError(u"Не удалось скачать свежий файл.")

        else:
            log.info(u"Файл %s скачан успешно", filepath)
            return filepath


def get_k_reg(route_el):
    # Коэффициент регулярности рейса в %
    kreg = route_el.get('kreg', u'').strip()

    kreg = kreg.replace(u'%', u'')

    try:
        return float(kreg)

    except ValueError:
        return 100.
