# coding: utf-8

import decimal
import hashlib
import json
import os.path
from collections import defaultdict
from datetime import timedelta, time

from lxml import etree
from django.conf import settings
from django.utils.encoding import smart_str

from common.cysix.builder import ChannelBlock, GroupBlock, ThreadBlock, ScheduleBlock, StoppointBlock
from common.settings.configuration import Configuration
from common.settings.utils import get_setting
from cysix.base import CysixConvertError
from cysix.two_stage_import.factory import CysixTSIFactory
from cysix.two_stage_import.route_importer import CysixTSIRouteImporter
from travel.rasp.admin.importinfo.models.e_traffic import EtrafficMainBusStationInGroup
from common.utils.caching import cache_method_result, cache_method_result_with_exception
from travel.rasp.admin.lib.logs import get_current_file_logger
from common.utils.http import urlopen
from travel.rasp.admin.scripts.schedule.utils import RaspImportError
from travel.rasp.admin.scripts.schedule.utils.clean_duplicates import DuplicateCleaner
from travel.rasp.admin.scripts.schedule.utils.file_providers import PackageFileProvider
from travel.rasp.admin.scripts.schedule.utils.to_python_parsers import get_sql_datetime


log = get_current_file_logger()


LOGIN_URL = 'login/%(uid)s'
BASE_URL = 'http://api.e-traffic.ru/'
DEPOTS = 'depots;%(uid)s;%(key)s'
DEPOT = 'depot/%(depot_alias)s;%(uid)s;%(key)s'
SCHEDULE_DATES = 'schedule/dates/%(depot_alias)s;%(uid)s;%(key)s'
SCHEDULE_RACES_ALL = 'schedule/races/%(depot_alias)s;%(uid)s;%(key)s'
SCHEDULE_RACES_DATE = 'schedule/races/%(depot_alias)s/%(date)s;%(uid)s;%(key)s'
ETRAFFIC_UID_AND_SALT = get_setting('ETRAFFIC_UID_AND_SALT', {
    Configuration.PRODUCTION: ('yandex1', '861348560'),
    Configuration.TESTING: ('yandex2', '112452558'),
    Configuration.DEVELOPMENT: ('yandex3', '001561890'),
}, default=('yandex', '112334215'))

CURRENCY = u'RUR'


class ETrafficCysixFactory(CysixTSIFactory):
    def get_download_file_provider(self):
        log.info(u"Качаем данные %s", BASE_URL)

        return ETrafficCysixFileProvider(self.package)

    def get_route_importer(self):
        return ETrafficRouteImporter(self)


class ETrafficCysixFileProvider(PackageFileProvider):
    def __init__(self, package, supplier=None):
        self.uid, self.salt = ETRAFFIC_UID_AND_SALT
        super(ETrafficCysixFileProvider, self).__init__(package, supplier)

    @cache_method_result_with_exception
    def get_cysix_file(self):
        filepath = self.get_package_filepath('cysix.xml')

        if os.path.exists(filepath) and not settings.DEBUG:
            log.info(u"Данные уже были сконвертированы в общий xml %s", filepath)

            return filepath

        log.info(u"Конвертируем в общий xml %s", filepath)

        self.download_and_convert_data(filepath)

        log.info(u"Данные сконвертированы в общий xml %s", filepath)

        return filepath

    @cache_method_result
    def get_public_key(self):
        url = BASE_URL + (LOGIN_URL % {'uid': self.uid})

        log.info(u"Получаем ключ: %s", url)
        data = urlopen(url, timeout=settings.SCHEDULE_IMPORT_TIMEOUT).read()

        json_data = json.loads(data)

        if not json_data['success']:
            raise RaspImportError(u"Не смогли получить успешный ответ по %s" % url)

        return json_data['public']

    def retrieve_data(self, download_params):
        data = ''.join(super(ETrafficCysixFileProvider, self).retrieve_data(download_params))

        json_data = json.loads(data)

        return smart_str(json.dumps(json_data, indent=6, encoding='utf8', ensure_ascii=False))

    def get_depots(self):
        filepath = self.get_package_filepath('depots.json')

        url = BASE_URL + DEPOTS % {
            'key': self.get_key(),
            'uid': self.uid,
        }

        log.info(u"Пробуем скачать url %s", url)

        self.download_file(url, filepath)

        return filepath

    def get_depot(self, depot_alias):
        filepath = self.get_package_filepath('depot_%s.json' % depot_alias)

        url = BASE_URL + DEPOT % {
            'key': self.get_key(),
            'uid': self.uid,
            'depot_alias': depot_alias
        }

        log.info(u"Пробуем скачать url %s", url)

        self.download_file(url, filepath)

        return filepath

    def get_schedule_dates(self, depot_alias):
        filepath = self.get_package_filepath('schedule_dates_%s.json' % depot_alias)

        url = BASE_URL + SCHEDULE_DATES % {
            'key': self.get_key(),
            'uid': self.uid,
            'depot_alias': depot_alias
        }

        log.info(u"Пробуем скачать url %s", url)

        self.download_file(url, filepath)

        return filepath

    def get_schedule_races_all(self, depot_alias):
        filepath = self.get_package_filepath('schedule_races_all_%s.json' % depot_alias)

        url = BASE_URL + SCHEDULE_RACES_ALL % {
            'key': self.get_key(),
            'uid': self.uid,
            'depot_alias': depot_alias
        }

        log.info(u"Пробуем скачать url %s", url)

        self.download_file(url, filepath)

        return filepath

    def get_schedule_races_date(self, depot_alias, day):
        filepath = self.get_package_filepath('schedule_races_%s_%s.json' % (day, depot_alias))

        url = BASE_URL + SCHEDULE_RACES_DATE % {
            'key': self.get_key(),
            'uid': self.uid,
            'depot_alias': depot_alias,
            'date': day
        }

        log.info(u"Пробуем скачать url %s", url)

        self.download_file(url, filepath)

        return filepath

    def get_key(self):
        return md5(md5(self.uid) + md5(self.salt) + md5(self.get_public_key()))

    def download_and_convert_data(self, filepath):
        converter = ETrafficConverter()
        converter.convert(self, filepath)


class ETrafficRouteImporter(CysixTSIRouteImporter):
    def after_import(self):
        if not self.factory.package.new_trusted:
            try:
                cleaner = DuplicateCleaner(self.get_affected_routes().filter(script_protected=False),
                                           self.tariffs, self.distances)

                cleaner.clean()

            except Exception:
                log.exception(u"Ошибка при очистке дубликатов")

        super(ETrafficRouteImporter, self).after_import()


class ETrafficConverter(object):
    def __init__(self):
        self.main_station_by_group_code = {
            main_station.group_code: main_station
            for main_station in EtrafficMainBusStationInGroup.objects.all()
        }

    def convert(self, file_provider, out_filepath):
        with decimal.localcontext() as decimal_context:
            # decimal используем для цен, поэтому выставляем соответствующую точность
            decimal_context.prec = 2

            self._convert(file_provider, out_filepath)

    def _convert(self, file_provider, out_filepath):
        self.file_provider = file_provider

        channel_block = ChannelBlock(
            'bus',
            station_code_system='vendor',
            carrier_code_system='local',
            vehicle_code_system='local',
            timezone='start_station'
        )

        depots_json = read_json_from_file(file_provider.get_depots())

        for depot in depots_json['data']:
            try:
                self.create_main_station_if_not_exists(file_provider, depot)

                group_block = self.build_group(depot, channel_block)
                channel_block.add_group_block(group_block)
            except CysixConvertError, e:
                log.error(unicode(e))
            except Exception:
                log.exception(u"Ошибка построения группы %s", depot['alias'])

        with open(out_filepath, 'w') as f:
            f.write(etree.tostring(channel_block.get_element(),
                                   xml_declaration=True, encoding='utf-8', pretty_print=True))

    def create_main_station_if_not_exists(self, file_provider, depot):
        if depot['alias'] in self.main_station_by_group_code:
            return

        main_station = ETrafficMainStation(file_provider, depot)
        if not main_station.is_valid():
            raise CysixConvertError(
                u'Для новой группы %s не удалось определить главный автовокзал. '
                u'Нужно руками внести данные в "E-traffic: Главный автовокзал группы"', depot['alias']
            )

        self.create_main_station(depot, main_station)

    def create_main_station(self, depot, main_station):
        group_code = depot['alias']
        code = main_station.code
        title = main_station.title
        main_station = EtrafficMainBusStationInGroup.objects.create(group_code=group_code, code=code, title=title)
        self.main_station_by_group_code[group_code] = main_station
        log.info(u'Для группы %s автоматически определили главный автовокзал %s %s', group_code, code, title)

    def build_group(self, depot, channel_block):
        dates_json = read_json_from_file(self.file_provider.get_schedule_dates(depot['alias']))

        races_files = []
        for date_info in dates_json['data']:
            races_files.append(self.file_provider.get_schedule_races_date(depot['alias'], date_info['date']))

        def races_generator():
            for races_file in races_files:
                self.file_provider.get_schedule_races_date(depot['alias'], date_info['date'])

                races_json = read_json_from_file(races_file)

                for race in races_json['data']:
                    yield race

        races = races_generator()

        group_block = GroupBlock(channel_block, title=depot['name'], code=depot['alias'])

        self.current_depot = group_block.code

        self.build_threads(races, group_block)

        return group_block

    def build_threads(self, races, group_block):
        races_by_key = defaultdict(list)

        for race in races:
            races_by_key[self.get_key(race)].append(race)

        for races in races_by_key.itervalues():
            try:
                self.build_thread(races, group_block)

            except CysixConvertError, e:
                log.error(unicode(e))

    def build_thread(self, races, group_block):
        main_race = races[0]

        if not main_race['stations']:
            return

        log.info(u'Начинаем разбор нитки %s - %s', main_race['num'] or u'', main_race['name'])

        # формируем сырые данные перед исправлением races
        thread_raw_data = json.dumps(races, indent=6, ensure_ascii=False, encoding='utf8')

        self.fix_races(races)

        thread_block = ThreadBlock(
            group_block,
            title=main_race['name'],
            number=main_race['num'] or None,
        )
        if main_race['carrier']:
            thread_block.carrier = group_block.add_local_carrier(title=main_race['carrier'])

        thread_block.set_raw_data(thread_raw_data)

        start_dt = get_sql_datetime(main_race['dispatch_date'])

        thread_block.add_schedule_block(ScheduleBlock(
            thread_block,
            days=u';'.join(get_sql_datetime(r['dispatch_date']).strftime('%Y-%m-%d') for r in races),
            canceled=u'0' if unicode(main_race['enabled']).strip() == u'1' else u'1',
            times=str(start_dt.time()),
        ))

        stops = self.build_stops(main_race, start_dt, group_block, thread_block)

        self.fix_first_station_and_check_thread(stops, start_dt, group_block, thread_block)

        for stoppoint_block in stops:
            thread_block.add_stoppoint_block(stoppoint_block)

        # Сюда приходим только, если нитка идет от главного автовокзала группы.
        # Можно смело добавлять данные для покупки
        self.add_fares_to_thread(races, stops, group_block, thread_block)

        group_block.add_thread_block(thread_block)

    def build_stops(self, main_race, start_dt, group_block, thread_block):
        self.fix_race_stop_times(main_race, start_dt)

        stops = []

        first_index = 0
        last_index = len(main_race['stations']) - 1
        for index, etraffic_station in enumerate(main_race['stations']):
            if index == last_index:
                times_and_shifts = self.get_last_times_and_shifts(start_dt, etraffic_station,
                                                                  race_arrival=main_race['arrival_date'])
            elif index == first_index:
                times_and_shifts = self.get_first_times_and_shifts(start_dt, etraffic_station)
            else:
                times_and_shifts = self.get_times_and_shifts(start_dt, etraffic_station)

            stoppoint_block = StoppointBlock(
                thread_block,
                group_block.add_station(etraffic_station['name'], etraffic_station['code']),
                distance=etraffic_station.get('distance') or None,
                arrival_time=times_and_shifts.arrival_time,
                arrival_day_shift=times_and_shifts.arrival_day_shift,
                departure_time=times_and_shifts.departure_time,
                departure_day_shift=times_and_shifts.departure_day_shift,
            )

            stoppoint_block.price = etraffic_station['sale_price']

            stops.append(stoppoint_block)

        return stops

    def get_times_and_shifts(self, start_dt, etraffic_station):
        arrival_dt = get_sql_datetime(etraffic_station['arrival_date'], silent=True)
        departure_dt = None

        if arrival_dt is not None and etraffic_station['stop_time']:
            departure_dt = arrival_dt + timedelta(minutes=etraffic_station['stop_time'])

        return StopTimesAndShifts.make_from_dts(start_dt, arrival_dt, departure_dt)

    def get_first_times_and_shifts(self, start_dt, etraffic_station):
        arrival_dt = None
        departure_dt = get_sql_datetime(etraffic_station['arrival_date'], silent=True)

        if etraffic_station['stop_time']:
            log.warning(u'Указано время стоянки %s для первой станци', etraffic_station['stop_time'])

        return StopTimesAndShifts.make_from_dts(start_dt, arrival_dt, departure_dt)

    def get_last_times_and_shifts(self, start_dt, etraffic_station, race_arrival):
        arrival_dt = get_sql_datetime(etraffic_station['arrival_date'], silent=True)

        if arrival_dt is None:
            arrival_dt = get_sql_datetime(race_arrival, silent=True)

        return StopTimesAndShifts.make_from_dts(start_dt, arrival_dt, None)

    def add_fares_to_thread(self, races, stops, group_block, thread_block):
        fare_block = group_block.add_local_fare()

        etraffic_races = {race['dispatch_date']: race['code'] for race in races}

        for stop in stops[1:]:
            if stop.price > 0:
                order_data = {
                    'etraffic_races': etraffic_races,
                    'depot': self.current_depot,
                    'to_code': stop.station.code,
                }

                fare_block.add_price_block(
                    str(stop.price),
                    CURRENCY,
                    stops[0].station,
                    stop.station,
                    data=json.dumps(order_data, ensure_ascii=False, encoding='utf8'),
                )

        thread_block.set_fare_block(fare_block)

    def fix_first_station_and_check_thread(self, stops, start_dt, group_block, thread_block):
        self.skip_thread_if_depot_in_thread_and_not_first(stops)

        if self.stoppoint_is_depot(stops[0]):
            return

        action = self.main_station_by_group_code[self.current_depot].action_on_thread_without_main_station
        if action == 'skip':
            raise CysixConvertError(u'Не нашли главный автовокзал группы в нитке. Пропускаем нитку.')
        elif action == 'add_first':
            self.add_depot_to_stops(stops, start_dt, group_block, thread_block)
        else:
            raise CysixConvertError(u'Неизвестное действие %s. Пропускаем нитку.', action)

    def add_depot_to_stops(self, stops, start_dt, group_block, thread_block):
        first_stoppoint_block = StoppointBlock(
            thread_block,
            group_block.add_station(
                self.main_station_by_group_code[self.current_depot].title,
                u'root_{}'.format(self.main_station_by_group_code[self.current_depot].code)
            ),
            distance='0',
            departure_time=start_dt.time(),
            departure_day_shift='0',
        )

        stops.insert(0, first_stoppoint_block)

    def stoppoint_is_depot(self, stoppoint_block):
        depot_code = self.main_station_by_group_code[self.current_depot].code
        depot_title = self.main_station_by_group_code[self.current_depot].title

        return (depot_code, depot_title) == (stoppoint_block.station.code, stoppoint_block.station.title)

    def skip_thread_if_depot_in_thread_and_not_first(self, stops):
        for stop in stops[1:]:
            if self.stoppoint_is_depot(stop):
                CysixConvertError(u'Не первая станция маршрута - главный автовокзал группы. '
                                  u'Пропускаем нитку.')

    RACE_INFO_KEY_ATTRIBUTES = ('enabled', 'description', 'platform', 'type', 'carrier', 'name', 'depot_id')
    STATION_KEY_ATTRIBUTES = ('distance', 'code', 'name', 'sale_price')

    def get_key(self, race):
        key = ''
        race_info = dict((k, v) for k, v in race.items() if k in self.RACE_INFO_KEY_ATTRIBUTES)
        race_info['dep_time'] = get_sql_datetime(race['dispatch_date']).time()
        race_info['arr_time'] = get_sql_datetime(race['dispatch_date']).time()

        key += repr(race_info)

        for station in race['stations']:
            station_info = dict((k, v) for k, v in station.items() if k in self.STATION_KEY_ATTRIBUTES)
            station_info['time'] = race['arrival_date'] and get_sql_datetime(race['arrival_date']).time()
            station_info['stop_time'] = race['stop_time'] if 'stop_time' in race else ''

            key += repr(station_info)

        return key

    def fix_races(self, races):
        for race in races:
            for stop in race['stations']:
                # Может быть null, "0" и 0
                if stop['stop_time']:
                    stop['stop_time'] = int(str(stop['stop_time']))

                else:
                    stop['stop_time'] = 0

                stop['sale_price'] = decimal.Decimal(str(stop.get('sale_price', 0)))  # Может быть "0" и 0

    def fix_race_stop_times(self, race, start_dt):
        fake_time = time(0, 0, 0)

        def fix_obj_arrival_date(obj, msg):
            arrival_dt = get_sql_datetime(obj['arrival_date'], silent=True)

            if arrival_dt is not None:
                if arrival_dt.time() == fake_time and arrival_dt.date() == start_dt.date():
                    log.info(msg, obj['name'])

                    obj['arrival_date'] = None

        fix_obj_arrival_date(race, u"Пропускаем нулевое время прибытия для маршрута '%s'")

        for stop in race['stations']:
            fix_obj_arrival_date(stop, u"Пропускаем нулевое время для станции '%s'")


class StopTimesAndShifts(object):
    def __init__(self):
        self.arrival_time = None
        self.arrival_day_shift = None
        self.departure_time = None
        self.departure_day_shift = None

    @classmethod
    def make_from_dts(cls, start_dt, arrival_dt, departure_dt):
        obj = cls()

        obj.arrival_time, obj.arrival_day_shift = obj.make_time_and_shift(start_dt, arrival_dt)
        obj.departure_time, obj.departure_day_shift = obj.make_time_and_shift(start_dt, departure_dt)

        return obj

    @staticmethod
    def make_time_and_shift(start_dt, dt):
        if dt is None:
            return None, None

        return dt.strftime('%H:%M:%S'), str((dt.date() - start_dt.date()).days)


class ETrafficMainStation(object):
    def __init__(self, file_provider, depot):
        depot_stations = read_json_from_file(file_provider.get_depot(depot['alias']))
        self.code = depot_stations['data'].get('code')
        self.title = depot_stations['data'].get('name')

    def is_valid(self):
        return self.code and self.title


def md5(value):
    return hashlib.md5(value).hexdigest()


def read_json_from_file(filepath):
    with open(filepath) as f:
        return json.load(f)
