# coding: utf8

import travel.rasp.admin.scripts.load_project  # noqa

import os.path
import logging
from argparse import ArgumentParser
from datetime import datetime, timedelta
from ftplib import FTP
from StringIO import StringIO
from xml.etree import ElementTree
from xml.etree.ElementTree import ParseError

from django.conf import settings

from library.python import resource
from common.models.geo import Station, StationMajority, Settlement, Region
from common.models.schedule import RThreadType, Supplier, Route, RThread, RTStation
from common.models.transport import TransportType
from common.utils.caching import cache_method_result_with_exception, cache_method_result
from common.utils.date import timedelta2minutes
from common.utils.unicode_csv import UnicodeDictWriter, UnicodeDictReader
from travel.rasp.admin.importinfo.bus_stations import StationFinder2, Finder2SupplierStation, SF2ContextSearcher
from travel.rasp.admin.importinfo.models import OriginalThreadData
from travel.rasp.admin.importinfo.models.bus import MTAInternalNumberBlackList, MTADistrictMapping, MTASettlementMapping
from travel.rasp.admin.lib.fileutils import remove_tmp_dir
from travel.rasp.admin.lib.import_bounds import import_bounds
from travel.rasp.admin.lib.logs import print_log_to_stdout, create_current_file_run_log, get_script_log_context, ylog_context
from travel.rasp.admin.lib.un7zip import get_7zip_filedict
from travel.rasp.admin.lib.unzip import unpack_zip_file
from travel.rasp.admin.scripts.schedule.utils import SupplierMappingContext
from travel.rasp.admin.scripts.schedule.utils.errors import RaspImportError
from travel.rasp.admin.scripts.schedule.utils.mask_builders import MaskBuilder
from travel.rasp.admin.scripts.schedule.utils.report import ScheduleImportReporter
from travel.rasp.admin.scripts.schedule.utils.route_loader import SupplierRouteUpdater, BulkRouteSaver, CompactThreadNumberBuilder
from travel.rasp.admin.scripts.schedule.utils.schedule_validation import simple_thread_validator, ScheduleValidator
from travel.rasp.admin.scripts.schedule.utils.times_approximation import fill_middle_times
from travel.rasp.admin.scripts.utils.file_wrapper.uploaders import get_schedule_upload_wrapper
from travel.rasp.admin.scripts.utils.import_file_storage import get_schedule_temporary_date_filepath, get_schedule_filepath


log = logging.getLogger(__name__)


script_name = os.path.basename(__file__)


STATIONS_FILENAME = 'stops.xml'
ROUTES_FILENAME = 'schedule.xml'

MTA_HOST = 'granny.homedns.org'
MTA_USER = 'Yandex'
MTA_PASSWORD = 'MTA_rasp'
DAYS_FORWARD = 300


class MTAStation(Finder2SupplierStation):
    generalize_station_titles = [u'вокзал', u'автовокзал', u'площадь', u'автостанция']

    def __init__(self, element, context, code=None):
        self.context = context
        self.element = element
        self._code = code

        if element is None:
            if not code:
                raise RaspImportError(u'Не указан ни элемент, ни код')

            self._code = code
            self._title = ''
        else:
            self._code = self.element.get('stop_code').strip()
            self._title = self.element.get('stop_name').strip()

        self.supplier = Supplier.objects.get(code='mta')
        self.t_bus = TransportType.objects.get(code='bus')
        self.station_majority = StationMajority.objects.get(code='station')
        self.exclude_station_majority = StationMajority.objects.get(code='exclude_from_path')

    def get_code(self):
        return self._code

    def get_title(self):
        return self._title

    def get_settlement_region_district(self):
        settlement = None
        region = Region.objects.get(pk=1)
        district = None

        if self.element is None:
            return settlement, region, district

        try:
            settlement_mapping = MTASettlementMapping.objects.get(area_code=self.element.get('area_code'),
                                                                  stop_city=self.element.get('stop_city'))
            settlement = settlement_mapping.settlement

            if settlement:
                region = settlement.region

        except MTASettlementMapping.DoesNotExist:
            pass

        try:
            district_mapping = MTADistrictMapping.objects.get(area_code=self.element.get('area_code'))
            district = district_mapping.district
            if not region:
                region = district_mapping.region
        except MTADistrictMapping.DoesNotExist:
            pass

        if not settlement and not region:
            log.error(u"Нет привязок для станции %s", ElementTree.tostring(self.element, 'utf8').decode('utf8'))

        return settlement, region, district

    def get_existed_station(self):
        params = self.get_stops().get(self.code)
        if not params:
            return None

        if params['station_id']:
            return Station.objects.get(id=int(params['station_id']))

    def create_station(self):
        settlement, region, district = self.get_settlement_region_district()

        station = Station(t_type=self.t_bus, title=self.title)
        station.majority = self.station_majority
        station.settlement = settlement
        station.country = settlement and settlement.country
        station.region = settlement and settlement.region
        station.time_zone = (settlement and settlement.time_zone) or (region and region.time_zone)
        station.region = region
        station.district = district
        station.not_generalize = station.title.lower() not in self.generalize_station_titles

        if not station.region:
            station.region = Region.objects.get(id=1)  # московская область
            station.time_zone = station.region.time_zone

        self.raise_station_majority(station)

        self.override_attributes_from_file(station)

        station.save()

        # Старый кеш аттрибутов settlement, district и т.п., несет проблемы в дальнейшем, это конечно костыль.
        return Station.objects.get(pk=station.id)

    def raise_station_majority(self, station):
        if (station.settlement and station.settlement.title.lower() == station.title.lower() and
                station.settlement.majority_id == 5):  # Поселок или деревня
            station.majority_id = 1  # Главная в городе

    def override_attributes_from_file(self, station):
        params = self.get_stops().get(self.code)
        if params:
            station.title = params['title']
            station.majority_id = int(params['majority_id'])
            station.not_generalize = params['not_generalize'] != u"1"  # Признаки нужно инвертировать

            if params['distinct_id']:
                station.district_id = int(params['distinct_id'])

            if params['delete'] == u"1":
                station.majority = self.exclude_station_majority

    _stops = {}

    @classmethod
    def get_stops(cls):
        if cls._stops:
            return cls._stops

        supplier = Supplier.objects.get(code='mta')

        reader = UnicodeDictReader(StringIO(resource.find('www/db/scripts/data/schedule/mta/mtastop3.csv')),
                                   delimiter=';', encoding='cp1251', strip_values=True)
        for row in reader:
            if row['code']:
                cls._stops[row['code']] = row

        return cls._stops


class MtaMappingContext(SupplierMappingContext):
    """
    Учитываем только route_number, route_title, first_station_code, last_station_code
    """
    def __init__(self, route_data, thread_data):
        self.route_number = route_data['number']
        self.route_title = route_data['title']

        rtstations = thread_data['rtstations']

        self.first_station_code = rtstations[0]['station_code']
        self.last_station_code = rtstations[-1]['station_code']


class MtaImporter(object):
    def __init__(self, mta_data_provider):
        self.supplier = Supplier.objects.get(code='mta')
        self.t_bus = TransportType.objects.get(code='bus')
        self.basic_thread_type = RThreadType.objects.get(id=RThreadType.BASIC_ID)

        self.data_provider = mta_data_provider

        self.mask_builder = MaskBuilder(*import_bounds(back=7, forward=DAYS_FORWARD))

        self.finder = StationFinder2(self.supplier, log, searcher_class=SF2ContextSearcher)

        self.updater = SupplierRouteUpdater.create_route_updater(
            self.supplier, log=log, saver=BulkRouteSaver(log=log),
            thread_number_builder=CompactThreadNumberBuilder, remove_old_routes=True
        )
        self.validator = ScheduleValidator(log)
        self.validator.add_thread_validator(simple_thread_validator)
        self.updater.add_presave_thread_hook(self.validator.validate_thread)

    @cache_method_result_with_exception
    def get_station(self, code, context):
        mta_station = self.get_mta_station(code, context)
        log.info(u"Ищем станцию %s", mta_station)
        return self.finder.find(mta_station)

    def get_mta_station(self, code, context):
        stop_tag = self.data_provider.get_stop_tag_by_code(code)
        if stop_tag is None:
            log.error(u"На нашли станцию %s в stops.xml", code)
            return MTAStation(None, context, code=code)
        else:
            return MTAStation(stop_tag, context)

    def do_import(self):
        self.clean_original_data()

        routes = {}
        for route_data in self.data_provider.get_routes_data():
            try:
                number = route_data['number']
                log.info(u"Разбираем маршрут %s %s", number, route_data['title'])
                log.set_context(u"%s %s" % (number, route_data['title']))

                threads = self.parse_threads(route_data)
                for thread in threads:
                    route_uid = thread.gen_route_uid(use_stations=True)
                    if route_uid not in routes:
                        route = Route(t_type=self.t_bus, supplier=self.supplier, route_uid=route_uid)
                        route.threads = []
                        routes[route_uid] = route
                    self.add_thread_to_route(routes[route_uid], thread)
            except Exception:
                log.exception(u"Ошибки ипорта маршрута %s %s", route_data['number'],
                              route_data['title'])
            finally:
                log.set_context(u"")

        for route in routes.itervalues():
            self.updater.add_route(route)

            for thread in route.threads:
                self.update_orig_data(thread)

        self.updater.update()
        self.bind_original_data()

    def add_thread_to_route(self, route, thread):
        thread.route = route
        thread.gen_title()
        thread.gen_import_uid()

        matched_thread = next((_t for _t in route.threads if is_equal_threads(thread, _t)), None)
        if matched_thread:
            matched_thread.mask = matched_thread.mask | thread.mask
            matched_thread.year_days = str(matched_thread.mask)
            matched_thread.raw_data += u'\n\n' + thread.raw_data
        else:
            route.threads.append(thread)

    def parse_threads(self, route_data):
        thread_datas = route_data['threads']

        threads = []
        for thread_data in thread_datas:
            thread = self.parse_thread(route_data, thread_data)
            if thread:
                threads.append(thread)

        return threads

    def parse_thread(self, route_data, thread_data):
        mask = self.make_mask(thread_data)
        if not mask:
            log.warning(u"Нитка с пустыми днями хождения start=%s end=%s days=%s", thread_data['period_start'],
                        thread_data['period_end'], thread_data['days'])
            return

        context = MtaMappingContext(route_data, thread_data)
        rts_parser = MTARTStationParser(thread_data, self, mask.dates()[0], context)
        try:
            rts_parser.parse()
        except RaspImportError, e:
            log.error(unicode(e))
            return

        rtstations = rts_parser.rtstations
        self.fill_middle_times(rtstations)

        if len(rtstations) < 2:
            log.error(u"В нитке только одна станция, пропукаем нитку")
            return

        thread = RThread(
            type=self.basic_thread_type,
            number=route_data['number'],
            year_days=str(mask),
            tz_start_time=rts_parser.start_time,
            time_zone=rtstations[0].time_zone,
            t_type=self.t_bus,
            supplier=self.supplier,
        )
        thread.mask = mask
        thread.rtstations = rtstations
        thread.raw_data = thread_data['raw_data']

        return thread

    def make_mask(self, thread_data):
        period_start = datetime.strptime(thread_data['period_start'], '%d-%m-%Y').date()
        period_end = datetime.strptime(thread_data['period_end'], '%d-%m-%Y').date()
        days = thread_data['days'].replace(u"-", u"")

        return self.mask_builder.mask_from_days_of_week(days, period_start, period_end)

    def fill_middle_times(self, rtstations):
        self.fill_departure_and_arrival(rtstations)

        fill_middle_times(rtstations, ignore_bad_duration=True)

        self.calc_tz_departure_and_tz_arrival(rtstations)
        self.unset_departure_and_arrival(rtstations)

    def fill_departure_and_arrival(self, rtstations):
        for rts in rtstations:
            rts.departure = rts.tz_departure
            rts.arrival = rts.tz_arrival

    def calc_tz_departure_and_tz_arrival(self, rtstations):
        for rts in rtstations:
            rts.tz_departure = rts.departure
            rts.tz_arrival = rts.arrival

    def unset_departure_and_arrival(self, rtstations):
        for rts in rtstations:
            rts.departure = None
            rts.arrival = None

    def clean_original_data(self):
        OriginalThreadData.fast_delete_by_supplier_id(self.supplier.id, log)

    def bind_original_data(self):
        for id_, uid in (RThread.objects.filter(route__two_stage_package=None, supplier=self.supplier)
                                .values_list('id', 'uid')):
            OriginalThreadData.objects.filter(thread_uid=uid).update(thread=id_)

    def update_orig_data(self, thread):
        # TODO: Тут точно можно ускорить, например, собирать все кусочки отдельно и записывать один раз
        orig_data, __ = OriginalThreadData.objects.get_or_create(
            thread_uid=thread.uid,
            package=None,
            supplier=self.supplier
        )
        orig_data.raw = (orig_data.raw or u'') + thread.raw_data
        orig_data.save()


def is_equal_threads(thread1, thread2):
    assert thread1.import_uid and thread2.import_uid

    return thread1.import_uid == thread2.import_uid


class MTARTStationParser(object):
    start_time = None

    def __init__(self, thread_data, importer, first_day, context):
        self.thread_data = thread_data
        self.importer = importer
        self.first_day = first_day
        self.rtstations = []
        self.context = context
        self.exclude_station_majority = StationMajority.objects.get(code='exclude_from_path')

    def parse(self):
        if len(self.thread_data['rtstations']) < 1:
            raise RaspImportError(u"Не хватает станций в нитке")

        self.start_time = datetime.strptime(self.thread_data['rtstations'][0]['time'], '%H:%M:%S').time()

        first_rts_data = self.thread_data['rtstations'][0]
        first_rts = RTStation()
        first_rts.station = self.importer.get_station(first_rts_data['station_code'], self.context)

        time_zone = first_rts.station.time_zone

        first_rts.time_zone = time_zone
        first_rts.tz_arrival = None
        first_rts.tz_departure = 0

        self.rtstations.append(first_rts)

        previous_station = first_rts.station

        start_dt = datetime.combine(self.first_day, self.start_time)
        prev_dt = start_dt
        prev_minutes = 0

        for rts_data in self.thread_data['rtstations'][1:]:
            rts = RTStation()
            rts.time_zone = time_zone

            try:
                rts.station = self.importer.get_station(rts_data['station_code'], self.context)
            except KeyError:
                log.error(u"Не нашли станции с кодом %s в stops.xml пропускаем", rts_data['station_code'])
                continue

            if rts.station == previous_station:
                log.warning(u"Станция %s встречалась на предыдущей остановке, пропускаем",
                            rts.station.title)
                continue

            previous_station = rts.station

            # Почему эта проверка после previous_station = rts.station
            if rts.station.majority == self.exclude_station_majority:
                continue

            departure_time = datetime.strptime(rts_data['time'], '%H:%M:%S').time()

            departure_dt = datetime.combine(self.first_day, departure_time)

            if departure_dt < prev_dt:
                departure_dt += timedelta(1)

            # приемлемое время между станциями
            if timedelta2minutes(departure_dt - prev_dt) < 60 * 12:

                prev_dt = departure_dt

                rts.tz_departure = int(timedelta2minutes(departure_dt - start_dt))

                rts.tz_arrival = rts.tz_departure - 1

                if rts.tz_arrival <= prev_minutes:
                    log.info(u"Слишком близко время старта на %s, время в пути %s",
                             rts.station.title, rts.tz_arrival - prev_minutes)

                prev_minutes = rts.tz_departure

            self.rtstations.append(rts)

        last_station_code = self.thread_data['rtstations'][-1]['station_code']

        try:
            last_station = self.importer.get_station(last_station_code, self.context)
        except KeyError:
            raise RaspImportError(u"Не нашли последней станции по коду %s" % last_station_code)

        if self.rtstations[0].station.majority == self.exclude_station_majority:
            raise RaspImportError(u"Должна быть исключена первая станция из маршрута")

        if last_station.majority == self.exclude_station_majority:
            raise RaspImportError(u"Должна быть исключена последняя станция из маршрута")

        self.rtstations[-1].tz_arrival = self.rtstations[-1].tz_departure

        last_bad_rtstations = []
        # Считаем первую остановку хорошей в любом случае RASPADMIN-792
        for rts in self.rtstations[:0:-1]:
            if rts.tz_arrival is None:
                last_bad_rtstations.append(rts)
            else:
                break

        last_bad_rtstations.reverse()

        if last_bad_rtstations:
            if len(last_bad_rtstations) > 1:
                for rts in last_bad_rtstations:
                    rts.tz_arrival = prev_minutes + 5
                    rts.tz_departure = rts.tz_arrival + 1
                    prev_minutes = rts.tz_arrival

            else:
                self.rtstations[-1].tz_arrival = prev_minutes + 10

        self.rtstations[-1].tz_departure = None

        return self.rtstations


class FileMtaDataProvider(object):
    schedule_file_path = None
    stops_file_path = None

    def __init__(self, schedule_file_path, stops_file_path):
        self.schedule_file_path = schedule_file_path
        self.stops_file_path = stops_file_path

    @cache_method_result
    def get_stops(self):
        stops = dict()
        try:
            for _, element in ElementTree.iterparse(self.get_stops_file_path()):
                if element.tag == 'stop':
                    stops[element.get('stop_code', u"").strip()] = element
        except ParseError:
            pass

        return stops

    def get_stop_tag_by_code(self, code):
        return self.get_stops().get(code)

    def get_stops_file_path(self):
        return self.stops_file_path

    def get_schedule_file_path(self):
        return self.schedule_file_path

    def get_routes_data(self):
        excluded_internal_numbers = MTAInternalNumberBlackList.get_regexp_set()
        for __, element in ElementTree.iterparse(self.get_schedule_file_path()):
            if element.tag == 'route':
                route = {}

                internal_number = element.get('internal_number', u"").strip()

                route['number'] = element.get('route_number')
                route['title'] = element.get('route_title')
                route['threads'] = self.get_threads(element)

                if internal_number in excluded_internal_numbers:
                    log.info(u"Пропускаем маршрут %s %s, у него внутренний номер %s", route['number'],
                             route['title'], internal_number)
                    continue
                else:
                    log.info(u"Обрабатываем маршрут %s %s, с внутренним номером %s", route['number'],
                             route['title'], internal_number)

                yield route

    def get_threads(self, route_element):
        threads = []
        for element in route_element.findall('./reis'):
            threads.append({
                'period_'
                'start': element.get('period_start'),
                'period_end': element.get('period_end'),
                'days': element.get('days_of_week'),
                'rtstations': self.get_rtstations(element),
                'raw_data': u'\n'.join((
                    ElementTree.tostring(get_top_element_only(route_element), encoding='utf-8').decode('utf-8'),
                    ElementTree.tostring(element, encoding='utf-8').decode('utf-8'),
                ))
            })

        return threads

    def get_rtstations(self, element):
        rtstations = []
        for element in element.findall('./reis_point'):
            rtstations.append({
                'station_code': element.get('stop_code'),
                'time': element.get('time')
            })

        return rtstations


class FtpMtaDataProvider(FileMtaDataProvider):
    schedule_archive_filename = 'MTA-rasp.7z'
    schedule_archive_filepath = None

    stops_archive_filename = 'stops.zip'
    stops_archive_filepath = None

    tmp_dir = os.path.join("/tmp/yandex/rasp/mta/")

    def __init__(self, take_stations_from_stopsxml=False):
        self.supplier = Supplier.objects.get(code='mta')
        self.take_stations_from_separated_file = take_stations_from_stopsxml
        self.all_stations = {}

    def get_schedule_file_path(self):
        if self.schedule_file_path:
            return self.schedule_file_path

        self.prepare_schedule_file()

        return self.schedule_file_path

    def get_stops_file_path(self):
        if self.stops_file_path:
            return self.stops_file_path

        if self.take_stations_from_separated_file:
            self.prepare_stops_file()
        else:
            self.prepare_schedule_file()

        return self.stops_file_path

    def get_schedule_archive_file_path(self):
        return get_schedule_temporary_date_filepath(self.schedule_archive_filename, self.supplier)

    def prepare_schedule_file(self):
        schedule_archive_filepath = self.get_schedule_archive_file_path()
        if not os.path.exists(schedule_archive_filepath):
            self.schedule_archive_filepath = self.download_file(self.schedule_archive_filename,
                                                                schedule_archive_filepath)
        else:
            self.schedule_archive_filepath = schedule_archive_filepath
            log.info(u"Файл уже есть %s", schedule_archive_filepath)

        filedict = get_7zip_filedict(self.schedule_archive_filepath, os.path.join(self.tmp_dir, 'schedule'), log)

        self.schedule_file_path = filedict[ROUTES_FILENAME]

        if not self.take_stations_from_separated_file:
            self.stops_file_path = filedict[STATIONS_FILENAME]

    def get_stops_archive_file_path(self):
        return get_schedule_temporary_date_filepath(self.stops_archive_filename, self.supplier)

    def prepare_stops_file(self):
        stops_archive_filepath = self.get_stops_archive_file_path()
        if not os.path.exists(stops_archive_filepath):
            self.stops_archive_filepath = self.download_file(self.stops_archive_filename, stops_archive_filepath)
        else:
            self.stops_archive_filepath = stops_archive_filepath
            log.info(u"Файл уже есть %s", stops_archive_filepath)

        filedict = unpack_zip_file(self.stops_archive_filepath, os.path.join(self.tmp_dir, 'stops'))

        self.stops_file_path = filedict[STATIONS_FILENAME]

    def download_file(self, filename, filepath):
        log.info(u"Скачиваем файл %s в %s", filename, filepath)
        ftp = FTP(MTA_HOST, "Yandex", "MTA_rasp", timeout=settings.SCHEDULE_IMPORT_TIMEOUT)
        f = open(filepath, "w")

        try:
            ftp.retrbinary("RETR " + filename, f.write, 1024)
            ftp.close()
            f.close()
        except Exception:
            os.remove(filepath)
            raise

        log.info(u"Скачали файл %s в %s", filename, filepath)
        return filepath

    def clean(self):
        remove_tmp_dir(self.tmp_dir)
        self.schedule_archive_filepath = None
        self.stops_archive_filepath = None
        self.stops_file_path = None
        self.schedule_file_path = None


class SupplierFileDataProvider(FileMtaDataProvider):
    tmp_dir = os.path.join("/tmp/yandex/rasp/mta/")

    def __init__(self):
        self.all_stations = {}
        self.supplier = Supplier.objects.get(code='mta')
        self.schedule_file_zip = os.path.join(settings.SCHEDULE_UPLOAD, self.supplier.filename)

    def get_schedule_file_path(self):
        if self.schedule_file_path:
            return self.schedule_file_path

        self.unzip_supplier_file()

        return self.schedule_file_path

    def get_stops_file_path(self):
        if self.stops_file_path:
            return self.stops_file_path

        self.unzip_supplier_file()

        return self.stops_file_path

    def unzip_supplier_file(self):
        file_wrapper = get_schedule_upload_wrapper(self.schedule_file_zip)
        file_wrapper.download()

        files = unpack_zip_file(self.schedule_file_zip)

        if 'stops.xml' not in files:
            raise RaspImportError(u"Не нашли файла stops.xml в списке")

        if 'schedule.xml' not in files:
            raise RaspImportError(u"Не нашли файла schedule.xml в списке")

        self.stops_file_path = files['stops.xml']
        self.schedule_file_path = files['schedule.xml']


def import_mta_from_ftp(take_stations_from_stopsxml=False):
    data_provider = FtpMtaDataProvider(take_stations_from_stopsxml)

    import_mta(data_provider)


def import_mta_from_files(schedule_file, stops_file):
    data_provider = FileMtaDataProvider(schedule_file, stops_file)

    import_mta(data_provider)


def import_mta(data_provider):
    supplier = Supplier.objects.get(code='mta')

    importer = MtaImporter(data_provider)
    try:
        importer.do_import()
    finally:
        ScheduleImportReporter.send_bus_report(supplier, importer.finder, importer.validator)


def import_mta_from_supplier_file(download=False):
    data_provider = SupplierFileDataProvider()
    log.info(u"Загрузка из zip-файла %s", data_provider.schedule_file_zip)
    data_provider.unzip_supplier_file()
    log.info(u"Загрузка из файлов %s, %s", data_provider.stops_file_path, data_provider.schedule_file_path)
    import_mta(data_provider)


import_launcher = import_mta_from_supplier_file


def advance_checks_launcher():
    try:
        data_provider = SupplierFileDataProvider()
        if (data_provider.get_schedule_file_path() and
                data_provider.get_stops_file_path()):
            log.info(u"Все необходимые файлы есть в архиве")
            return True
        else:
            return False
    except RaspImportError, e:
        log.error(unicode(e))

    return False


def get_settlement_csv():
    provider = FtpMtaDataProvider()
    cities = set()
    try:
        for _, element in ElementTree.iterparse(provider.get_stops_file_path()):
            city = element.get('stop_city')
            cities.add(city)
    finally:
        provider.clean()

    fields = ('title', 'id', 'settlement_region')

    out_file = open('mta_settlements.csv', 'w')

    writer = UnicodeDictWriter(out_file, fields, delimiter=';')
    writer.writehead()

    for city in cities:
        rowdict = {'title': city}
        try:
            settlement = Settlement.objects.get(title=city)
            rowdict['id'] = settlement.id
            rowdict['settlement_region'] = settlement.region and settlement.region.title
        except Settlement.MultipleObjectsReturned:
            settlements = Settlement.objects.filter(title=city)
            rowdict['id'] = u", ".join(unicode(s.id) for s in settlements)
            rowdict['settlement_region'] = u", ".join(s.region and s.region.title or u"" for s in settlements)
        except Settlement.DoesNotExist:
            pass

        writer.writerow(rowdict)

    out_file.close()


def get_top_element_only(el):
    el = ElementTree.fromstring(ElementTree.tostring(el))
    del el[:]

    return el


usage = u"Usage: python %prog [options]"


# Если файл запустили из консоли парсим параметры и запускаем скрипт
if __name__ == '__main__':
    with ylog_context(**get_script_log_context()):
        create_current_file_run_log()

        parser = ArgumentParser(usage=usage, description=__doc__)
        parser.add_argument('-v', '--verbose', action="store_true",
                            help=u'выводить лог на экран')
        parser.add_argument('-s', '--get-settlement-csv', action="store_true")
        parser.add_argument('-t', '--take-stations-from-stopsxml', action="store_true",
                            help=u'Скачивать stops.xml для станций')
        parser.add_argument('-f', '--import-form-files', help=u'Импортировать из schedule_file, stops_file')
        parser.add_argument('-d', '--import-from-default-file', action="store_true",
                            help=u'Импортировать файлы в пути по умолчанию')
        args = parser.parse_args()

        if args.verbose:
            print_log_to_stdout()

        if args.get_settlement_csv:
            get_settlement_csv()
        elif args.import_form_files:
            import_mta_from_files(args.import_form_files[1], args.import_form_files[2])
        elif args.import_from_default_file:
            import_mta_from_supplier_file()
        else:
            import_mta_from_ftp(args.take_stations_from_stopsxml)


