# coding: utf-8

"""
Данные поставщика - xml,
основной информационный элемент - Table - содержит, в основном, данные точка-точка.

В элементе Table в тэге GuzergahSaat записан весь маршрут, но к сожалению, в большенстве
случаев с неправильными временами, поэтому эти времена не используем.

Алгорим построения нитки такой:

1. Берем маршрут из GuzergahSaat, но без времен и ставим времена из пар точка-точка.
   Используем скорректированный вручную порядок остановок (при наличии).

2. Выкидываем станции без времен (если такие есть)

3. При этом, набор данных одного маршрута может оказаться несовместимым по той или иной причине,
   В этом случае импортируем данные точка-точка.

4. Импортируем данные, которые не вошли в целые маршруты, как данные точка-точка.
   То есть любой элемент Table импортируется либо в составе маршрута, или отдельно
   как точка-точка.
"""

import codecs
import json
import hashlib
import re
import os.path
from collections import defaultdict, OrderedDict
from contextlib import closing
from datetime import datetime, timedelta, date
from ftplib import FTP
from pprint import pformat

import dateutil.parser
import pytz
from django.conf import settings
from lxml import etree

from common.cysix.builder import ChannelBlock, GroupBlock, ThreadBlock, StoppointBlock, ScheduleBlock
from travel.rasp.library.python.common23.date import environment

from cysix.base import safe_parse_xml
from cysix.tsi_converter import CysixTSIConverterFileProvider, CysixTSIConverterFactory
from django.utils.functional import cached_property
from travel.rasp.admin.importinfo.models.ipektur import IpekturStopsOrder
from common.utils.caching import cache_method_result
from travel.rasp.admin.lib.logs import get_current_file_logger
from travel.rasp.admin.lib.unpack_with_fallback import unpack
from travel.rasp.admin.lib.xmlutils import get_sub_tag_text
from travel.rasp.admin.scripts.schedule.utils.file_providers import PackageFileProvider
from travel.rasp.admin.scripts.schedule.utils import RaspImportError
from travel.rasp.admin.scripts.utils.import_file_storage import get_schedule_temporary_date_filepath
from common.utils.date import timedelta2minutes


log = get_current_file_logger()

FTP_HOST = 'ftp.ipekbilgisayar.com.tr'
FTP_USER = 'yandex'
FTP_PASSWORD = 'yan*dex5'

CURRENCY = 'TRY'

TIMEZONE = 'Asia/Istanbul'

SCHEDULE_DAYS = 30

SAFE_DIFF_MINUTES = 1

CHAR_MAPPING = {
    ord(u'ı'): ord(u'i'),
    ord(u'ü'): ord(u'u'),
    ord(u'ö'): ord(u'o'),
    ord(u'ğ'): ord(u'g'),
    ord(u'ç'): ord(u'c'),
    ord(u'ş'): ord(u's'),
}


class IpekturCysixFactory(CysixTSIConverterFactory):
    def get_raw_download_file_provider(self):
        return IpekturFTPFileProvider(self.package)

    def get_raw_package_file_provider(self):
        return IpekturPackageFileProvider(self.package)

    def get_converter_file_provider(self, raw_file_provider):
        return IpekturCysixFileProvider(self.package, raw_file_provider)


class IpekturCysixFileProvider(CysixTSIConverterFileProvider):
    def convert_data(self, filepath):
        c = Converter(self.raw_file_provider)

        c.convert(filepath)


class ConstructThreadError(RaspImportError):
    pass


class Converter(object):
    def __init__(self, provider):
        self.provider = provider

        self.stations = dict()
        self.raw_threads_by_warning = defaultdict(list)
        self.unused_raw_tables = []

        self.number_of_skipped_routes = 0

        today = environment.today()
        self.days = ';'.join([(today + timedelta(days=i)).strftime('%Y-%m-%d')
                              for i in xrange(SCHEDULE_DAYS)])

        self.stops_order_substitutions = IpekturStopsOrder.get_substitutions_dict()

    def convert(self, filepath):
        raw_data = self.provider.get_raw_data()

        channel_el = self.build_channel_element(raw_data)

        with open(filepath, 'w') as f:
            f.write(etree.tostring(
                channel_el, xml_declaration=True, encoding='utf-8', pretty_print=True
            ))

    def build_channel_element(self, raw_data):
        channel_block = ChannelBlock(
            'bus',
            station_code_system='vendor',
            carrier_code_system='vendor',
            vehicle_code_system='local',
            timezone=TIMEZONE
        )

        raw_threads = defaultdict(list)

        for filepath in raw_data.get_filepaths():
            root_el = self.get_root_el(filepath)

            if root_el is None:
                continue

            for raw_table_el in root_el.findall('Table'):
                raw_threads[get_key(raw_table_el)].append(raw_table_el)

        for number, raw_table_els in enumerate(raw_threads.values()):
            stop_order_filepath = get_schedule_temporary_date_filepath(
                '%s.txt' % number,
                self.provider.package
            )

            self.build_raw_thread(raw_table_els, stop_order_filepath)

        group_block = GroupBlock(channel_block, title='routes', code='routes')

        for warning_code, raw_threads in self.raw_threads_by_warning.iteritems():
            log.info(u'Warning code %s, number of routes %s', warning_code, len(raw_threads))

            for raw_thread in raw_threads:
                self.build_full_thread(raw_thread, group_block)

        log.info(u'Number of skipped routes %s', self.number_of_skipped_routes)

        channel_block.add_group_block(group_block)

        p2p_group_block = GroupBlock(channel_block, title='p2p', code='p2p')

        for raw_table in self.unused_raw_tables:
            self.build_p2p_thread(raw_table, p2p_group_block)

        channel_block.add_group_block(p2p_group_block)

        return channel_block.get_element()

    def build_raw_thread(self, raw_table_els, stops_order_filepath):
        try:
            raw_thread = RawThread(raw_table_els, self.stops_order_substitutions, stops_order_filepath)

        except ConstructThreadError, e:
            thread_description = u'"{}" firm "{} {}"'.format(
                get_sub_tag_text(raw_table_els[0], 'HatNo'),
                get_sub_tag_text(raw_table_els[0], 'FirmaNo', silent=True),
                get_sub_tag_text(raw_table_els[0], 'FirmaAdi', silent=True)
            )

            log.error(u"Can't construct full thread %s, skip. %s",
                      thread_description, e)

            self.number_of_skipped_routes += 1

            self.unused_raw_tables.extend(RawTable(table_el) for table_el in raw_table_els)

        else:
            self.raw_threads_by_warning[raw_thread.warning_code.code].append(raw_thread)

            self.unused_raw_tables.extend(raw_thread.unused_tables)

    def build_full_thread(self, raw_thread, group_block):
        thread_block = ThreadBlock(group_block, number=raw_thread.get_number())

        start_dt = raw_thread.stops[0].departure

        thread_block.add_schedule_block(ScheduleBlock(thread_block, self.days,
                                                      times=start_dt.strftime('%H:%M:%S')))

        for stop in raw_thread.stops:
            code = raw_thread.get_station_code(stop.station)

            station_block = self.get_station_block(group_block, code, stop.station)

            stoppoint_block = StoppointBlock(
                thread_block,
                station_block,
                departure_shift=stop.get_departure_shift(start_dt),
                arrival_shift=stop.get_arrival_shift(start_dt)
            )

            thread_block.add_stoppoint_block(stoppoint_block)

        # fares
        fare_block = group_block.add_local_fare()

        for table in raw_thread.raw_tables:
            price, order_data = table.get_price_and_order_data()

            if price and price > 0:
                departure_code = raw_thread.get_station_code(table.departure_station)
                departure_block = self.get_station_block(
                    group_block, departure_code, table.departure_station
                )

                arrival_code = raw_thread.get_station_code(table.arrival_station)
                arrival_block = self.get_station_block(
                    group_block, arrival_code, table.arrival_station
                )

                fare_block.add_price_block('%.2f' % price, CURRENCY,
                                           departure_block, arrival_block, data=order_data)

                thread_block.set_fare_block(fare_block)

        # other
        thread_block.set_raw_data(raw_thread.get_raw_data())
        self.add_carrier_to_thread(raw_thread, group_block, thread_block)
        self.add_t_model_to_thread(raw_thread, group_block, thread_block)

        group_block.add_thread_block(thread_block)

    def build_p2p_thread(self, table, group_block):
        thread_block = ThreadBlock(group_block, number=table.get_number())

        start_dt = table.departure_dt

        thread_block.add_schedule_block(ScheduleBlock(thread_block, self.days,
                                                      times=start_dt.strftime('%H:%M:%S')))
        # stoppoints
        departure_block = self.get_station_block(
            group_block, table.get_departure_code(), table.departure_station)

        thread_block.add_stoppoint_block(StoppointBlock(
            thread_block, departure_block, departure_shift=0,
            in_station_schedule=0
        ))

        arrival_block = self.get_station_block(
            group_block, table.get_arrival_code(), table.arrival_station)

        thread_block.add_stoppoint_block(StoppointBlock(
            thread_block, arrival_block, arrival_shift=table.duration))

        # fares
        fare_block = group_block.add_local_fare()

        price, order_data = table.get_price_and_order_data()

        if price and price > 0:
            fare_block.add_price_block('%.2f' % price, CURRENCY,
                                       departure_block, arrival_block, data=order_data)

        thread_block.set_fare_block(fare_block)

        # other
        thread_block.set_raw_data(table.get_raw_data())
        self.add_carrier_to_thread(table, group_block, thread_block)
        self.add_t_model_to_thread(table, group_block, thread_block)

        group_block.add_thread_block(thread_block)

    def add_carrier_to_thread(self, raw, group_block, thread_block):
        carrier_code, carrier_title, phone = raw.get_carrier_code_title_phone()

        if carrier_code and carrier_title:
            thread_block.carrier = group_block.add_carrier(carrier_title, carrier_code, phone=phone)

    def add_t_model_to_thread(self, raw, group_block, thread_block):
        thread_block.vehicle = group_block.add_local_vehicle(raw.get_transport_model_title())

    def get_station_block(self, group_block, code, title):
        station_block = self.stations.get(code, None)

        if station_block is None:
            station_block = group_block.add_station(title.title, code)

            station_block.add_legacy_station(title.title, '')

            self.stations[code] = station_block

        return station_block

    def get_root_el(self, filepath):
        try:
            return safe_parse_xml(filepath).getroot()

        except etree.LxmlError, e:
            log.error(u'Error parsing xml-file %s. %s', filepath, e)

            return None


class RawThread(object):
    def __init__(self, raw_table_els, stops_order_substitutions, stops_order_filepath):
        self.raw_table_els = raw_table_els
        self.stops_order_substitutions = stops_order_substitutions
        self.stops_order_filepath = stops_order_filepath

        self.warning_code = WarningCode()  # сгруппируем нитки в зависимости от качества данных

        self.raw_tables = [RawTable(table_el) for table_el in raw_table_els]

        self.stops = self._init_stops(self.raw_tables[0])

        self._filter_raw_tables()

        if not self.raw_tables:
            raise ConstructThreadError(u'There is no valid data for building route.')

        self._build_stops_times()

        self._drop_stops_without_time()

        if len(self.stops) < 2:
            raise ConstructThreadError(u'Not enough stops for building route.')

        self.unused_tables = self._get_unused_tables()

    def _init_stops(self, table):
        original_stops = IpekturStops(get_sub_tag_text(table.table_el, 'GuzergahSaat'))

        stops = original_stops.get_stops()

        if not stops:
            raise ConstructThreadError(u'Empty GuzergahSaat tag.')

        key = tuple(original_stops.get_canonical_titles())

        right_stops = [RawStop(title) for title in self.stops_order_substitutions.get(key, list())]

        # если еще не знаем правильного порядка, то сохраним файл для обработки порядка вручную
        if not right_stops:
            write_raw_tables_summary(self.raw_table_els, self.stops_order_filepath)

        return right_stops or stops

    def _filter_raw_tables(self):
        stations = [stop.station for stop in self.stops]

        old_length = len(self.raw_tables)

        self.raw_tables = [raw_table for raw_table in self.raw_tables
                           if raw_table.arrival_station in stations]

        if old_length != len(self.raw_tables):
            self.warning_code.add('has_wrong_table')

    def _build_stops_times(self):
        first_stop = None  # будет заполнен, если отправления нет в маршруте.

        for table in self.raw_tables:
            # departure
            try:
                stop = next((x for x in self.stops if x.station == table.departure_station))
                stop.departures.append(table.departure_dt)

            except StopIteration:
                if first_stop is None:
                    first_stop = RawStop(table.departure_station, is_originaly_in_route=False)

                else:
                    if first_stop.station != table.departure_station:
                        raise ConstructThreadError(
                            u'There is more than one departure that not listed in GuzergahSaat tag'
                        )

                first_stop.departures.append(table.departure_dt)

            # arrival
            stop = next((x for x in self.stops if x.station == table.arrival_station))
            stop.arrivals.append(table.arrival_dt)

        if first_stop is not None:
            self.stops.insert(0, first_stop)

            self.warning_code.add('first_stop_not_in_GuzergahSaat')

        for stop in self.stops:
            stop.validate_and_build_times()

        if any(stop.has_different_arrivals for stop in self.stops):
            self.warning_code.add('has_different_arrivals')

    def _drop_stops_without_time(self):
        """
        Остановки без времени приводят к fuzzy-time.
        Туркам fuzzy-time хуже отсутствия остановки, спровите Уура.
        """

        old_length = len(self.stops)

        self.stops = [stop for stop in self.stops if stop.departure or stop.arrival]

        if old_length != len(self.stops):
            self.warning_code.add('has_stops_without_time')

    def get_station_code(self, title):
        titles = [title] + [stop.station for stop in self.stops]

        return hashlib.sha256(u'\n'.join(map(unicode, titles)).encode('utf8')).hexdigest()[:50]

    def get_raw_data(self):
        return u'\n'.join([r.get_raw_data() for r in self.raw_tables])

    def get_carrier_code_title_phone(self):
        return self.raw_tables[0].get_carrier_code_title_phone()

    def get_transport_model_title(self):
        return self.raw_tables[0].get_transport_model_title()

    def get_number(self):
        return self.raw_tables[0].get_number()

    def _get_unused_tables(self):
        unused_tables = []

        stations = [s.station for s in self.stops]

        for raw_table_el in self.raw_table_els:
            raw_table = RawTable(raw_table_el)

            if raw_table.arrival_station not in stations or raw_table.departure_station not in stations:
                unused_tables.append(raw_table)

        return unused_tables


class RawStop(object):
    def __init__(self, station, is_originaly_in_route=True):
        self.station = IpekturStationTitle(station)

        self.departure = None
        self.arrival = None

        self.departures = []
        self.arrivals = []

        self.has_different_arrivals = False

        self.is_originaly_in_route = is_originaly_in_route

    def validate_and_build_times(self):
        if self.departures:
            min_departure = min(self.departures)
            max_departure = max(self.departures)

            if not is_almost_the_same_dts(min_departure, max_departure):
                raise ConstructThreadError(u'Station %s. Different departure times.' % self.station)

            self.departure = min_departure

        if self.arrivals:
            min_arrival = min(self.arrivals)
            max_arrival = max(self.arrivals)

            if is_almost_the_same_dts(min_arrival, max_arrival):
                self.arrival = max_arrival

            else:
                log.warning(u'Station %s. Different arrival times, ignore them.', self.station)

                self.has_different_arrivals = True

        # если есть отправление, то игнорируем прибытие (оно как правило неточное)
        if self.departure is not None:
            self.arrival = None

    def get_departure_shift(self, start_dt):
        return self.get_shift_from_dt(start_dt, self.departure)

    def get_arrival_shift(self, start_dt):
        return self.get_shift_from_dt(start_dt, self.arrival)

    def get_shift_from_dt(self, start_dt, dt):
        return int((dt - start_dt).total_seconds()) if dt is not None else u''


def is_almost_the_same_dts(dt, dt2):
    if dt is None or dt2 is None:
        return True

    return abs(timedelta2minutes(dt - dt2)) <= SAFE_DIFF_MINUTES


class IpekturStops(object):
    def __init__(self, original_stops_order_text):
        self.original_stops_order_text = original_stops_order_text.strip()

    @cached_property
    def titles(self):
        titles = []

        for station_and_time in self.original_stops_order_text.split(u','):
            if not station_and_time:
                continue

            title, _ = station_and_time.split(u'|')
            titles.append(IpekturStationTitle(title))

        return titles

    def get_canonical_titles(self):
        return [title.title for title in self.titles]

    def get_stops(self):
        return [RawStop(title) for title in self.titles]


class RawTable(object):
    def __init__(self, table_el):
        self.table_el = table_el

        self.departure_station = IpekturStationTitle(table_el.find(u'KalkisYeri').text.strip())
        self.arrival_station = IpekturStationTitle(table_el.find(u'VarisYeri').text.strip())

        part_duration = self.get_duration()
        self.duration = int(part_duration.total_seconds()) if part_duration is not None else None

        self.departure_dt = get_tr_datetime(table_el.find(u'YerelInternetSaat').text.strip())

        self.arrival_dt = None

        if self.departure_dt is not None and part_duration is not None:
            self.arrival_dt = self.departure_dt + part_duration

    def get_price_and_order_data(self):
        tags_for_order_data = ('SeferTakipNo', 'FirmaNo', 'KalkisYeri', 'VarisYeri', 'YerelInternetSaat')

        try:
            price = float(get_sub_tag_text(self.table_el, 'BiletFiyatiInternet'))
            url = get_sub_tag_text(self.table_el, 'BiletallUrl', silent=True).strip()

            if not url:
                # если вернуть None в качестве order_data,
                # то при импорте с данными для покупки в order_data подставится
                # значение по-умолчанию, что не подходит для нашего случчая.
                return price, '{}'

            order_data = dict()

            for tag in tags_for_order_data:
                order_data[tag] = get_sub_tag_text(self.table_el, tag, silent=True).strip()

            return price, json.dumps(order_data, ensure_ascii=False, encoding='utf8')

        except (TypeError, ValueError):
            return None, None

    def get_duration(self):
        duration_data = get_sub_tag_text(self.table_el, 'SeyahatSuresi', silent=True)
        if duration_data:
            duration_data = duration_data[:19]
            basis = datetime(1900, 1, 1, 0, 0, 0)
            dt = datetime.strptime(duration_data, '%Y-%m-%dT%H:%M:%S')

            return dt - basis

    def get_departure_code(self):
        return self.get_station_code(self.departure_station)

    def get_arrival_code(self):
        return self.get_station_code(self.arrival_station)

    def get_station_code(self, title):
        titles = [title] + [self.departure_station, self.arrival_station]

        return hashlib.sha256(u'\n'.join(map(unicode, titles)).encode('utf8')).hexdigest()[:50]

    def get_raw_data(self):
        return etree.tostring(self.table_el, pretty_print=True,
                              xml_declaration=False, encoding=unicode)

    def get_carrier_code_title_phone(self):
        code = get_sub_tag_text(self.table_el, 'FirmaNo', silent=True).strip()
        title = get_sub_tag_text(self.table_el, 'FirmaAdi', silent=True).strip()
        phone = get_sub_tag_text(self.table_el, 'FirmaTelefon', silent=True).strip()

        if phone.startswith(u'0'):
            phone = u'+90' + phone[1:]

        return code, title, phone

    def get_transport_model_title(self):
        title = get_sub_tag_text(self.table_el, 'OtobusTipi', silent=True).strip()
        title += u' ' + get_sub_tag_text(self.table_el, 'OTipAciklamasi', silent=True).strip()
        title = title.strip()

        return title

    def get_number(self):
        return get_sub_tag_text(self.table_el, 'HatNo')


def get_key(table_el):
    parts = [
        get_sub_tag_text(table_el, 'HatNo', silent=True),
        get_sub_tag_text(table_el, 'FirmaNo', silent=True),
        get_sub_tag_text(table_el, 'FirmaAdi', silent=True),

        # TODO привести к стандартному виду
        get_sub_tag_text(table_el, 'GuzergahSaat', silent=True),
    ]
    return tuple(parts)


class IpekturStationTitle(object):
    """
    Используем только каноническое наименование для станций
    """

    def __init__(self, title):
        # когда title только с английскими буквами, он приходит как str, поэтому переводим в unicode
        self.title = self.get_canonical_title(unicode(title))

    def __unicode__(self):
        return self.title

    def __eq__(self, other):
        return self.title == other.title

    def __ne__(self, other):
        return not self.__eq__(other)

    @classmethod
    def get_canonical_title(cls, title):
        return title.lower().translate(CHAR_MAPPING)


class RawData(object):
    def __init__(self, filemap):
        self.filemap = filemap

    def get_filepaths(self):
        return self.filemap.itervalues()


class IpekturPackageFileProvider(PackageFileProvider):
    def get_raw_data(self):
        return RawData(self.unpack_package_archive())


class IpekturFTPFileProvider(PackageFileProvider):
    def get_raw_data(self):
        return RawData(self.get_schedule_files())

    @cache_method_result
    def get_schedule_files(self):
        return self.get_filemap_from_archive(self.download_schedule_archive())

    def get_filemap_from_archive(self, archive_path):
        if archive_path.endswith('.zip'):
            unpack_methods = ('zip', 'rar')

        elif archive_path.endswith('.rar'):
            unpack_methods = ('rar', 'zip')

        else:
            raise RaspImportError(u'Not supported archive format {}'.format(archive_path))

        archive_filemap = unpack(archive_path, unpack_methods)

        filemap = self.filter_filemap(archive_filemap)

        if not filemap:
            raise RaspImportError(u'XML-file was not found in archive %s' % pformat(archive_filemap))

        return filemap

    re_schedule_file_name = re.compile(r'^\w[\w\d]*[.]xml$', re.U | re.I)

    def filter_filemap(self, file_to_path_map):
        filemap = dict()

        for fname in file_to_path_map:
            if self.re_schedule_file_name.match(fname):
                filemap[fname] = file_to_path_map[fname]

        return filemap

    def download_schedule_archive(self):
        ftp_files_by_dates = OrderedDict()

        with closing(FTP(host=FTP_HOST, user=FTP_USER, passwd=FTP_PASSWORD,
                         timeout=settings.SCHEDULE_IMPORT_TIMEOUT)) as ftp:

            files = ftp.nlst()

            for filename in filter(lambda fname: fname.endswith(('.zip', '.rar')), files):
                try:
                    date_args = reversed(map(int, filename.split('.')[:3]))
                    export_date = date(*date_args)
                    ftp_files_by_dates[export_date] = filename

                except (ValueError, TypeError):
                    log.warning(u"Archive doesn't have export date. Skip it %s", filename)

        if not ftp_files_by_dates:
            raise RaspImportError(
                u'No files suitable for import was found. Need dd.mm.YYYY.zip file. %s' % files)

        ftp_files_by_dates = OrderedDict(
            (k, ftp_files_by_dates[k]) for k in sorted(ftp_files_by_dates, reversed=True)
        )

        last_archive_name = ftp_files_by_dates.values()[0]

        filepath = get_schedule_temporary_date_filepath(last_archive_name, self.package)

        if os.path.exists(filepath):
            log.info(u'File %s was already downloaded, so we will use it.', filepath)

            return filepath

        self.download(last_archive_name, filepath)

        log.info(u'File %s was downloaded successfully.', filepath)

        return filepath

    def download(self, filename, filepath):
        url = 'ftp://{user}:{password}@{host}/{filename}'.format(
            host=FTP_HOST,
            user=FTP_USER,
            password=FTP_PASSWORD,
            filename=filename,
        )

        try:
            self.download_file(url, filepath)

            return filepath

        except Exception:
            raise RaspImportError(u'Error download %s' % filename)


def get_tr_datetime(datetime_str):
    try:
        return dateutil.parser.parse(datetime_str).astimezone(pytz.timezone(TIMEZONE))

    except ValueError:
        return None


class WarningCode(object):
    def __init__(self):
        self._parts = set()

    def add(self, warning_code_part):
        self._parts.add(warning_code_part)

    @property
    def code(self):
        return '--'.join(self._parts) or 'good'

    def __unicode__(self):
        return self.code


def write_raw_tables_summary(raw_table_els, filepath):
    output_data = u''

    first_raw_table_el = raw_table_els[0]

    # (1) write header
    output_data += u'\nHatNo: %s\nFirmaNo: %s\nFirmaAdi: %s\nGuzergahSaat: %s\n' % (
        get_sub_tag_text(first_raw_table_el, 'HatNo', silent=True),
        get_sub_tag_text(first_raw_table_el, 'FirmaNo', silent=True),
        get_sub_tag_text(first_raw_table_el, 'FirmaAdi', silent=True),
        get_sub_tag_text(first_raw_table_el, 'GuzergahSaat', silent=True)
    )

    output_data += u'---------------------------\n'

    # (2) make stops
    stops = IpekturStops(get_sub_tag_text(first_raw_table_el, 'GuzergahSaat')).get_stops()

    # (3) fill stops times and other_stops
    raw_tables = [RawTable(table_el) for table_el in raw_table_els]

    for table in raw_tables:
        # departure
        try:
            stop = next((x for x in stops if x.station == table.departure_station))
            if table.departure_dt not in stop.departures:
                stop.departures.append(table.departure_dt)

        except StopIteration:
            new_stop = RawStop(table.departure_station, is_originaly_in_route=False)
            new_stop.departures.append(table.departure_dt)

            stops.append(new_stop)

        # arrival
        try:
            stop = next((x for x in stops if x.station == table.arrival_station))
            if table.arrival_dt not in stop.arrivals:
                stop.arrivals.append(table.arrival_dt)

        except StopIteration:
            new_stop = RawStop(table.arrival_station, is_originaly_in_route=False)
            new_stop.arrivals.append(table.arrival_dt)

            stops.append(new_stop)

    other_stops = list(stop for stop in stops if not stop.is_originaly_in_route)
    stops = list(stop for stop in stops if stop.is_originaly_in_route)

    # (4) write data
    for stop in stops:
        output_data += u'%s %s %s\n' % (
            stop.station,
            [d.strftime("%Y-%m-%d %H:%M:%S") for d in stop.departures],
            [a.strftime("%Y-%m-%d %H:%M:%S") for a in stop.arrivals]
        )

    output_data += u'---------------------------\n'

    for stop in other_stops:
        output_data += u'%s %s %s\n' % (
            stop.station,
            [d.strftime("%Y-%m-%d %H:%M:%S") for d in stop.departures],
            [a.strftime("%Y-%m-%d %H:%M:%S") for a in stop.arrivals]
        )

    output_data += u'---------------------------\n'

    # (5) write raw data for reference
    for raw_table_el in raw_table_els:
        output_data += etree.tostring(raw_table_el, pretty_print=True,
                                      xml_declaration=False, encoding=unicode) + u'\n'

    # (6) save
    with codecs.open(filepath, 'w', encoding='utf-8') as f:
        f.write(output_data)
