# coding: utf-8

import os.path
from datetime import timedelta
import re
from urllib2 import Request

from django.conf import settings
from lxml import etree
from common.cysix.builder import ChannelBlock, GroupBlock, ThreadBlock, ScheduleBlock, StoppointBlock

from cysix.two_stage_import import CysixMaskParser
from cysix.two_stage_import.factory import CysixTSIFactory
from travel.rasp.library.python.common23.date import environment
from common.utils.caching import cache_method_result_with_exception, cache_method_result
from travel.rasp.admin.lib.logs import get_current_file_logger
from travel.rasp.admin.lib.xmlutils import get_sub_tag_text
from travel.rasp.admin.scripts.schedule.utils import RaspImportError
from travel.rasp.admin.scripts.schedule.utils.mask_builders import MaskBuilder
from travel.rasp.admin.scripts.schedule.utils.file_providers import PackageFileProvider
from travel.rasp.admin.scripts.schedule.utils.unicode_utf8_pprint import uformat
from travel.rasp.admin.scripts.utils.to_python_parsers import parse_date


log = get_current_file_logger()
MAX_FORWARD_DAYS = 20
WSDL_URL = 'http://92.255.201.138:9080/ScheduleAPI/schedule.asmx'
LOGIN = 'yaschedule'
PASSWORD = 'yaschedule2013'


class KazanCysixFactory(CysixTSIFactory):
    def get_download_file_provider(self):
        log.info(u"Качаем данные %s", self.package.url)

        return KazanWSDLCysixFileProvider(self.package)

    def get_package_file_provider(self):
        log.info(u"Берем данные из приложенного файла")

        return KazanPackageFileProvider(self.package)

    @cache_method_result
    def get_settings(self):
        settings = super(KazanCysixFactory, self).get_settings()

        settings.calculate_not_filled_last_stations_times = True
        settings.calculate_bad_distances_from_geo = True

        return settings


class KazanPackageFileProvider(PackageFileProvider):
    def __init__(self, package):
        super(KazanPackageFileProvider, self).__init__(package)

        file_map = self.get_unpack_map_with_trimmed_dirs_from_package_archive()

        filenames = [('schedules.xml', 'SelectSchedule.xml'),
                     ('main_stations.xml', 'SelectStations.xml'),
                     ('routes.xml', 'SelectRoutes.xml'),
                     ('schedule_rtstations.xml', 'SelectScheduleTripHalts.xml'),
                     ('stations.xml', 'SelectHalts.xml'),
                     ('carriers.xml', 'SelectAtp.xml')]

        for name_varians in filenames:

            for name in name_varians:
                if name in file_map:
                    break
            else:
                raise RaspImportError(u"В приложенном архиве должен быть один из файлов {}"
                                      .format(name_varians))

        self.schedules_file = file_map.get('schedules.xml', None) or \
            file_map['SelectSchedule.xml']

        self.main_stations_file = file_map.get('main_stations.xml', None) or \
            file_map['SelectStations.xml']

        self.routes_file = file_map.get('routes.xml', None) or \
            file_map['SelectRoutes.xml']

        self.schedule_rtstations_file = file_map.get('schedule_rtstations.xml', None) or \
            file_map['SelectScheduleTripHalts.xml']

        self.stations_file = file_map.get('stations.xml', None) or \
            file_map['SelectHalts.xml']

        self.carriers_file = file_map.get('carriers.xml', None) or \
            file_map['SelectAtp.xml']

    def get_carriers_file(self):
        return self.carriers_file

    def get_stations_file(self):
        return self.stations_file

    def get_routes_file(self):
        return self.routes_file

    def get_main_stations_file(self):
        return self.main_stations_file

    def get_schedules_file(self):
        return self.schedules_file

    def get_schedule_rtstations_file(self):
        return self.schedule_rtstations_file

    @cache_method_result_with_exception
    def get_cysix_file(self):
        filepath = self.get_package_filepath('cysix.xml')

        if os.path.exists(filepath) and not settings.DEBUG:
            log.info(u"Данные уже были сконвертированы в общий xml %s", filepath)
            return filepath

        log.info(u"Конвертируем в общий xml %s", filepath)

        self.dowload_and_convert_data(filepath)

        log.info(u"Данные сконвертированы в общий xml %s", filepath)

        return filepath

    def dowload_and_convert_data(self, filepath):
        c = Converter(self)

        return c.convert(filepath)


class KazanWSDLCysixFileProvider(PackageFileProvider):
    def __init__(self, package):
        self.package = package
        super(KazanWSDLCysixFileProvider, self).__init__(package)

    @cache_method_result_with_exception
    def get_cysix_file(self):
        filepath = self.get_package_filepath('cysix.xml')

        if os.path.exists(filepath) and not settings.DEBUG:
            log.info(u"Данные уже были сконвертированы в общий xml %s", filepath)
            return filepath

        log.info(u"Конвертируем в общий xml %s", filepath)

        self.dowload_and_convert_data(filepath)

        log.info(u"Данные сконвертированы в общий xml %s", filepath)

        return filepath

    def dowload_and_convert_data(self, filepath):
        c = Converter(self)

        return c.convert(filepath)

    def get_carriers_file(self):
        query_tree = etree.Element('{AV.ScheduleAPI}SelectAtp', nsmap={None: 'AV.ScheduleAPI'})
        self.add_credentials(query_tree)

        log.info(u"Получаем список перевозчиков SelectAtp")

        return self.download_file(query_tree, self.get_package_filepath('carriers.xml'))

    def get_stations_file(self):
        query_tree = etree.Element('{AV.ScheduleAPI}SelectHalts', nsmap={None: 'AV.ScheduleAPI'})
        self.add_credentials(query_tree)

        log.info(u"Получаем список остановок SelectHalts")

        return self.download_file(query_tree, self.get_package_filepath('stations.xml'))

    def get_routes_file(self):
        query_tree = etree.Element('{AV.ScheduleAPI}SelectRoutes', nsmap={None: 'AV.ScheduleAPI'})
        self.add_credentials(query_tree)

        log.info(u"Получаем список маршрутов SelectRoutes")

        return self.download_file(query_tree, self.get_package_filepath('routes.xml'))

    def get_main_stations_file(self):
        query_tree = etree.Element('{AV.ScheduleAPI}SelectStations', nsmap={None: 'AV.ScheduleAPI'})
        self.add_credentials(query_tree)

        log.info(u"Получаем список автовокзалов SelectStations")

        return self.download_file(query_tree, self.get_package_filepath('main_stations.xml'))

    def get_schedules_file(self):
        query_tree = etree.Element('{AV.ScheduleAPI}SelectSchedule', nsmap={None: 'AV.ScheduleAPI'})
        self.add_credentials(query_tree)

        main_station = etree.SubElement(query_tree, 'StationId')
        main_station.text = u"-1"

        log.info(u"Получаем список получаем расписание SelectSchedule")

        return self.download_file(query_tree, self.get_package_filepath('schedules.xml'))

    def get_schedule_rtstations_file(self):
        query_tree = etree.Element('{AV.ScheduleAPI}SelectScheduleTripHalts', nsmap={None: 'AV.ScheduleAPI'})
        self.add_credentials(query_tree)

        main_station = etree.SubElement(query_tree, 'StationId')
        main_station.text = u"-1"

        log.info(u"Получаем список получаем остановки рейсов SelectScheduleTripHalts")

        return self.download_file(query_tree, self.get_package_filepath('schedule_rtstations.xml'))

    def get_request(self, query_tree):
        query = etree.tostring(query_tree, pretty_print=True, xml_declaration=False, encoding=unicode)
        data = BASE_TEMPLATE.format(query=query).encode('utf-8')
        request = Request(WSDL_URL, data=data, headers={'Content-Type': 'text/xml; charset=utf-8'})

        return request

    def retrieve_data(self, download_params):
        data = ''.join(super(KazanWSDLCysixFileProvider, self).retrieve_data(download_params))
        tree = etree.fromstring(data)
        yield etree.tostring(tree, pretty_print=True, encoding='utf-8', xml_declaration=True)

    def add_credentials(self, query_tree):
        login = etree.SubElement(query_tree, 'Login')
        login.text = LOGIN

        password = etree.SubElement(query_tree, 'Password')
        password.text = PASSWORD


class Converter(object):
    def __init__(self, provider):
        self.provider = provider

    def convert(self, filepath):
        self.stations = []
        self.carriers = []
        self.vehicles = []
        self.fares = []

        self.fill_raw_carriers()
        self.fill_raw_stations()
        self.fill_raw_routes()
        self.fill_raw_main_stations()
        self.fill_schedules(self.raw_routes)
        self.fill_schedule_rtstations(self.raw_stations)

        self.today = environment.today()

        self.mask_builder = MaskBuilder(self.today - timedelta(1), self.today + timedelta(MAX_FORWARD_DAYS))

        channel_block = ChannelBlock('bus', station_code_system='vendor', carrier_code_system='vendor')

        for main_station in self.main_stations.values():
            self.build_group(channel_block, main_station)

        channel_block.to_xml_file(filepath)

    def build_group(self, channel_block, main_station):
        self.stations = []
        self.carriers = []
        self.vehicles = []
        self.fares = []

        group_block = GroupBlock(channel_block, main_station['main_station_title'], main_station['main_station_code'])

        schedules = self.schedules_by_main_station[main_station['main_station_code']]

        self.parse_threads(group_block, schedules)

        channel_block.add_group_block(group_block)

    def parse_threads(self, group_block, schedules):
        for schedule in schedules:
            route_title = self.raw_routes.get(schedule['raw_route_id'])
            try:
                self.parse_schedule(group_block, schedule)
            except RaspImportError as e:
                log.error(u"Ошибка при разборе schedule %s %s %s:", schedule['schedule_id'], route_title, unicode(e))
            except Exception:
                log.exception(u"Неожиданная ошибка при разборе schedule %s %s", schedule['schedule_id'], route_title)

    def parse_schedule(self, group_block, schedule):
        title = self.raw_routes.get(schedule['raw_route_id'])
        route_stations = self.schedule_rtstations[schedule['schedule_station_id']]

        log.info(u"Разбираем %s %s", title, schedule['schedule_station_id'])

        log.debug(u"%s\n%s\n%s", title, uformat(schedule), uformat(route_stations))

        thread_block = ThreadBlock(group_block, title, schedule['schedule_station_id'])

        self.make_schedule(thread_block, schedule)

        carrier_code = schedule['carrier_id']
        carrier_title = self.raw_carriers.get(carrier_code, u"")

        carrier_block = group_block.add_carrier(carrier_title, carrier_code)
        thread_block.carrier = carrier_block

        raw_rtstations = self.schedule_rtstations.get(schedule['schedule_station_id'], [])

        start_station_code = schedule['start_station_id']
        start_station_title = self.raw_stations.get(start_station_code, u"")
        start_station_block = group_block.add_station(start_station_title, start_station_code)

        start_stoppoint_block = StoppointBlock(thread_block, start_station_block)
        start_stoppoint_block.departure_time = schedule['start_time']

        thread_block.add_stoppoint_block(start_stoppoint_block)

        for raw_rtstation in raw_rtstations:
            distance = raw_rtstation['distance'].replace(',', '.')
            arrival_shift = None
            if raw_rtstation['duration'] and raw_rtstation['duration'] != '00:00':
                hours, minutes = map(int, raw_rtstation['duration'].split(':'))
                arrival_shift = hours * 3600 + minutes * 60

            station_code = raw_rtstation['stop_station_id']
            station_title = self.raw_stations.get(station_code, u"")

            station_block = group_block.add_station(station_title, station_code)

            stoppoint_block = StoppointBlock(thread_block, station_block, distance=distance,
                                             arrival_shift=arrival_shift)

            thread_block.add_stoppoint_block(stoppoint_block)

        group_block.add_thread_block(thread_block)

    def get_mask(self, schedule_el):
        return CysixMaskParser.parse_mask(schedule_el, self.mask_builder)

    def make_schedule(self, thread_block, schedule):
        schedule_block_kw = self.parse_days(schedule['days'])

        thread_block.add_schedule_block(ScheduleBlock(thread_block, times=schedule['start_time'],
                                                      **schedule_block_kw))

    short_days = {
        u'пн': u'1', u'вт': u'2', u'ср': u'3', u'чт': u'4', u'пт': u'5', u'сб': u'6', u'вс': u'7'
    }
    short_days_re_text = u"|".join(short_days.keys())
    short_days_re = re.compile(ur"(?:\s*,?\s*(" + short_days_re_text + u")\s*)+", re.U)
    long_days = {
        u'понедельник': u'1', u'вторник': u'2', u'среда': u'3', u'четверг': u'4', u'пятница': u'5',
        u'суббота': u'6', u'субботы': u'6',
        u'воскресенье': u'7', u'воскресение': u'7', u'воскресения': u'7'
    }
    long_days_re_text = u"|".join(long_days.keys())
    long_days_re = re.compile(ur"(?:\s*,?\s*(" + long_days_re_text + u")\s*)+", re.U)
    through_the_day_re = re.compile(ur'^\s*через\s+день,\s*[cс]\s*(?P<start_date>\d+-\d+-\d+)\s*$',
                                    re.I + re.U)

    def parse_days(self, kazan_days):
        days = kazan_days.lower()

        if days in [u'ежедневно', u'рабочие дни', u'выходные дни']:
            return {
                'days': days
            }

        elif days.startswith(u"кроме"):
            try:
                schedule_block_kw = self.parse_days(days.replace(u"кроме", u"").strip())
            except RaspImportError as e:
                raise RaspImportError(u"Ошибка разбора строки дней хождения '{}'\n{}".format(days, e))

            return {
                'days': u"ежедневно",
                'exclude_days': schedule_block_kw['days']
            }

        elif self.long_days_re.match(days):
            out_days = set()
            for name in self.long_days.keys():
                if name in days:
                    out_days.add(self.long_days[name])

            days = u"".join(out_days)

            return {
                'days': days
            }

        elif self.short_days_re.match(days):
            out_days = set()
            for name in self.short_days.keys():
                if name in days:
                    out_days.add(self.short_days[name])

            days = u"".join(out_days)

            return {
                'days': days
            }

        elif self.through_the_day_re.match(days):
            match = self.through_the_day_re.match(days)
            start_date = parse_date(match.groupdict()['start_date'])

            return {
                'days': u'через день',
                'period_start_date': start_date.strftime('%Y-%m-%d')
            }

        else:
            raise RaspImportError(u"Ошибка разбора строки дней хождения '{}'".format(days))

    def add_carrier(self, thread_el, timetable_el):
        carrier_title = get_sub_tag_text(timetable_el, 'atp', silent=True).strip().replace(ur'\"', u'"')
        if carrier_title:
            code = self.create_carrier(carrier_title)
            thread_el.set('carrier_code', code)

    @cache_method_result
    def create_carrier(self, carrier_title):
        code = unicode(len(self.carriers))
        self.carriers.append(etree.Element('carrier', {
            'title': carrier_title,
            'code': code
        }))

        return code

    def fill_raw_carriers(self):
        self.raw_carriers = {}

        filepath = self.provider.get_carriers_file()

        for table in etree.parse(filepath).findall('.//Table1'):
            carrier_code = get_sub_tag_text(table, 'ATPID').strip()
            carrier_title = get_sub_tag_text(table, 'ATPNAME').strip()

            self.raw_carriers[carrier_code] = carrier_title

    def fill_raw_stations(self):
        self.raw_stations = {}

        filepath = self.provider.get_stations_file()

        for table in etree.parse(filepath).findall('.//Table1'):
            station_code = get_sub_tag_text(table, 'HALTID').strip()
            station_title = get_sub_tag_text(table, 'HALTNAME').strip()

            self.raw_stations[station_code] = station_title

    def fill_raw_main_stations(self):
        self.main_stations = {}

        filepath = self.provider.get_main_stations_file()

        for table in etree.parse(filepath).findall('.//Table1'):
            main_station_code = get_sub_tag_text(table, 'STATIONID').strip()
            main_station_title = get_sub_tag_text(table, 'STATIONNAME').strip()
            station_code = get_sub_tag_text(table, 'STATIONHALTID').strip()
            station_title = get_sub_tag_text(table, 'STATIONHALTNAME').strip()

            self.main_stations[main_station_code] = {
                'main_station_code': main_station_code,
                'main_station_title': main_station_title,
                'station_code': station_code,
                'station_title': station_title
            }

    def fill_raw_routes(self):
        self.raw_routes = {}

        filepath = self.provider.get_routes_file()

        for table in etree.parse(filepath).findall('.//Table1'):
            route_code = get_sub_tag_text(table, 'ROUTEID').strip()
            route_title = get_sub_tag_text(table, 'ROUTENAME').strip()

            self.raw_routes[route_code] = route_title

    def fill_schedules(self, raw_routes):
        self.schedules_by_main_station = {}

        filepath = self.provider.get_schedules_file()

        for table in etree.parse(filepath).findall('.//Table1'):
            schedule = {
                'main_station_id': get_sub_tag_text(table, 'STATIONID').strip(),
                'start_station_id': get_sub_tag_text(table, 'HALTSTARTID').strip(),
                'schedule_id': get_sub_tag_text(table, 'SCHEDULEID').strip(),
                'schedule_station_id': get_sub_tag_text(table, 'SCHEDULESTATIONID').strip(),
                'days': get_sub_tag_text(table, 'TRIPPERIODICITY').strip(),
                'raw_route_id': get_sub_tag_text(table, 'ROUTEID').strip(),
                'start_time': get_sub_tag_text(table, 'TRIPTIME').strip(),
                'place_count': get_sub_tag_text(table, 'PLACECOUNT').strip(),
                'carrier_id': get_sub_tag_text(table, 'ATPID').strip(),
            }

            schedule['title'] = raw_routes.get(schedule['raw_route_id'])

            self.schedules_by_main_station.setdefault(schedule['main_station_id'], []).append(schedule)

    def fill_schedule_rtstations(self, raw_stations):
        self.schedule_rtstations = {}

        filepath = self.provider.get_schedule_rtstations_file()

        for table in etree.parse(filepath).findall('.//Table1'):
            leg = {
                'schedule_station_id': get_sub_tag_text(table, 'SCHEDULESTATIONID').strip(),
                'start_station_id': get_sub_tag_text(table, 'STARTHALTID').strip(),
                'stop_station_id': get_sub_tag_text(table, 'STOPHALTID').strip(),
                'path_index': get_sub_tag_text(table, 'ROUTEHALTINDEX').strip(),
                'distance': get_sub_tag_text(table, 'ROUTEHALTDISTANCE', silent=True).strip(),
                'duration': get_sub_tag_text(table, 'ROUTEHALTARRIVAL', silent=True).strip(),
            }

            if leg['distance']:
                leg['distance'] = leg['distance'].replace(u'.', u'').replace(u',', u'.').strip()
                if leg['distance'] == u'.00':
                    leg['distance'] = u'0'

            leg['start_title'] = raw_stations.get(leg['start_station_id'])
            leg['stop_title'] = raw_stations.get(leg['stop_station_id'])

            self.schedule_rtstations.setdefault(leg['schedule_station_id'], []).append(leg)

        for path in self.schedule_rtstations.values():
            path.sort(key=lambda leg: int(leg['path_index']))


BASE_TEMPLATE = u"""
<?xml version="1.0" encoding="utf-8"?>
<soap:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
 xmlns:xsd="http://www.w3.org/2001/XMLSchema"
 xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">
  <soap:Body>
{query}
  </soap:Body>
</soap:Envelope>
""".strip()

