# coding: utf-8

import json
import os.path
from datetime import date, time

from django.conf import settings
from lxml import etree

from common.cysix.builder import ChannelBlock, GroupBlock, ThreadBlock, StoppointBlock, ScheduleBlock
from common.utils.caching import cache_method_result_with_exception, cache_method_result
from common.utils.unicode_csv import UnicodeDictReader
from cysix.two_stage_import.factory import CysixTSIFactory
from travel.rasp.admin.lib.logs import get_current_file_logger
from travel.rasp.admin.scripts.schedule.utils.ftp import get_ftp_file, FtpDownloadError
from travel.rasp.admin.scripts.schedule.utils import RaspImportError
from travel.rasp.admin.scripts.schedule.utils.file_providers import PackageFileProvider
from travel.rasp.admin.lib.exceptions import SimpleUnicodeException


FTP_URL = 'ftp://ftp.kmvavto.nichost.ru/rasp.zip'
FTP_USER = 'kmvavto_yandex'
FTP_PASSWORD = 'fnteisyi'

PRIVATE_URL = 'ftp://kmvavto_yandex:fnteisyi@ftp.kmvavto.nichost.ru/rasp.zip'

CURRENCY = u'RUR'

log = get_current_file_logger()


class KmvaCysixFactory(CysixTSIFactory):
    def get_download_file_provider(self):
        log.info(u"Качаем данные по %s", FTP_URL)

        return KmvaCysixFileProvider(self.package)

    def get_package_file_provider(self):
        return KmvaCysixPackageFileProvider(self.package, self.package.supplier)


class KmvaFTPFileProvider(PackageFileProvider):
    @cache_method_result
    def get_schedule_files(self):
        files = []

        filepath = self.get_package_filepath('rasp.zip')

        self.download_file(PRIVATE_URL, filepath)

        fmap = self.unzip_files(filepath)

        for fname in fmap:
            if fname.endswith('.txt'):
                files.append(fmap[fname])

        if not files:
            raise RaspImportError(u"Не нашли нужных файлов в архиве")

        return files

    def retrieve_data(self, ftp_url):
        wget, stdout, stderr = get_ftp_file(ftp_url, ftp_user=FTP_USER, ftp_password=FTP_PASSWORD)

        if wget.returncode != 0:
            raise FtpDownloadError(u'Не удалось скачать файлы')

        yield stdout


class KmvaCysixFileProvider(KmvaFTPFileProvider):
    @cache_method_result_with_exception
    def get_cysix_file(self):
        filepath = self.get_package_filepath('cysix.xml')

        if os.path.exists(filepath) and not settings.DEBUG:
            log.info(u"Данные уже были сконвертированы в общий xml %s", filepath)
            return filepath

        log.info(u"Конвертируем в общий xml %s", filepath)

        self.unzip_and_convert_data(filepath)

        log.info(u"Данные сконвертированы в общий xml %s", filepath)

        return filepath

    def unzip_and_convert_data(self, filepath):
        c = Converter(self)
        filepath = c.convert(filepath)
        return filepath


class KmvaCysixPackageFileProvider(PackageFileProvider):
    @cache_method_result_with_exception
    def get_cysix_file(self):
        filepath = self.get_package_filepath('cysix.xml')

        if os.path.exists(filepath) and not settings.DEBUG:
            log.info(u"Данные уже были сконвертированы в общий xml %s", filepath)
            return filepath

        log.info(u"Конвертируем в общий xml %s", filepath)

        self.unzip_and_convert_data(filepath)

        log.info(u"Данные сконвертированы в общий xml %s", filepath)

        return filepath

    def unzip_and_convert_data(self, filepath):
        c = Converter(self)
        filepath = c.convert(filepath)
        return filepath

    def get_schedule_files(self):
        return self.get_files_from_archive_with_ext('txt')


class ConvertError(SimpleUnicodeException):
    pass


class Converter(object):
    def __init__(self, provider):
        self.provider = provider
        self.schedule_files = []

    def convert(self, filepath):
        self.schedule_files = self.provider.get_schedule_files()

        channel_block = ChannelBlock('bus', station_code_system='vendor', carrier_code_system='local',
                                     vehicle_code_system='local', timezone='start_station')

        group_title = u'Кавминводыавто'
        group_code = 'kmvavto'
        group_block = GroupBlock(channel_block, group_title, group_code)

        stations = self.build_stations(group_block)
        fares = self.build_fares(group_block)

        self.build_threads(group_block, stations, fares)

        channel_block.add_group_block(group_block)

        channel_el = channel_block.get_element()

        channel_el = add_legacy_stations(channel_el)

        with open(filepath, 'w') as f:
            f.write(etree.tostring(channel_el, xml_declaration=True, encoding='utf-8', pretty_print=True))

    def build_threads(self, group_block, stations, fares):
        raw_routes = self.get_raw_routes()

        for route_key, route in raw_routes.iteritems():
            try:
                thread_block = self.build_thread(group_block, route_key, route, stations, fares)
                group_block.add_thread_block(thread_block)
            except ConvertError, e:
                log.error(e.msg)

    def build_thread(self, group_block, route_key, route, stations, fares):
        title = route['title']
        thread_block = ThreadBlock(group_block, title=title)

        period = route['period']
        raw_days = route['days']
        days, period_start_date = parse_mask(period, raw_days, title)

        raw_data = dict()
        raw_data['route'] = route['raw']
        raw_data['rtstations'] = []

        raw_rtstations = self.get_raw_rtstations()
        rtstations = raw_rtstations.get(route_key, [])
        rtstations = sorted(rtstations, key=lambda k: int(k['order']))
        for i, rts in enumerate(rtstations):
            station = stations.get(rts['station_code'], None)
            if not station:
                raise ConvertError(u"В нитке '%s' не удалось найти станцию с кодом '%s'" % (title, rts['station_code']))
            if i + 1 != int(rts['order']):
                log.warning(u"В нитке '%s' станция '%s' идет под номером '%s' вместо '%s'.",
                            title, station.title, i + 1, rts['order'])

            rts['raw']['station'] = station.raw
            raw_data['rtstations'].append(rts['raw'])

            arrival_time = parse_time(rts['arrival_time'])
            departure_time = parse_time(rts['departure_time'])
            is_technical = None
            if rts['is_technical']:
                is_technical = int(rts['is_technical'])

            stoppoint_block = StoppointBlock(thread_block, station,
                                             arrival_time=arrival_time, departure_time=departure_time,
                                             is_technical=is_technical)

            thread_block.add_stoppoint_block(stoppoint_block)

            if i == 0:
                thread_block.add_schedule_block(ScheduleBlock(thread_block, days, times=departure_time,
                                                              period_start_date=period_start_date))

        fare_block = fares.get(route_key, None)
        if fare_block:
            thread_block.set_fare_block(fare_block)
            raw_data['fares'] = fare_block.raw

        raw_data = json.dumps(raw_data, indent=4, ensure_ascii=False, encoding='utf8')

        thread_block.set_raw_data(raw_data)

        return thread_block

    @cache_method_result
    def build_stations(self, group_block):
        stations = {}
        for s in self.get_raw_stations().itervalues():
            code = s['code']
            title = s['title']
            station_block = group_block.add_station(title, code)
            station_block.raw = s['raw']
            stations[code] = station_block
        return stations

    def build_fares(self, group_block):
        fares = {}
        stations = self.build_stations(group_block)
        for fare in self.get_raw_fares():

            code = fare['code']
            price_value = fare['price']
            currency = CURRENCY
            stop_from = stations.get(fare['stop_from'], None)
            stop_to = stations.get(fare['stop_to'], None)
            data = None

            if stop_from is None:
                log.error(u"При разборе тарифов не нашли станцию с кодом '%s'" % fare['stop_from'])
            elif stop_to is None:
                log.error(u"При разборе тарифов не нашли станцию с кодом '%s'" % fare['stop_to'])
            else:
                fare_block = group_block.add_fare(code)
                fare_block.add_price_block(price_value, currency, stop_from, stop_to, data=data)
                fare_block.raw = fare['raw']
                fares[code] = fare_block
        return fares

    @cache_method_result_with_exception
    def get_raw_routes(self):
        filename = 'routes.txt'
        filepath = self.find_file(filename)
        if not filepath:
            raise ConvertError(u'Не нашли файл рейсов %s' % filename)

        log.info(u"Разбираем %s" % filepath)
        raw_routes = {}
        reader = UnicodeDictReader(open(filepath), delimiter='\t', encoding='cp1251')
        for row in reader:
            number = row['KOD']
            if number:
                raw_routes[number] = {
                    'number': number,
                    'title': row['NAIM'],
                    'start_station': row['KODNP'],
                    'end_station': row['KODKP'],
                    'period': row['PERIOD'],
                    'days': row['DNI'],
                    'raw': row,
                }
        return raw_routes

    @cache_method_result_with_exception
    def get_raw_rtstations(self):
        filename = 'rtstations.txt'
        filepath = self.find_file(filename)
        if not filepath:
            raise ConvertError(u'Не нашли файл станций рейсов %s' % filename)

        log.info(u"Разбираем %s" % filepath)
        raw_rtstations = {}
        reader = UnicodeDictReader(open(filepath), delimiter='\t', encoding='cp1251')

        for row in reader:
            number = row['KODR']
            if number:
                raw_rtstations.setdefault(number, []).append({
                    'order': row['NOMPP'],
                    'station_code': row['KODOP'],
                    'arrival_time': row['VRPRIB'],
                    'departure_time': row['VROTPR'],
                    'is_technical': bool(int(row.get('TEXOST', u'0'))),
                    'raw': row,
                })

        return raw_rtstations

    @cache_method_result_with_exception
    def get_raw_stations(self):
        filename = 'stations.txt'
        filepath = self.find_file(filename)
        if not filepath:
            raise ConvertError(u'Не нашли файл станций %s' % filename)

        log.info(u"Разбираем %s" % filepath)
        raw_stations = {}
        reader = UnicodeDictReader(open(filepath), delimiter='\t', encoding='cp1251')
        for row in reader:
            code = row['KOD']
            title = row['NAIM']
            if code:
                raw_stations[code] = {
                    'code': code,
                    'title': title,
                    'raw': row,
                }
        return raw_stations

    @cache_method_result_with_exception
    def get_raw_fares(self):
        filename = 'tariffs.txt'
        filepath = self.find_file(filename)
        if not filepath:
            raise ConvertError(u'Не нашли файл тарифов %s' % filename)

        log.info(u"Разбираем %s" % filepath)
        raw_fares = []
        reader = UnicodeDictReader(open(filepath), delimiter='\t', encoding='cp1251')
        for row in reader:
            code = row['KODR']
            stop_from = row['KODBG']
            stop_to = row['KODPN']
            price = row['CENA']
            if price:
                price = price.replace(u',', u'.')
            if code:
                fare = {
                    'code': code,
                    'stop_from': stop_from,
                    'stop_to': stop_to,
                    'price': price,
                    'raw': row,
                }
                raw_fares.append(fare)
        return raw_fares

    def find_file(self, filename):
        try:
            return [f for f in self.schedule_files if f.endswith(u'/' + filename)][0]
        except IndexError:
            return None


ALL_DAYS = frozenset(u'1234567')


def parse_mask(period, raw_days, route_title):
    period_start_date = None
    if period == u'Все' and not raw_days:
        days = u'1234567'
    elif period == u'Четные' and not raw_days:
        days = u'четные'
    elif period == u'Нечетные' and not raw_days:
        days = u'нечетные'
    elif period == u'Через день':
        days = u'через день'
        period_start_date = parse_date(raw_days)
        if not period_start_date:
            raise ConvertError(u"В нитке '%s' не удалось разобрать дни хождения. period '%s', days '%s'."
                               u" Маска 'через день' должна сопровождаться датой старта"
                               % (route_title, period, raw_days))
    elif period == u'Дни':
        raw_days_set = set(raw_days)
        if raw_days_set.issubset(ALL_DAYS):
            days = raw_days
        else:
            raise ConvertError(u"В нитке '%s' не удалось разобрать дни хождения. period '%s', days '%s'."
                               % (route_title, period, raw_days))
    else:
        raise ConvertError(u"В нитке '%s' не удалось разобрать дни хождения. period '%s', days '%s'"
                           % (route_title, period, raw_days))
    return days, period_start_date


def parse_time(start_time_txt):
    try:
        return time(*map(int, start_time_txt.split('.'))).strftime('%H:%M')
    except ValueError:
        return None


def parse_date(date_txt):
    try:
        day, month, year = map(int, date_txt.split('.'))
        return date(year, month, day).strftime('%Y-%m-%d')
    except ValueError:
        return None


def add_legacy_stations(channel_el):
    for stations_el in channel_el.findall('.//stations'):
        for station_el in stations_el.findall('./station'):
            legacy_station_el = etree.Element('legacy_station')
            legacy_station_el.set('code', station_el.attrib.get('code', ''))
            legacy_station_el.set('title', station_el.attrib.get('title', ''))
            legacy_station_el.set('type', 'raw')
            station_el.append(legacy_station_el)
    return channel_el
