# coding: utf-8

import json
import os.path
import time as os_time
from collections import OrderedDict
from datetime import time
from functools import wraps
from lxml import etree

from django.conf import settings
import requests

from common.cysix.builder import ChannelBlock, GroupBlock, ThreadBlock, StoppointBlock, ScheduleBlock
from common.models.geo import Region
from common.utils.caching import cache_method_result
from common.utils.unicode_csv import UnicodeDictReader
from cysix.tsi_converter import CysixTSIConverterFactory, CysixTSIConverterFileProvider
from travel.rasp.admin.lib.logs import get_current_file_logger
from travel.rasp.admin.lib.xls import XlsDictParser
from travel.rasp.admin.scripts.schedule.utils.file_providers import PackageFileProvider
from travel.rasp.admin.scripts.schedule.utils import RaspImportError


log = get_current_file_logger()


SCHEDULE_URL = 'http://avokzal.udm.ru/Files/Yandex/trips.xls'
STATIONS_URL = 'http://avokzal.udm.ru/Files/Yandex/stops.xls'
TERMINALS_URL = 'http://avokzal.udm.ru/Files/Yandex/stations.xls'

SCHEDULE_FILENAME = SCHEDULE_URL.split('/')[-1]
STATIONS_FILENAME = STATIONS_URL.split('/')[-1]
TERMINALS_FILENAME = TERMINALS_URL.split('/')[-1]

SUPPLIER_CODE = 'udmbus'
UDMURTIA_REGION_ID = 11148
REPLACE_ARRIVAL_FILEPATH = os.path.join(settings.DATA_PATH, 'schedule',
                                        SUPPLIER_CODE, 'replace_arrival.csv')
CURRENCY = 'RUR'

MAX_NUMBER_OF_DOWNLOAD_ATTEMPTS = 3

DAYS_OF_WEEK = {
    u'1': u'пн',
    u'2': u'вт',
    u'3': u'ср',
    u'4': u'чт',
    u'5': u'пт',
    u'6': u'сб',
    u'7': u'вс'
}


class UdmCysixFactory(CysixTSIConverterFactory):
    def get_raw_download_file_provider(self):
        return UdmRawDownloadFileProvider(self.package)

    def get_raw_package_file_provider(self):
        return UdmPackageFileProvider(self.package)

    def get_converter_file_provider(self, raw_file_provider):
        return UdmCysixFileProvider(self.package, raw_file_provider)


class UdmCysixFileProvider(CysixTSIConverterFileProvider):
    def convert_data(self, filepath):
        c = Converter(self.raw_file_provider)

        c.convert(filepath)


class ConvertError(RaspImportError):
    pass


class Converter(object):
    def __init__(self, provider):
        self.provider = provider

        self.stations = dict()

        self.time_zone = Region.objects.get(id=UDMURTIA_REGION_ID).time_zone

    def convert(self, filepath):
        raw_data = self.provider.get_raw_data()

        channel_el = self.build_channel_element(raw_data)

        with open(filepath, 'w') as f:
            f.write(etree.tostring(channel_el, xml_declaration=True, encoding='utf-8', pretty_print=True))

    def build_channel_element(self, raw_data):
        channel_block = ChannelBlock('bus', station_code_system='vendor', carrier_code_system='local',
                                     vehicle_code_system='local', timezone=self.time_zone)

        group_block = self.build_group(raw_data, channel_block)

        channel_block.add_group_block(group_block)

        return channel_block.get_element()

    def build_group(self, raw_data, channel_block):
        group_block = GroupBlock(channel_block, title='all', code='all')

        self.build_all_stations(raw_data, group_block)

        self.build_threads(raw_data, group_block)

        return group_block

    def build_all_stations(self, raw_data, group_block):
        for code, title in raw_data.get_raw_stations_dict().iteritems():
            station_block = group_block.add_station(title, code)

            station_block.add_legacy_station(title, code)

            self.stations[code] = station_block

    def build_threads(self, raw_data, group_block):
        first_thread_row = None
        thread_rows = []

        for row in raw_data.schedule_rows_iter():
            if has_not_important_data(row):
                continue

            if is_another_thread(row, first_thread_row):
                self.build_thread(group_block, thread_rows)

                thread_rows = []
                first_thread_row = row

            thread_rows.append(row)

        self.build_thread(group_block, thread_rows)

    def build_thread(self, group_block, thread_rows):
        if not thread_rows:
            return

        raw_thread = RawThread(thread_rows)

        if raw_thread.departure_time is None:
            log.error(u"Пропускаем нитку '%s'. Нет времени старта.", raw_thread.title)

            return

        start_station = self.get_station_or_none(raw_thread.start_station_code)

        if start_station is None:
            log.error(u"Пропускаем нитку '%s'. Не удалось найти стартовую станцию с кодом '%s'.",
                      raw_thread.title, raw_thread.start_station_code)

            return

        carrier_block = group_block.add_local_carrier(raw_thread.company_title)
        vehicle_block = group_block.add_local_vehicle(raw_thread.model_title)

        thread_block = ThreadBlock(
            group_block,
            title=raw_thread.title,
            number=raw_thread.number if raw_thread.number else u'',
            vehicle=vehicle_block,
            carrier=carrier_block,
        )

        thread_block.add_schedule_block(ScheduleBlock(thread_block, raw_thread.schedule))

        self.add_stoppoints_to_thread(raw_thread, thread_block, start_station)

        self.add_fares_to_thread(raw_thread, group_block, thread_block)

        thread_block.set_raw_data(raw_thread.raw_data)

        group_block.add_thread_block(thread_block)

    def add_stoppoints_to_thread(self, raw_thread, thread_block, start_station):
        start_stoppoint_block = StoppointBlock(
            thread_block, start_station,
            departure_time=raw_thread.departure_time.strftime('%H:%M:%S')
        )
        thread_block.add_stoppoint_block(start_stoppoint_block)

        for row in raw_thread.stoppoints:
            station = self.get_station_or_none(row['station_to'])

            if station is None:
                continue

            stoppoint_block = StoppointBlock(
                thread_block,
                station,
                arrival_time=row['arrival_time'].strftime('%H:%M:%S') if row['arrival_time'] else u'',
                distance=u'%.2f' % row['distance'] if row['distance'] else u'',
            )

            thread_block.add_stoppoint_block(stoppoint_block)

    def add_fares_to_thread(self, raw_thread, group_block, thread_block):
        fare_block = group_block.add_local_fare()

        for row in raw_thread.tariffs:
            station_from_block = self.get_station_or_none(row['station_from'])
            station_to_block = self.get_station_or_none(row['station_to'])

            if station_from_block is None or station_to_block is None:
                continue

            order_data = {
                'station_from_udm_code': row['station_from'],
                'station_to_udm_code': row['station_to'],
            }

            order_data = json.dumps(order_data, ensure_ascii=False, encoding='utf8')

            fare_block.add_price_block(str(row['tariff']), CURRENCY,
                                       station_from_block, station_to_block, data=order_data)

            thread_block.set_fare_block(fare_block)

    @cache_method_result
    def get_station_or_none(self, code):
        station = self.stations.get(code, None)

        if station is None:
            log.warning(u"Не удалось найти станцию с кодом '%s'.", code)

        return station


class UdmPackageFileProvider(PackageFileProvider):
    def get_raw_data(self):
        return RawData(self.unpack_package_archive())


class DownloadError(RaspImportError):
    pass


class UdmRawDownloadFileProvider(PackageFileProvider):
    def get_raw_data(self):
        return RawData({
            SCHEDULE_FILENAME: self._download(SCHEDULE_URL, SCHEDULE_FILENAME),
            STATIONS_FILENAME: self._download(STATIONS_URL, STATIONS_FILENAME),
            TERMINALS_FILENAME: self._download(TERMINALS_URL, TERMINALS_FILENAME)
        })

    def _download(self, url, filename):
        filepath = self.get_package_filepath(filename)

        log.info(u'Download %s from %s', filename, url)

        return self.download_file(url, filepath)

    def retrieve_data(self, url):
        for _ in range(MAX_NUMBER_OF_DOWNLOAD_ATTEMPTS):
            try:
                responce = requests.get(url, timeout=settings.SCHEDULE_IMPORT_TIMEOUT)

                return [responce.content]

            except Exception:
                pass

        raise DownloadError(u'Не удалось скачать {} Сделано попыток: {}'.format(
            url, MAX_NUMBER_OF_DOWNLOAD_ATTEMPTS))


class RawData(object):
    def __init__(self, filemap):
        self.filemap = filemap

        self.arrival_replaces = self.init_arrival_replaces()

    def get_schedule_filepath(self):
        return self.filemap.get(SCHEDULE_FILENAME)

    def get_stations_filepath(self):
        return self.filemap.get(STATIONS_FILENAME)

    def get_terminals_filepath(self):
        return self.filemap.get(TERMINALS_FILENAME)

    def init_arrival_replaces(self):
        arrival_replaces = dict()

        replace_arrival = UnicodeDictReader(open(REPLACE_ARRIVAL_FILEPATH), encoding='cp1251',
                                            delimiter=';')

        for replace_arrival_row in replace_arrival:
            replace_arrival_row['station_from'] = parse_station_code(replace_arrival_row['station_from'])
            replace_arrival_row['station_to'] = parse_station_code(replace_arrival_row['station_to'])
            replace_arrival_row['departure_time'] = parse_time(replace_arrival_row['departure_time'])

            arrival_key = self.get_arrival_key_from_row(replace_arrival_row)

            arrival_replaces[arrival_key] = parse_time(replace_arrival_row['arrival_time'])

        return replace_arrival

    def get_raw_stations_dict(self):
        def add_stations(stations, filepath):
            parser = XlsDictParser(
                filepath,
                transform_to_text=True,
                fieldnames=STATION_ROW_PARCE_FUNCTIONS.keys(),
                strip_values=True,
            )

            for rowdict in parser:
                for key, parse_func in STATION_ROW_PARCE_FUNCTIONS.iteritems():
                    rowdict[key] = parse_func(rowdict[key])

                if not rowdict['code']:
                    continue

                stations[rowdict['code']] = rowdict['title']

        raw_stations = dict()

        add_stations(raw_stations, self.get_stations_filepath())
        add_stations(raw_stations, self.get_terminals_filepath())

        return raw_stations

    def schedule_rows_iter(self):
        parser = XlsDictParser(
            self.get_schedule_filepath(),
            transform_to_text=True,
            fieldnames=SCHEDULE_ROW_PARCE_FUNCTIONS.keys(),
            strip_values=True,
        )

        for row in parser:
            for key, parse_func in SCHEDULE_ROW_PARCE_FUNCTIONS.iteritems():
                row[key] = parse_func(row[key])

            self.replace_arrival(row)

            yield row

    def replace_arrival(self, row):
        arrival_key = self.get_arrival_key_from_row(row)

        if arrival_key in self.arrival_replaces:
            log.info(u'Нитка %s. Станция прибытия %s. Заменяем время прибытия с %s на %s,',
                     row['route_title'], row['station_to'],
                     row['arrival_time'], self.arrival_replaces[arrival_key])

            row['arrival_time'] = self.arrival_replaces[arrival_key]

    def get_arrival_key_from_row(self, row):
        return (
            row['route_title'],
            row['departure_time'],
            row['station_from'],
            row['station_to']
        )


class RawThread(object):
    def __init__(self, rows):
        rows = self.filter_duplicates(rows)

        self.start_station_code = rows[0]['station_from']
        self.departure_time = rows[0]['departure_time']

        self.title = rows[0]['route_title']
        self.number = rows[0]['number']
        self.company_title = rows[0]['company_title']
        self.model_title = rows[0]['model_title']

        self.schedule = self.parse_schedule(rows[0]['template_text'])

        self.stoppoints, self.tariffs, self.raw_data = self.parse(rows)

    def filter_duplicates(self, rows):
        unique_rows = []

        row_keys = set()

        for row in rows:
            row_key = (row['route_title'], row['departure_time'],
                       row['station_from'], row['station_to'], row['template_text'])

            if row_key in row_keys:
                log.info(u'Исключили строчку дубликат')

                continue

            row_keys.add(row_key)

            unique_rows.append(row)

        return unique_rows

    def parse(self, rows):
        stoppoints = []
        tariffs = []
        raw_data = []

        for row in rows:
            raw_data.append(json.dumps(row, indent=4, encoding='utf8', ensure_ascii=False, cls=TimeEncoder))

            if row['station_from'] == self.start_station_code:
                stoppoints.append(row)

            tariff = row['tariff']

            if tariff is not None and tariff > 0:
                tariffs.append(row)

        return stoppoints, tariffs, u'\n'.join(raw_data)

    def parse_schedule(self, template_text):
        template_text = template_text.lower()

        if u'еженедельно' in template_text:
            schedule = []

            for weekday_number, weekday_text in DAYS_OF_WEEK.iteritems():
                if weekday_text in template_text:
                    schedule.append(weekday_number)

            return u''.join(schedule)

        if u'по четным дням' in template_text:
            return u'четные'

        if u'по нечетным дням' in template_text:
            return u'нечетные'

        return template_text


def is_another_thread(row, first_row):
    if not first_row:
        return True

    if row['route_title'] != first_row['route_title']:
        return True

    if (
            row['station_from'] == first_row['station_from'] and
            (
                row['departure_time'] != first_row['departure_time'] or
                row['template_text'] != first_row['template_text']
            )
    ):
        return True

    return False


def has_not_important_data(row):
    return row['departure_time'] is None or not row['template_text']


class TimeEncoder(json.JSONEncoder):
    def default(self, obj):
        if isinstance(obj, time):
            encoded_object = obj.strftime('%H:%M:%S')

        else:
            encoded_object = json.JSONEncoder.default(self, obj)

        return encoded_object


def safe_transform(value, func=None):
    """
    Преобразует велечину value к нужному типу, либо возвращает None
    в случае ValueError.
    Если передан единственный параметр функция, ведет себя как декоратор
    @param value:
    @param func:
    """

    if callable(value):
        function = value

        @wraps(function)
        def safe(_value):
            return safe_transform(_value, function)

        return safe

    try:
        return func(value)

    except ValueError:
        return None


@safe_transform
def parse_station_code(cell_value):
    value = int(cell_value)
    format = u"%09d"

    return format % value


time_formats = ["%H:%M:%S", "%H:%M"]


def parse_time(cell_value):
    if not cell_value:
        return None

    for format in time_formats:
        try:
            return time(*os_time.strptime(cell_value, format)[3:6])

        except ValueError:
            pass

    log.error(u"Не смогли разобрать время '%s'", cell_value)

    return None


def parse_number(number):
    number = unicode(int(number)) if isinstance(number, float) else number
    number = number.replace(u" ", u"").upper()

    return number


def do_nothing(arg):
    return arg


STATION_ROW_PARCE_FUNCTIONS = OrderedDict((
    ('code', parse_station_code),
    ('title', do_nothing)
))

SCHEDULE_ROW_PARCE_FUNCTIONS = OrderedDict((
    ('_empty', do_nothing),
    ('station_from', parse_station_code),
    ('station_to', parse_station_code),
    ('departure_time', parse_time),
    ('arrival_time', parse_time),
    ('number', parse_number),
    ('route_title', do_nothing),
    ('company_title', do_nothing),
    ('model_title', do_nothing),
    ('bus_type', do_nothing),
    ('template_text', do_nothing),
    ('tariff', safe_transform(lambda t: round(float(t), 2))),
    ('duration', safe_transform(lambda d: int(d))),  # лучше не использовать, т.к. неправильные значения
    ('distance', safe_transform(lambda d: float(d))),
))
