# -*- coding: utf-8 -*-

import os.path
import time as os_time
import urllib2
from HTMLParser import HTMLParser

from common.cysix.builder import ChannelBlock, GroupBlock, ThreadBlock, StoppointBlock, ScheduleBlock
from cysix.tsi_converter import CysixTSIConverterFactory, CysixTSIConverterFileProvider

from django.conf import settings
from django.utils.functional import cached_property
from lxml import etree

from travel.rasp.admin.lib.logs import get_current_file_logger
from travel.rasp.admin.lib.http_timeout_handler import get_http_timeout_handler_class
from common.utils.caching import cache_method_result
from travel.rasp.admin.scripts.schedule.utils import RaspImportError
from travel.rasp.admin.scripts.schedule.utils.file_providers import PackageFileProvider
from travel.rasp.admin.scripts.schedule.utils.to_python_parsers import get_date

from travel.rasp.admin.scripts.utils.import_file_storage import get_schedule_temporary_date_filepath


log = get_current_file_logger()


LOGIN = 'yandex'
PASSWORD = 'MsuIYhe4oUQPh45sG'

BASE_URL = 'schedule.t2t.in.ua'

SOURCE_POINTS_URL = 'https://schedule.t2t.in.ua/pap/sh/get_stoppoints'

POINT_SCHEDULE_URL = 'https://schedule.t2t.in.ua/pap/sh/get_schedule?src=%(source_point_id)s'

ATTEMPTS_TO_DOWNLOAD = 3


class UkrmintransCysixFactory(CysixTSIConverterFactory):
    def get_raw_package_file_provider(self):
        return UkrmintransPackageFileProvider(self.package)

    def get_raw_download_file_provider(self):
        log.info(u"Получаем данные по %s", SOURCE_POINTS_URL)

        return UkrmintransHttpFileProvider(self.package)

    def get_converter_file_provider(self, raw_file_provider):
        return UkrmintransConverterFileProvider(self.package, raw_file_provider)


class UkrmintransPackageFileProvider(PackageFileProvider):

    def get_schedule_files(self):
        if self.is_archive_package_file():
            source_points_file, raw_files = self.get_files_map_from_archive_with_ext('.xml')

            if raw_files:

                log.info(u'Обрабатываем файлы %s, %s', source_points_file, raw_files)

                return source_points_file, raw_files

        raise RaspImportError(u"Для импорта нужно более одного файла. "
                              u"Приложите все необходимые файлы в одном архиве.")

    def get_files_map_from_archive_with_ext(self, ext):
        if not ext.startswith('.'):
            ext = '.' + ext

        file_map = self.unpack_package_archive()

        source_points_file = None
        raw_files = dict()

        for filename, filepath in file_map.items():
            if filename == 'source_points.xml':
                source_points_file = filepath

            elif filename.endswith(ext):
                raw_files[filename] = filepath

        if source_points_file is None:
            raise RaspImportError(u"Для импорта нужн файл 'source_points.xml'.")

        return source_points_file, raw_files


class CantGetSupplierFile(RaspImportError):
    pass


class UkrmintransHttpFileProvider(PackageFileProvider):
    def get_schedule_files(self):
        return self.download_schedule_files()

    def download_schedule_files(self):
        source_points_file = self.get_supplier_file('source_points.xml', SOURCE_POINTS_URL)
        raw_files = dict()

        tree = etree.parse(source_points_file)
        for source_point in tree.findall('.//stoppoint'):
            source_id = source_point.get('src')

            filename = u"source_point_%s_schedule.xml" % source_id
            url = POINT_SCHEDULE_URL % {'source_point_id': source_id}

            try:
                shedule_file = self.get_supplier_file(filename, url)
                raw_files[filename] = shedule_file

            except CantGetSupplierFile, e:
                log.error(unicode(e))

        return source_points_file, raw_files

    def get_supplier_file(self, filename, url):
        filepath = get_schedule_temporary_date_filepath(filename, self.package)
        log.info(u"Качаем файл %s в %s", filename, filepath)

        if os.path.exists(filepath):
            log.info(u"Файл %s уже скачан в %s", filename, filepath)
            return filepath

        log.info(u"Качаем файл %s по %s", filename, url)

        for i in range(ATTEMPTS_TO_DOWNLOAD):
            if self.download_supplier_file(filepath, url):
                return filepath

        raise CantGetSupplierFile(u"Не удалось скачать файл %s. Было предпринято попыток - %s."
                                  % (filename, ATTEMPTS_TO_DOWNLOAD))

    def download_supplier_file(self, filepath, url):
        os_time.sleep(10)

        opener = self._get_opener()
        data = opener.open(url, timeout=settings.SCHEDULE_IMPORT_TIMEOUT).read()

        if data[:100].strip().startswith('<?xml'):
            with open(filepath, 'w') as f:
                f.write(data)

            return True

        return False

    @cache_method_result
    def _get_opener(self):
        pm = urllib2.HTTPPasswordMgrWithDefaultRealm()
        pm.add_password(None, BASE_URL, LOGIN, PASSWORD)

        auth_handler = urllib2.HTTPBasicAuthHandler(pm)

        http_handler_class = get_http_timeout_handler_class(settings.SCHEDULE_IMPORT_TIMEOUT)

        return urllib2.build_opener(http_handler_class, auth_handler)


class UkrmintransConverterFileProvider(CysixTSIConverterFileProvider):
    def __init__(self, package, raw_file_provider):
        self.package = package
        self.raw_file_provider = raw_file_provider
        super(UkrmintransConverterFileProvider, self).__init__(self.package, self.raw_file_provider)

    def convert_data(self, filepath):
        source_points_file, raw_files = self.raw_file_provider.get_schedule_files()

        c = Converter(source_points_file, raw_files)
        filepath = c.convert(filepath)
        return filepath


class ConvertError(RaspImportError):
    pass


class UkrmintransGroup(object):
    def __init__(self, filename, raw_files):
        self.filename = filename
        self._filepath = raw_files[filename]

    @cached_property
    def group_raw_el(self):
        return etree.parse(self._filepath).getroot()


class Converter(object):
    def __init__(self, source_points_file, raw_files):
        self.source_points_file = source_points_file
        self.raw_files = raw_files

    def convert(self, filepath):

        channel_block = ChannelBlock('bus', station_code_system='vendor', carrier_code_system='local',
                                     vehicle_code_system='local', timezone='start_station')

        source_points_el = etree.parse(self.source_points_file)

        for source_stoppoint_el in source_points_el.findall('.//stoppoint'):
            group_block = self.build_group(source_stoppoint_el, channel_block)

            channel_block.add_group_block(group_block)

        channel_el = channel_block.get_element()

        with open(filepath, 'w') as f:
            f.write(etree.tostring(channel_el, xml_declaration=True, encoding='utf-8', pretty_print=True))

    def build_group(self, source_stoppoint_el, channel_block):

        group_title = source_stoppoint_el.get('rus_name', u"").strip()
        group_code = source_stoppoint_el.get('src', u"").strip()

        group_block = GroupBlock(channel_block, group_title, group_code)

        filename = u"source_point_%s_schedule.xml" % group_code
        if filename in self.raw_files:
            ukr_group = UkrmintransGroup(filename, self.raw_files)

            stations = self.build_stations(group_block, ukr_group.group_raw_el, source_stoppoint_el)

            self.build_threads(group_block, ukr_group, stations)
        else:
            log.error(u'Не нашли файла для source_point %s %s', group_title, group_code)

        return group_block

    def build_stations(self, group_block, group_raw_el, source_stoppoint_el):
        stations = dict()

        for s_el in group_raw_el.findall('./stopslist/stoppoint'):
            code = s_el.get('vendor_id', u'').strip()
            title = s_el.get('name', u'').strip()
            station_block = group_block.add_station(title, code)

            stations[code] = station_block

            old_code = get_old_station_code(source_stoppoint_el, s_el)
            station_block.add_legacy_station(title, old_code)

        return stations

    def build_threads(self, group_block, ukr_group, stations):
        for raw_thread_el in ukr_group.group_raw_el.findall('./threads/thread'):
            try:
                thread_block = self.build_thread(group_block, raw_thread_el, stations,
                                                 ukr_group.filename)
                group_block.add_thread_block(thread_block)
            except ConvertError, e:
                log.error(unicode(e))

    def build_thread(self, group_block, raw_thread_el, stations, filename):
        number = raw_thread_el.get('number', u"").strip()
        title = raw_thread_el.get('title', u"").strip()
        thread_name = u"{}: {}".format(number, title)
        thread_block = ThreadBlock(group_block, title=title, number=number)

        self.add_stoppoints_to_thread(thread_block, raw_thread_el, thread_name, stations)
        self.add_schedule_to_thread(thread_block, raw_thread_el, thread_name)
        self.add_raw_data_to_thread(thread_block, raw_thread_el, filename)

        return thread_block

    def add_stoppoints_to_thread(self, thread_block, raw_thread_el, thread_name, stations):
        station_els = raw_thread_el.findall('./stations/station')

        if not station_els:
            raise ConvertError(u"Пропускаем нитку '{}'. "
                               u"Не удалось найти станции следования.".format(thread_name))

        for raw_stoppoint in station_els:
            code = raw_stoppoint.get('vendor_id', u"").strip()
            arrival = raw_stoppoint.get('arrival_time', u"").strip()
            departure = raw_stoppoint.get('departure_time', u"").strip()

            station = stations.get(code, None)
            if station is None:
                raise ConvertError(u"Пропускаем нитку '{}'. "
                                   u"Не удалось найти станцию с кодом '{}'.".format(thread_name, code))

            stoppoint_block = StoppointBlock(thread_block, station, arrival_time=arrival, departure_time=departure)
            thread_block.add_stoppoint_block(stoppoint_block)

    def add_schedule_to_thread(self, thread_block, raw_thread_el, thread_name):
        days = []

        for raw_schedule_el in raw_thread_el.findall('./schedules/schedule'):
            date = raw_schedule_el.get('date', u"").strip()

            if get_date(date, silent=True, fmt="%Y-%m-%d") is not None:
                days.append(date)
            else:
                log.error(u"Нитка '%s'. Не удалось разобрать дату '%s'.", thread_name, date)

        if not days:
            log.error(u"Нитка '%s'. Не удалось найти расписание.", thread_name)

        days = u";".join(days)

        schedule = ScheduleBlock(thread_block, days)
        thread_block.add_schedule_block(schedule)

    def add_raw_data_to_thread(self, thread_block, raw_thread_el, filename):
        raw_data = etree.tounicode(raw_thread_el, pretty_print=True)

        raw_data = HTMLParser().unescape(raw_data)

        thread_block.set_raw_data(u'{}\n\n{}'.format(filename, raw_data))


def get_old_station_code(source_stoppoint_el, station_raw_el):
    code_parts = []

    local_code = station_raw_el.get('local_code', u"").strip()
    code_parts.append(local_code)

    etaton_code = station_raw_el.get('etalon_code', u"").strip()

    if is_good_etalon_code(etaton_code):
        code_parts.append(etaton_code)

    else:
        src = source_stoppoint_el.get('src', u"").strip()
        code_parts.append(src)

        city_idx = source_stoppoint_el.get('city_idx', u"").strip()
        code_parts.append(city_idx)

    return u"_".join(code_parts)


def is_good_etalon_code(etalon_code):
    return etalon_code not in (u"0", u"-1") and len(etalon_code) > 4
