# coding: utf-8

from __future__ import unicode_literals

import travel.rasp.admin.scripts.load_project  # noqa

import logging
import argparse

from django.db import transaction

from common.models.schedule import Supplier, RThread
from common.models.transport import TransportType
from common.utils.metrics import task_progress_report
from travel.rasp.admin.lib.maintenance.flags import flags
from travel.rasp.admin.lib.maintenance.scripts import job
from travel.rasp.admin.lib.logs import print_log_to_stdout, create_current_file_run_log, get_script_log_context, ylog_context
from travel.rasp.admin.scripts.schedule.import_af import import_af
from travel.rasp.admin.www.utils.mysql import fast_delete_threads_with_routes


log = logging.getLogger(__name__)


@transaction.atomic
def reimport_af(t_type_code=None, add_import=False, check_maintenance_flag=True):
    if check_maintenance_flag:
        if flags['maintenance']:
            log.error('Идет работа с базой данных нельзя запускать скрипт')
            return
        if add_import:
            flags['maintenance'] = job.AF_ADDIMPORT.flag_value
        else:
            flags['maintenance'] = job.AF_REIMPORT.flag_value

    af = Supplier.objects.get(code='af')

    t_type = TransportType.objects.get(code=t_type_code)

    # Если переимпорт, то чистим базу
    if not add_import:
        log.info('Удаляем все маршруты от А.Ф. с типом транспорта %s', t_type.title_ru)
        fast_delete_threads_with_routes(RThread.objects.filter(t_type=t_type, supplier=af))

    log.info('Импотрируем машруты от А.Ф. с типом транспорта %s', t_type.title_ru)
    import_af(t_type_code=t_type.code, add_import=add_import)

    log.info('Импорт закончен')
    if check_maintenance_flag:
        flags['maintenance'] = 0


if __name__ == '__main__':
    with ylog_context(**get_script_log_context()), task_progress_report('re_import_af_schedule'):
        create_current_file_run_log()

        parser = argparse.ArgumentParser()
        parser.add_argument('t_type_code', choices=[t.code for t in TransportType.objects.all()], help='Тип транспорта',
                            metavar='T_TYPE_CODE')
        parser.add_argument('-v', '--verbose', action="store_true")
        parser.add_argument('-a', '--add-import', action="store_true", help='Делать доимпорт')
        parser.add_argument('--ignore-maintenance-flag', action="store_false", default=True,
                            help='Игнорировать флаг работы с базой', dest='check_maintenance_flag')
        args = parser.parse_args()

        if args.verbose:
            print_log_to_stdout()

        reimport_af(t_type_code=args.t_type_code, add_import=args.add_import,
                    check_maintenance_flag=args.check_maintenance_flag)
