# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import timedelta
from collections import defaultdict

from common.models.schedule import DeLuxeTrain
from travel.rasp.library.python.common23.date import environment
from common.apps.im_logs.models import get_im_train_car_types


log = logging.getLogger(__name__)


IM_CHECKING_DAYS_NUMBER = 7


class DeLuxeTrainType(object):
    def __init__(self, title, im_title, wagon_types_str, numbers_str, min_im_records_count):
        self.title = title if title else ''
        self.im_title = im_title if im_title else ''
        self.wagon_types = set((wagon_types_str if wagon_types_str else '').split('/'))
        self.numbers = set(numbers_str.split('/'))
        self.min_im_records_count = min_im_records_count


def _make_error_message(is_missed, train_number, train_title, im_train_title_wagon_pairs):
    return 'Поезд {} ошибочно {} в списке "{}". Неправильные вагоны: {}'.format(
        train_number,
        'отсутствует' if is_missed else 'присутствует',
        train_title,
        ', '.join('("{}", "{}")'.format(im_train_title, wagon) for im_train_title, wagon in im_train_title_wagon_pairs)
    )


def get_deluxe_trains_errors():
    """
    Проверка правильности заполнения таблицы DeLuxeTrain, с использованием группированных логов ИМ
    :return: сообщение, содержащее все ошибки
    """
    log.info('Проверка таблицы DeLuxeTrain')
    types_by_numbers = defaultdict(list)  # Словарь типов поездов, ключи - номера поездов
    types_by_wagons = defaultdict(list)  # Словарь типов поездов, ключи - пары: имя поезда в ИМ, тип вагона

    # Загружаем в словари все фирменные поезда, для которых нужна проверка
    for deluxe_train in DeLuxeTrain.objects.all():
        train_type = DeLuxeTrainType(
            deluxe_train.title, deluxe_train.im_title,
            deluxe_train.wagon_types, deluxe_train.numbers,
            deluxe_train.min_im_records_count
        )
        if deluxe_train.need_check:
            for wagon_type in train_type.wagon_types:
                types_by_wagons[(train_type.im_title, wagon_type)].append(train_type)
        if deluxe_train.need_reverse_check:
            for number in train_type.numbers:
                types_by_numbers[number].append(train_type)

    # Недостающие и лишние номера поездов,
    # ключи - пары: номер поезда, имя поезда в базе
    # значения - множество несоответствующих пар: имя поезда в ИМ, тип вагона
    missed_numbers = defaultdict(set)  # Недостающие номера поездов, множество пар: номер поезда, имя поезда в базе
    extra_numbers = defaultdict(set)  # Лишние номера поездов, множество пар: номер поезда, имя поезда в базе

    today = environment.today()
    day_from = today - timedelta(days=IM_CHECKING_DAYS_NUMBER)
    day_to = today - timedelta(days=1)

    for im_type in get_im_train_car_types(day_from, day_to):
        if not im_type.train_number or not im_type.wagon:
            continue

        # Пара (имя поезда, вагон) или только вагон, если имя не указано, соответствует неправильному номеру поезда
        train_types = []
        if (im_type.train_name, im_type.wagon) in types_by_wagons:
            train_types += types_by_wagons[(im_type.train_name, im_type.wagon)]
        if ('', im_type.wagon) in types_by_wagons:
            train_types += types_by_wagons[('', im_type.wagon)]

        if train_types:
            has_number = False
            is_enough_records = False

            for train_type in train_types:
                has_number = has_number or im_type.train_number in train_type.numbers
                is_enough_records = is_enough_records or im_type.records_count >= train_type.min_im_records_count

            if not has_number and is_enough_records:
                for train_type in train_types:
                    missed_numbers[(im_type.train_number, train_type.title)].add((im_type.train_name, im_type.wagon))

        # Номер поезда не соответствует имени поезда и вагону
        if im_type.train_number in types_by_numbers:
            train_types = types_by_numbers[im_type.train_number]
            has_wagon = False
            is_enough_records = False

            for train_type in train_types:
                has_wagon = has_wagon or (
                    im_type.wagon in train_type.wagon_types
                    or (train_type.im_title and im_type.train_name == train_type.im_title)
                )
                is_enough_records = is_enough_records or im_type.records_count >= train_type.min_im_records_count

            if not has_wagon and is_enough_records:
                for train_type in train_types:
                    extra_numbers[(im_type.train_number, train_type.title)].add((im_type.train_name, im_type.wagon))

    messages = []
    for (train_number, train_title), im_train_title_wagon_pairs in missed_numbers.items():
        messages.append(_make_error_message(True, train_number, train_title, im_train_title_wagon_pairs))
    for (train_number, train_title), im_train_title_wagon_pairs in extra_numbers.items():
        messages.append(_make_error_message(False, train_number, train_title, im_train_title_wagon_pairs))

    result = '\n'.join(messages)
    log.info('Найденные ошибки в DeLuxeTrain: {}'.format(result))
    return result
